#ifndef FOLDERMODEL_H
#define FOLDERMODEL_H

#include <QAbstractTableModel>
#include <QDir>
#include <QFileIconProvider>
#include <QFileSystemWatcher>
#include <QBrush>
#include <QFont>
#include <QMutex>
#include <QPalette>
class Preferences;
class ThumbnailWorker;

class FolderModel : public QAbstractTableModel
{
    Q_OBJECT
public:
    enum Column {
        Name = 0,
        Extension,
        Size,
        LastModified,
        ColumnCount,
    };
    typedef QVector<QString> History;

    explicit FolderModel(QObject *parent = 0);
    ~FolderModel();

    void            clearPixmapCache();
    QString         error() const;
    QIcon           fileIcon(const QModelIndex &index) const;
    QFileInfo       fileInfo(const QModelIndex &index) const;
    QString         fileName(const QModelIndex &index) const;
    QString         filePath(const QModelIndex &index) const;
    QDir::Filters   filter() const;
    const History&  history() const;
    bool            isActive() const;
    bool            isDir(const QModelIndex &index) const;
    bool            isHistoryBegin() const;
    bool            isHistoryEnd() const;
    bool            isMarked(const QModelIndex &index) const;
    QFileInfoList   markedItems() const;
    QModelIndex     mkdir(const QString &name);
    QStringList     nameFilters() const;
    QPixmap         pixmap(const QModelIndex &index, const QSize &size) const;
    QString         rootPath() const;
    QModelIndex     search(const QString &value, const QModelIndex &start = QModelIndex(), int step = 1);
    void            setActive();
    void            setFilter(QDir::Filters filters);
    void            setHistoryAt(const QString &path);
    void            setNameFilters(const QStringList &nameFiltes);
    void            setRootPath(const QString &path, bool addHistory = true);
    void            setSorting(QDir::SortFlags sort);
    QDir::SortFlags sorting() const;
    QModelIndex     touch(const QString &name);
    void            updateAppearance(const Preferences &prefs);

    static FolderModel* activeModel();

signals:

public slots:
    void    fsWatcher_directoryChanged(const QString &path);
    void    onCdHome();
    void    onCdRoot();
    void    onCdUp();
    void    onHistoryBack();
    void    onHistoryForward();
    void    onMarkAll();
    void    onMarkAllFiles();
    void    onMarkAllOff();
    void    onMarkInvert();
    void    refresh();
    void    thumbnail_Ready(const QString &path, const QPixmap &pixmap);

private:
    static FolderModel* m_activeModel;

    typedef QMap<QString, int> CheckContainer;
    typedef QMap<QString, QPixmap> PixmapContainer;
    typedef QMap<QString, QDateTime> DateTimeContainer;

    QString             m_error;
    QDir                m_dir;
    QFileInfoList       m_fileInfoList;
    CheckContainer      m_checkStates;
    QFileIconProvider   m_IconProvider;
    QFileSystemWatcher  m_fsWatcher;
    History             m_history;
    int                 m_historyPos;
    ThumbnailWorker*    m_worker;
    PixmapContainer     m_pixmapCache;
    DateTimeContainer   m_lastModifiedCache;
    QMutex              m_pixmapCacheMutex;
    QPalette            m_Palette;
    QFont               m_font;

    bool    isDotFile(const QModelIndex &index) const;
    void    setError(const QString &error = QString());

    const QBrush&   base() const;
    const QBrush&   text() const;
    const QBrush&   marked() const;
    const QBrush&   markedText() const;
    const QBrush&   hidden() const;
    const QBrush&   readOnly() const;
    const QBrush&   system() const;

    // QAbstractItemModel interface
public:
    int rowCount(const QModelIndex &parent = QModelIndex()) const;
    int columnCount(const QModelIndex &parent = QModelIndex()) const;
    QVariant data(const QModelIndex &index, int role = Qt::DisplayRole) const;
    Qt::ItemFlags flags(const QModelIndex &index) const;
    bool setData(const QModelIndex &index, const QVariant &value, int role = Qt::EditRole);
    Qt::DropActions supportedDropActions() const;
    QStringList mimeTypes() const;
};

inline QString FolderModel::error() const
{
    return m_error;
}

inline QFileInfo FolderModel::fileInfo(const QModelIndex &index) const
{
    Q_ASSERT(index.isValid());
    return m_fileInfoList[index.row()];
}

inline QString FolderModel::fileName(const QModelIndex &index) const
{
    Q_ASSERT(index.isValid());
    return m_fileInfoList[index.row()].fileName();
}

inline QString FolderModel::filePath(const QModelIndex &index) const
{
    Q_ASSERT(index.isValid());
    return m_fileInfoList[index.row()].absoluteFilePath();
}

inline QDir::Filters FolderModel::filter() const
{
    return m_dir.filter();
}

inline const FolderModel::History &FolderModel::history() const
{
    return m_history;
}

inline bool FolderModel::isActive() const
{
    return m_activeModel == this;
}

inline bool FolderModel::isDir(const QModelIndex &index) const
{
    return m_fileInfoList[index.row()].isDir();
}

inline bool FolderModel::isHistoryBegin() const
{
    return m_historyPos == 0;
}

inline bool FolderModel::isHistoryEnd() const
{
    return m_historyPos == m_history.size() - 1;
}

inline bool FolderModel::isDotFile(const QModelIndex &index) const
{
    return fileName(index).lastIndexOf(".") == 0;
}

inline QStringList FolderModel::nameFilters() const
{
    return m_dir.nameFilters();
}

inline void FolderModel::refresh()
{
    setRootPath(rootPath());
}

inline QString FolderModel::rootPath() const
{
    return m_dir.absolutePath();
}

inline void FolderModel::setActive()
{
    m_activeModel = this;
}

inline void FolderModel::setError(const QString &error)
{
    m_error = error;
}

inline void FolderModel::setFilter(QDir::Filters filters)
{
    m_dir.setFilter(filters);
}

inline void FolderModel::setSorting(QDir::SortFlags sort)
{
    m_dir.setSorting(sort);
}

inline void FolderModel::setNameFilters(const QStringList &nameFiltes)
{
    m_dir.setNameFilters(nameFiltes);
}

inline QDir::SortFlags FolderModel::sorting() const
{
    return m_dir.sorting();
}

inline void FolderModel::fsWatcher_directoryChanged(const QString &path)
{
    Q_UNUSED(path);
    refresh();
}

inline const QBrush& FolderModel::base() const
{
    return m_Palette.base();
}
inline const QBrush& FolderModel::text() const
{
    return m_Palette.text();
}
inline const QBrush& FolderModel::marked() const
{
    return m_Palette.highlight();
}
inline const QBrush& FolderModel::markedText() const
{
    return m_Palette.highlightedText();
}
inline const QBrush& FolderModel::hidden() const
{
    return m_Palette.dark();
}
inline const QBrush& FolderModel::readOnly() const
{
    return m_Palette.light();
}
inline const QBrush& FolderModel::system() const
{
    return m_Palette.mid();
}
#endif // FOLDERMODEL_H
