#include "preferences.h"
#include "bookmarkdialog.h"
#include "ui_bookmarkdialog.h"

#include <QDebug>
#include <QFileDialog>
#include <QFileInfo>
#include <QMessageBox>

///////////////////////////////////////////////////////////////////////////////
/// \brief BookmarkDialog::BookmarkDialog
/// \param parent   親ウィジェット
///
/// コンストラクタ
///
BookmarkDialog::BookmarkDialog(QWidget *parent) :
    QDialog(parent),
    ui(new Ui::BookmarkDialog),
    m_isReadOnly(false)
{
    ui->setupUi(this);
    resize(parent->width() * 0.8, height());

    ui->tableWidget->setColumnCount(2);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief BookmarkDialog::~BookmarkDialog
///
/// デストラクタ
///
BookmarkDialog::~BookmarkDialog()
{
    delete ui;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief BookmarkDialog::setEditMode
/// \param edit 編集モードの場合はtrue
///
void BookmarkDialog::setEditMode(bool edit)
{
    ui->buttonAdd->setVisible(edit);
    ui->buttonDelete->setVisible(edit);
    ui->buttonDown->setVisible(edit);
    ui->buttonUp->setVisible(edit);

    m_isReadOnly = !edit;
    if (m_isReadOnly) {
        setWindowTitle(tr("ブックマークを開く"));
        connect(ui->tableWidget, SIGNAL(doubleClicked(QModelIndex)), this, SLOT(accept()));
    }

    Preferences prefs(this);
    for (int i = 1; ; i++) {
        QString name = prefs.getBookmarkEntry(i);
        if (name.isEmpty()) {
            break;
        }

        QString path = prefs.getBookmarkPath(i);
        insertData(i - 1, name, path);
    }
    ui->tableWidget->resizeColumnsToContents();
    ui->tableWidget->resizeRowsToContents();
    ui->tableWidget->horizontalHeader()->setStretchLastSection(true);
    ui->tableWidget->setCurrentCell(0, 0);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief BookmarkDialog::selectedIndex
/// \return 現在の行番号
///
int BookmarkDialog::selectedIndex() const
{
    return ui->tableWidget->currentRow();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief BookmarkDialog::insertData
/// \param row  行番号
/// \param name 名前
/// \param path パス
///
/// テーブルウィジェットにデータを追加します。
///
void BookmarkDialog::insertData(int row, const QString &name, const QString &path)
{
    ui->tableWidget->insertRow(row);

    QTableWidgetItem *iName = new QTableWidgetItem(name);
    if (m_isReadOnly) {
        iName->setFlags(iName->flags() ^ Qt::ItemIsEditable);
    }
    ui->tableWidget->setItem(row, 0, iName);

    QTableWidgetItem *iPath = new QTableWidgetItem(path);
    iPath->setFlags(iPath->flags() ^ Qt::ItemIsEditable);
    ui->tableWidget->setItem(row, 1, iPath);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief BookmarkDialog::moveSelectedRows
/// \param up   上げる場合はtrue, 下げる場合はfalse
///
/// 選択行を上または下に移動します。
///
void BookmarkDialog::moveSelectedRows(bool up)
{
    qDebug() << "BookmarkDialog::moveSelectedRows();" << up;

    QList<QTableWidgetItem*> selected = ui->tableWidget->selectedItems();
    if (selected.isEmpty()) {
        return;
    }

    QVector<QString> before;
    QVector<QString> after;

    for (int i = 0; i < ui->tableWidget->rowCount(); i++) {
        QString name = ui->tableWidget->item(i, 0)->data(Qt::DisplayRole).toString();
        QString path = ui->tableWidget->item(i, 1)->data(Qt::DisplayRole).toString();
        before << name + "¥t" + path;
    }

    after.resize(before.size());

    QVector<int> newRows;
    foreach (QTableWidgetItem *item, selected) {
        if (item->column() != 0) {
            int newRow = (up) ? item->row() - 1 : item->row() + 1;
            if (newRow < 0) {
                return;
            }
            else if (newRow >= ui->tableWidget->rowCount()) {
                return;
            }

            after[newRow] = before[item->row()];
            before[item->row()] = QString::null;
            newRows << newRow;
        }
    }

    int n = 0;
    for (int i = 0; i < ui->tableWidget->rowCount(); i++) {
        if (!before[i].isEmpty()) {
            while (!after[n].isEmpty()) {
                n++;
            }
            after[n] = before[i];
        }
    }

    while (ui->tableWidget->rowCount() > 0) {
        ui->tableWidget->removeRow(0);
    }

    n = 0;
    foreach (const QString &val, after) {
        QStringList list = val.split("¥t");
        insertData(n, list[0], list[1]);
        n++;
    }

    ui->tableWidget->resizeColumnsToContents();
    ui->tableWidget->resizeRowsToContents();

    foreach (int i, newRows) {
        for (n = 0; n < 2; n++) {
            QModelIndex index = ui->tableWidget->model()->index(i, n);
            ui->tableWidget->selectionModel()->select(index, QItemSelectionModel::Select);
        }
    }

}

///////////////////////////////////////////////////////////////////////////////
/// \brief BookmarkDialog::accept
///
/// OKボタンクリック時の処理を行います。
///
void BookmarkDialog::accept()
{
    Preferences prefs(this);
    prefs.clearBookmark();

    QModelIndex index;
    int i;
    for (i = 0; i < ui->tableWidget->rowCount(); i++) {
        index = ui->tableWidget->model()->index(i, 0);
        QString name = ui->tableWidget->model()->data(index).toString();
        if (name.isEmpty()) {
            ui->tableWidget->setCurrentCell(i, 0);
            QMessageBox::critical(this, tr("エラー"), tr("名前が未入力です。"));
            return;
        }

        index = ui->tableWidget->model()->index(i, 1);
        QString path = ui->tableWidget->model()->data(index).toString();

        prefs.addBookmark(name, path);
    }

    QDialog::accept();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief BookmarkDialog::on_buttonDelete_clicked
///
/// 削除ボタンクリック時の処理を行います。
///
void BookmarkDialog::on_buttonDelete_clicked()
{
    foreach (const QTableWidgetItem *item, ui->tableWidget->selectedItems()) {
        if (item->column() == 0) {
            ui->tableWidget->removeRow(item->row());
        }
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief BookmarkDialog::on_buttonUp_clicked
///
/// ↑ボタンクリック時の処理を行います。
///
void BookmarkDialog::on_buttonUp_clicked()
{
    moveSelectedRows(true);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief BookmarkDialog::on_buttonDown_clicked
///
/// ↓ボタンクリック時の処理を行います。
///
void BookmarkDialog::on_buttonDown_clicked()
{
    moveSelectedRows(false);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief BookmarkDialog::on_buttonAdd_clicked
///
/// 追加ボタンクリック時の処理を行います。
///
void BookmarkDialog::on_buttonAdd_clicked()
{
    static QString initPath = QDir::homePath();

    QString path = QFileDialog::getExistingDirectory(
                this, tr("フォルダを選択"), initPath);
    if (!path.isEmpty()) {
        QString name(QFileInfo(path).fileName());
        if (name.isEmpty()) {
            name = "root";
        }
        insertData(ui->tableWidget->rowCount(), name, path);
        initPath = path;

        ui->tableWidget->resizeColumnsToContents();
        ui->tableWidget->resizeRowsToContents();
    }
}
