#include "foldermodel.h"
#include "thumbnaildelegate.h"

#include <QApplication>
#include <QDebug>
#include <QPainter>
#include <QStylePainter>

///////////////////////////////////////////////////////////////////////////////
/// \brief ThumbnailDelegate::ThumbnailDelegate
/// \param parent   親オブジェクト
///
/// コンストラクタ
///
ThumbnailDelegate::ThumbnailDelegate(QObject *parent) :
    QStyledItemDelegate(parent)
{
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ThumbnailDelegate::paint
/// \param painter  ペインターオブジェクト
/// \param option   オプション
/// \param index    インデックス
///
/// 描画を行います
///
void ThumbnailDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const
{
    const FolderModel *model = static_cast<const FolderModel*>(index.model());
    bool checked = model->data(index, Qt::CheckStateRole).toInt() == Qt::Checked;

    // マーク背景
    if (checked) {
        painter->fillRect(option.rect, model->data(index, Qt::BackgroundRole).value<QBrush>());
    }
    // ハイライト背景
    if (option.state & QStyle::State_Selected) {
        painter->fillRect(option.rect, option.palette.highlight());
    }

    // ファイル名
    QFont font = model->data(index, Qt::FontRole).value<QFont>();
    QFontMetrics fm(font);
    QSize padding(20, fm.height());
    QRect rc(option.rect);
    rc.setLeft(rc.left() + padding.width());
    QString name = fm.elidedText(model->fileName(index), Qt::ElideMiddle, rc.width());
    painter->setFont(font);
    painter->setPen(model->data(index, Qt::ForegroundRole).value<QBrush>().color());
    painter->drawText(rc, name, Qt::AlignCenter | Qt::AlignBottom);

    // チェックボックス
    if (model->flags(index) & Qt::ItemIsUserCheckable) {
        QStyleOptionButton optBtn;
        optBtn.rect = option.rect;
        optBtn.state = option.state;
        if (checked)
            optBtn.state |= QStyle::State_On;
        qApp->style()->drawControl(QStyle::CE_CheckBox, &optBtn, painter);
    }

    // アイコンまたは画像
    QSize size(option.rect.size() - padding);
    QPixmap pixmap = model->pixmap(index, size);
    painter->drawPixmap(padding.width() + option.rect.left() + (size.width() - pixmap.width()) / 2,
                        option.rect.top() + (size.height() - pixmap.height()) / 2,
                        pixmap);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ThumbnailDelegate::sizeHint
/// \param option   オプション
/// \param index    インデックス
/// \return サイズを返します。
///
QSize ThumbnailDelegate::sizeHint(const QStyleOptionViewItem &option, const QModelIndex &index) const
{
    Q_UNUSED(option);

    const FolderModel *model = static_cast<const FolderModel*>(index.model());
    int pointSize = model->data(index, Qt::FontRole).value<QFont>().pointSize();

    return QSize(pointSize * 10, pointSize * 10 * 3 / 4) * m_scaleFactor;
}
