#include "global.h"
#include "preferences.h"
#include "thumbnailworker.h"
#include "win32.h"
#include "foldermodel.h"

#include <QApplication>
#include <QDateTime>
#include <QDebug>
#include <QPalette>
#include <QThread>

FolderModel* FolderModel::m_activeModel = NULL;

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::FolderModel
/// \param parent   親オブジェクト
///
/// コンストラクタ
///
FolderModel::FolderModel(QObject *parent) :
    QAbstractTableModel(parent),
    m_dir(),
    m_fileInfoList(),
    m_checkStates(),
    m_IconProvider(),
    m_fsWatcher(this),
    m_history(),
    m_historyPos(-1),
    m_worker(),
    m_pixmapCache(),
    m_lastModifiedCache(),
    m_pixmapCacheMutex(),
    m_Palette(),
    m_font()
{
    // サムネイル生成用スレッドを初期化する
    m_worker = new ThumbnailWorker();
    connect(m_worker, SIGNAL(resultReady(QString,QPixmap)), this, SLOT(thumbnail_Ready(QString,QPixmap)));
    m_worker->start();

    connect(&m_fsWatcher, SIGNAL(directoryChanged(QString)), this, SLOT(fsWatcher_directoryChanged(QString)));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::~FolderModel
///
/// デストラクタ
///
FolderModel::~FolderModel()
{
    m_worker->abort();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::clearPixmapCache
///
/// サムネイルキャッシュをクリアしたように見せかけます。
///
void FolderModel::clearPixmapCache()
{
    beginResetModel();
    endResetModel();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::fileIcon
/// \param index    アイテムのインデックス
/// \return アイコンを返します。
///
QIcon FolderModel::fileIcon(const QModelIndex &index) const
{
    if (fileName(index) == "..") {
        return QIcon("://images/Up.png");
    }
    return m_IconProvider.icon(fileInfo(index));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::isMarked
/// \param index    アイテムのインデックス
/// \return マークされていればtrueを返します。
///
bool FolderModel::isMarked(const QModelIndex &index) const
{
    CheckContainer::const_iterator it = m_checkStates.find(fileName(index));
    if (it != m_checkStates.end()) {
        return *it == Qt::Checked;
    }
    return false;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::markedItems
/// \return マークされているアイテムのリストを返します。
///
QFileInfoList FolderModel::markedItems() const
{
    QFileInfoList list;
    for (int n = 0; n < rowCount(); ++n) {
        QModelIndex index = this->index(n, 0);
        if (isMarked(index)) {
            list << fileInfo(index);
        }
    }

    return list;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::mkdir
/// \param name 作成するフォルダ名
/// \return 作成したフォルダのインデックス
///
/// フォルダを作成します。
///
QModelIndex FolderModel::mkdir(const QString &name)
{
    qDebug() << "FolderModel::mkdir()" << name;

    if (!m_dir.mkdir(name)) {
        return QModelIndex();
    }

    refresh();

    for (int n = 0; n < rowCount(); ++n) {
        if (m_fileInfoList[n].fileName() == name) {
            return index(n, 0);
        }
    }

    return QModelIndex();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::pixmap
/// \param index    アイテムのインデックス
/// \param size     要求サイズ
/// \return 画像またはアイコンを返します。
///
QPixmap FolderModel::pixmap(const QModelIndex &index, const QSize &size) const
{
    QPixmap pixmap;
    const_cast<FolderModel*>(this)->m_pixmapCacheMutex.lock();
    if (m_pixmapCache.find(filePath(index)) != m_pixmapCache.end()) {
        pixmap = m_pixmapCache[filePath(index)];
    }
    const_cast<FolderModel*>(this)->m_pixmapCacheMutex.unlock();

    if (!pixmap.isNull()) {
        double scaleX = 1.0 * size.width() / pixmap.width();
        double scaleY = 1.0 * size.height() / pixmap.height();
        double scaleFactor = qMin(scaleX, scaleY);
        if (scaleFactor > 1) {
            return pixmap;
        }
        return pixmap.scaled(pixmap.size() * scaleFactor,
                             Qt::IgnoreAspectRatio,
                             Qt::SmoothTransformation);
    }

    // とりあえずアイコンを返す。
    return fileIcon(index).pixmap(32, 32);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::search
/// \param value    検索する文字列
/// \param start    開始位置
/// \param step     ステップ数
/// \return 見つかったアイテムのインデックスを返します。
///
QModelIndex FolderModel::search(const QString &value, const QModelIndex &start, int step)
{
    qDebug() << "FolderModel::search()" << value << start << step;

    const QString searchText = value.toLower();
    for (int n = start.row() + step; 0 <= n && n < rowCount(); n += step) {
        if (m_fileInfoList[n].fileName().toLower().indexOf(searchText) != -1) {
            return index(n, 0);
        }
    }

    return QModelIndex();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::setHistoryAt
/// \param path 設定するパス
///
/// 指定したパスに履歴を移動します。
///
void FolderModel::setHistoryAt(const QString &path)
{
    qDebug() << "FolderModel::setHistoryAt()" << path;

    for (int n = 0; n < m_history.size(); ++n) {
        if (m_history[n] == path) {
            m_historyPos = n;
            setRootPath(path, false);
            return;
        }
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::setRootPath
/// \param path         設定するパス
/// \param addHistory   履歴に追加する場合はtrue
///
/// ルートパスを設定します。
///
void FolderModel::setRootPath(const QString &path, bool addHistory)
{
    qDebug() << "FolderModel::setRootPath()" << path;

    beginResetModel();
    setError();

    try {
        if (!QFileInfo::exists(path)) {
            throw tr("フォルダが存在しないか、利用できません：%1").arg(path);
        }

        QDir dir(path);
        dir.setFilter(m_dir.filter());
        dir.setNameFilters(m_dir.nameFilters());
        dir.setSorting(m_dir.sorting());

        QFileInfoList list = dir.entryInfoList();
        if (list.isEmpty()) {
            throw tr("ファイルリストを取得できません：%1").arg(path);
        }

        m_fileInfoList.clear();
        if (m_dir.absolutePath() != dir.absolutePath()) {
            m_fsWatcher.removePath(m_dir.absolutePath());
            m_fsWatcher.addPath(dir.absolutePath());
            m_checkStates.clear();
            m_dir.setPath(dir.absolutePath());
            if (addHistory) {
                m_history.resize(m_historyPos + 1);
                m_history << m_dir.absolutePath();
                m_historyPos = m_history.size() - 1;
            }
            m_worker->clearPath();
            m_pixmapCacheMutex.lock();
            m_pixmapCache.clear();
            m_pixmapCacheMutex.unlock();
            m_lastModifiedCache.clear();
        }

        foreach (const QFileInfo &fi, list) {
            if (fi.fileName() == "..") {
                if (!m_dir.isRoot()) {
                    m_fileInfoList.push_front(fi);
                }
                continue;
            }
            else if (!(filter() & QDir::System) &&
                     Win32::hasSystemAttribute(fi.absoluteFilePath()))
            {
                continue;
            }

            m_fileInfoList << fi;
            if (m_checkStates.find(fi.fileName()) == m_checkStates.end()) {
                m_checkStates[fi.fileName()] = Qt::Unchecked;
            }
            if (m_lastModifiedCache.find(fi.fileName()) == m_lastModifiedCache.end() ||
                m_lastModifiedCache[fi.fileName()] != fi.lastModified())
            {
                m_lastModifiedCache[fi.fileName()] = fi.lastModified();
                m_worker->addPath(fi.absoluteFilePath());
            }
        }
    }
    catch (const QString &s) {
        setError(s);
    }

    endResetModel();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::touch
/// \param name 作成するファイル名
/// \return 作成したファイルのインデックス
///
/// 空のファイルを作成します。
///
QModelIndex FolderModel::touch(const QString &name)
{
    QFile file(m_dir.absoluteFilePath(name));
    if (!file.open(QIODevice::WriteOnly)) {
        return QModelIndex();
    }
    file.close();

    refresh();

    for (int n = 0; n < rowCount(); ++n) {
        if (m_fileInfoList[n].fileName() == name) {
            return index(n, 0);
        }
    }

    return QModelIndex();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::updateAppearance
///
/// 外観を変更します。
///
void FolderModel::updateAppearance(const Preferences &prefs)
{
    qDebug() << "FolderModel::updateAppearance()";

    m_font = prefs.getFolderViewFont();
    m_Palette.setColor(QPalette::Base, prefs.folderViewBgColor(isActive()));
    m_Palette.setColor(QPalette::Text, prefs.folderViewFgColor(isActive()));
    m_Palette.setColor(QPalette::Highlight, prefs.folderViewMarkedBgColor(isActive()));
    m_Palette.setColor(QPalette::HighlightedText, prefs.folderViewMarkedFgColor(isActive()));
    m_Palette.setColor(QPalette::Dark, prefs.folderViewHiddenColor(isActive()));
    m_Palette.setColor(QPalette::Light, prefs.folderViewReadOnlyColor(isActive()));
    m_Palette.setColor(QPalette::Mid, prefs.folderViewSystemColor(isActive()));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::activeModel
/// \return アクティブなモデルを返します。
///
FolderModel *FolderModel::activeModel()
{
    return m_activeModel;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::onCdHome
///
/// ホームフォルダに移動します。
///
void FolderModel::onCdHome()
{
    setRootPath(QDir::homePath());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::onCdRoot
///
/// ルートフォルダに移動します。
///
void FolderModel::onCdRoot()
{
    setRootPath(QDir::rootPath());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::onCdUp
///
/// 親フォルダに移動します。
///
void FolderModel::onCdUp()
{
    if (!m_dir.isRoot()) {
        QDir dir(m_dir);
        dir.cdUp();
        setRootPath(dir.absolutePath());
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::onHistoryBack
///
/// 履歴を戻ります。
///
void FolderModel::onHistoryBack()
{
    if (m_historyPos > 0) {
        setRootPath(m_history[--m_historyPos], false);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::onHistoryForward
///
/// 履歴を進みます。
///
void FolderModel::onHistoryForward()
{
    if (m_historyPos < m_history.size() - 1) {
        setRootPath(m_history[++m_historyPos], false);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::onMarkAll
///
/// 全てのマークをONにします。
///
void FolderModel::onMarkAll()
{
    CheckContainer::iterator it;
    for (it = m_checkStates.begin(); it != m_checkStates.end(); ++it) {
        it.value() = Qt::Checked;
    }
    emit dataChanged(index(0, 0), index(rowCount(), columnCount()));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::onMarkAllFiles
///
/// 全てのフォルダをマークOFF、ファイルをマークONにします。
///
void FolderModel::onMarkAllFiles()
{
    foreach (const QFileInfo &fi, m_fileInfoList) {
        if (fi.fileName() == "..") {
            continue;
        }
        if (fi.isDir()) {
            m_checkStates[fi.fileName()] = Qt::Unchecked;
        }
        else {
            m_checkStates[fi.fileName()] = Qt::Checked;
        }
    }
    emit dataChanged(index(0, 0), index(rowCount(), columnCount()));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::onMarkAllOff
///
/// 全てのマークをOFFにします。
///
void FolderModel::onMarkAllOff()
{
    CheckContainer::iterator it;
    for (it = m_checkStates.begin(); it != m_checkStates.end(); ++it) {
        it.value() = Qt::Unchecked;
    }
    emit dataChanged(index(0, 0), index(rowCount(), columnCount()));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::onMarkInvert
///
/// 全てのマークを反転します。
///
void FolderModel::onMarkInvert()
{
    CheckContainer::iterator it;
    for (it = m_checkStates.begin(); it != m_checkStates.end(); ++it) {
        it.value() = (it.value() == Qt::Checked) ? Qt::Unchecked : Qt::Checked;
    }
    emit dataChanged(index(0, 0), index(rowCount(), columnCount()));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::thumbnail_Ready
/// \param path     ファイルパス
/// \param pixmap   サムネイル画像
///
/// サムネイルの生成終了時の処理を行います。
///
void FolderModel::thumbnail_Ready(const QString &path, const QPixmap &pixmap)
{
    qDebug() << "FolderModel::thumbnail_Ready()" << path;
    QFileInfo fi(path);
    if (fi.absolutePath() == m_dir.absolutePath()) {
        m_pixmapCacheMutex.lock();
        m_pixmapCache[path] = pixmap;
        m_pixmapCacheMutex.unlock();

        QModelIndex index = search(fi.fileName());
        if (index.isValid()) {
            emit dataChanged(index, index);
        }
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::rowCount
/// \param parent   (使用しません)
/// \return 行数を返します。
///
int FolderModel::rowCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent);
    return m_fileInfoList.size();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::columnCount
/// \param parent   (使用しません)
/// \return 列数を返します。
///
int FolderModel::columnCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent);
    return ColumnCount;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::data
/// \param index    インデックス
/// \param role     ロール
/// \return 該当のデータを返します。
///
QVariant FolderModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid()) {
        return QVariant();
    }

    switch (role) {
    case Qt::DisplayRole:
        switch (index.column()) {
        case Name:
            if (isDir(index) || isDotFile(index)) {
                return fileName(index);
            }
            return fileInfo(index).completeBaseName();

        case Extension:
            if (isDir(index) || isDotFile(index)) {
                return QVariant();
            }
            return fileInfo(index).suffix();

        case Size: // サイズ
            if (isDir(index)) {
                return QString("<DIR>");
            }
            return fileSizeToString(fileInfo(index).size());

        case LastModified: // 更新日時
            return fileInfo(index).lastModified().toString("yy/MM/dd hh:mm");
        }
        break;

    case Qt::DecorationRole:
        if (index.column() == Name) {
            return fileIcon(index);
        }
        break;

    case Qt::FontRole:
        return m_font;

    case Qt::TextAlignmentRole:
        switch (index.column()) {
        case Size:
        case LastModified:
            return Qt::AlignRight + Qt::AlignVCenter;
        }
        return Qt::AlignLeft + Qt::AlignVCenter;

    case Qt::BackgroundRole:
        if (isMarked(index)) {
            return marked();
        }
        return base();

    case Qt::ForegroundRole:
        if (isMarked(index)) {
            return markedText();
        }
        if (fileName(index) != ".." && Win32::hasSystemAttribute(filePath(index))) {
            return system();
        }
        if (fileName(index) != ".." && fileInfo(index).isHidden()) {
            return hidden();
        }
        if (fileName(index) != ".." && !fileInfo(index).isWritable()) {
            return readOnly();
        }
        return text();

    case Qt::CheckStateRole:
        if (index.column() == Name && fileName(index) != "..") {
            return isMarked(index) ? Qt::Checked : Qt::Unchecked;
        }
        break;
    }

    return QVariant();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::flags
/// \param index    インデックス
/// \return フラグを返します。
///
Qt::ItemFlags FolderModel::flags(const QModelIndex &index) const
{
    Qt::ItemFlags flags = Qt::ItemIsEnabled | Qt::ItemIsSelectable;

    if (!index.isValid()) {
        return flags | Qt::ItemIsDropEnabled;
    }

    if (m_fileInfoList[index.row()].fileName() != "..") {
        if (index.column() == Name) {
            flags |= Qt::ItemIsUserCheckable;
        }
        flags |= Qt::ItemIsDropEnabled;
    }
    return flags;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::setData
/// \param index    インデックス
/// \param value    値
/// \param role     ロール
/// \return 値を設定した場合はtrueを返します。
///
bool FolderModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    qDebug() << "FolderModel::setData()" << index;

    if (!index.isValid()) {
        return false;
    }

    switch (role) {
    case Qt::CheckStateRole:
        if (index.column() == Name && fileName(index) != "..") {
            m_checkStates[fileName(index)] = value.toInt();
            emit dataChanged(this->index(index.row(), 0),
                             this->index(index.row(), ColumnCount - 1));
            return true;
        }
        break;
    }

    return false;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::supportedDropActions
/// \return Qt::CopyActionを返します。
///
Qt::DropActions FolderModel::supportedDropActions() const
{
    qDebug() << "FolderModel::supportedDropActions()";

    return Qt::CopyAction;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderModel::mimeTypes
/// \return MIMEタイプのリストを返します。
///
QStringList FolderModel::mimeTypes() const
{
    qDebug() << "FolderModel::mimeTypes()";

    QStringList types;

    types << "text/uri-list";

    return types;
}
