/*
 * Copyright (C) 2009 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package android.gesture.cts;

import dalvik.annotation.TestTargetClass;

import android.gesture.GesturePoint;
import android.gesture.GestureStroke;
import android.gesture.OrientedBoundingBox;
import android.graphics.Path;
import android.graphics.RectF;

import junit.framework.TestCase;

/**
 * Simple compatibility unit tests for {@link GestureStroke}
 */
@TestTargetClass(GestureStroke.class)
public class GestureStrokeTest extends TestCase {

    private LineGestureStrokeHelper mHelper;

    @Override
    protected void setUp() throws Exception {
        mHelper = new LineGestureStrokeHelper();
    }

    /**
     * Test {@link GestureStroke#getPath()} when set of empty points are provided.
     */
    public void testGetPath_empty() {
        GestureStroke emptyStroke = mHelper.createGestureStroke();
        // null path currently returned
        assertNull(emptyStroke.getPath());
    }

    /**
     * Test {@link GestureStroke#getPath()} when a single point is provided.
     */
    public void testGetPath_singlePoint() {
        GestureStroke emptyStroke = mHelper.createGestureStroke(new GesturePoint(0, 0, 0));
        Path emptyPath = emptyStroke.getPath();
        // expect an empty path
        assertTrue(emptyPath.isEmpty());
    }

    /**
     * Test {@link GestureStroke#getPath()} when a line is provided.
     */
    public void testGetPath_line() {
        GestureStroke lineStroke = mHelper.createLineGesture();
        Path linePath = lineStroke.getPath();
        mHelper.assertLineBoundingBox(linePath);
    }

    /**
     * Test {@link android.gesture.GestureStroke#toPath(float, float, int)} returns expected results
     * for a single line, where the given toPath bounds exceeds the expected path.
     */
    public void testToPath_line() {
        // use the same
        GestureStroke lineStroke = mHelper.createLineGesture();
        // bound the path by the endpoint
        Path linePath = lineStroke.toPath(LineGestureStrokeHelper.LINE_END_POINT,
                LineGestureStrokeHelper.LINE_END_POINT, 2);

        // expect the same results generated by getPath
        mHelper.assertLineBoundingBox(linePath);
    }

    /**
     * Test for {@link android.gesture.GestureStroke#toPath(float, float, int)}
     *
     * Verifies method returns expected results for a single line, where the given toPath bounds
     * truncates the expected path.
     */
    public void testToPath_boundedLine() {
        // use the same
        GestureStroke lineStroke = mHelper.createLineGesture();
        // bound the path by the midpoint
        Path linePath = lineStroke.toPath(LineGestureStrokeHelper.LINE_MIDWAY_POINT,
                LineGestureStrokeHelper.LINE_MIDWAY_POINT, 2);
        RectF bounds = new RectF();
        linePath.computeBounds(bounds, true);
        // expect a square bounding box, starting at LINE_START_POINT and bounded by
        // LINE_QUARTER_POINT
        assertEquals(LineGestureStrokeHelper.LINE_QUARTER_POINT, bounds.bottom);
        assertEquals(LineGestureStrokeHelper.LINE_START_POINT, bounds.left);
        assertEquals(LineGestureStrokeHelper.LINE_QUARTER_POINT, bounds.right);
        assertEquals(LineGestureStrokeHelper.LINE_START_POINT, bounds.top);
    }

    /**
     * Test method for {@link android.gesture.GestureStroke#computeOrientedBoundingBox()}.
     *
     * Verifies method returns expected results for a single line.
     */
    public void testComputeOrientedBoundingBox() {
        GestureStroke line = mHelper.createLineGesture();
        OrientedBoundingBox box = line.computeOrientedBoundingBox();
        // expect a center of LINE_MIDWAY_POINT, LINE_MIDWAY_POINT, with a 45-degree orientation
        assertEquals(LineGestureStrokeHelper.LINE_MIDWAY_POINT, box.centerX);
        assertEquals(LineGestureStrokeHelper.LINE_MIDWAY_POINT, box.centerY);
        assertEquals(LineGestureStrokeHelper.LINE_ANGLE, box.orientation);
    }

    /**
     * Verifies that {@link android.gesture.GestureStroke#boundingBox} has expected values for
     * a simple straight line gesture
     */
    public void testBoundingBox_line() {
        GestureStroke line = mHelper.createLineGesture();
        mHelper.assertLineBoundingBox(line.boundingBox);
    }
}
