// Copyright (c) 2006-2009 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef NET_SOCKET_CLIENT_SOCKET_H_
#define NET_SOCKET_CLIENT_SOCKET_H_

#include "build/build_config.h"

// For struct sockaddr and socklen_t.
#if defined(OS_POSIX)
#include <sys/types.h>
#include <sys/socket.h>
#elif defined(OS_WIN)
#include <ws2tcpip.h>
#endif

#include "net/socket/socket.h"

namespace net {

class LoadLog;

class ClientSocket : public Socket {
 public:
  // Called to establish a connection.  Returns OK if the connection could be
  // established synchronously.  Otherwise, ERR_IO_PENDING is returned and the
  // given callback will run asynchronously when the connection is established
  // or when an error occurs.  The result is some other error code if the
  // connection could not be established.
  //
  // The socket's Read and Write methods may not be called until Connect
  // succeeds.
  //
  // It is valid to call Connect on an already connected socket, in which case
  // OK is simply returned.
  //
  // Connect may also be called again after a call to the Disconnect method.
  //
  virtual int Connect(CompletionCallback* callback, LoadLog* load_log) = 0;

  // Called to disconnect a socket.  Does nothing if the socket is already
  // disconnected.  After calling Disconnect it is possible to call Connect
  // again to establish a new connection.
  //
  // If IO (Connect, Read, or Write) is pending when the socket is
  // disconnected, the pending IO is cancelled, and the completion callback
  // will not be called.
  virtual void Disconnect() = 0;

  // Called to test if the connection is still alive.  Returns false if a
  // connection wasn't established or the connection is dead.
  virtual bool IsConnected() const = 0;

  // Called to test if the connection is still alive and idle.  Returns false
  // if a connection wasn't established, the connection is dead, or some data
  // have been received.
  virtual bool IsConnectedAndIdle() const = 0;

  // Identical to BSD socket call getpeername().
  // Needed by ssl_client_socket_nss and ssl_client_socket_mac.
  virtual int GetPeerName(struct sockaddr* name, socklen_t* namelen) = 0;
};

}  // namespace net

#endif  // NET_SOCKET_CLIENT_SOCKET_H_
