/*
 * Galatea Dialog Manager:
 * (c)2004 Takuya NISHIMOTO (nishimoto [atmark] m.ieice.org)
 *
 * $Id: RecogInterpreter.java,v 1.7 2009/01/28 13:51:17 nishimoto Exp $
 */
package galatea.document;

import galatea.dialog.RuntimeError;
import galatea.io.RecogInterpreterListener;
import galatea.logger.Logger;
import galatea.relaxer.event.EventEv;
import galatea.relaxer.event.EventGRAMINFO;
import galatea.relaxer.event.EventINPUT;
import galatea.relaxer.event.EventINPUTPARAM;
import galatea.relaxer.event.EventPHYPO;
import galatea.relaxer.event.EventRECOGFAIL;
import galatea.relaxer.event.EventRECOGOUT;
import galatea.relaxer.event.EventRECOGOUTRef;
import galatea.relaxer.event.EventSHYPO;
import galatea.relaxer.event.EventWHYPO;
import galatea.relaxer.event.IEventEvChoice;
import galatea.relaxer.event.IEventRECOGOUTChoice;
import galatea.scripting.ECMAScript;
import galatea.util.HashArray;
import galatea.util.Property;
import galatea.util.Util;

import java.io.StringReader;
import java.nio.charset.Charset;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class RecogInterpreter
{
	private Logger logger_ = new Logger(this.getClass());
	private String script_ = "";
	private String text_   = "";
	
	// private boolean accepted_ = false;
	private String dialogName_ = "";
	private HashArray slotAliases_ = new HashArray();
	
	private RecogInterpreterListener recogListener_;
	
	// INPUTPARAM
	private int frames_ = 0;
	private int msec_ = 0;
	
	// label:wd
	private Pattern pattern1_ = Pattern.compile("([^:]*):(.*)");
	// wd@slot=val
	private Pattern pattern2_ = Pattern.compile("(.*)@(.*)=(.*)");
	// wd@slot (assume wd=val)
	private Pattern pattern3_ = Pattern.compile("(.*)@(.*)");
	// arg1 arg2
	private Pattern pattern4_ = Pattern.compile("(\\S+) (.*)");
	// used by _parseToScript(word)
	private String parsedString_ = "";
	private String parsedSlot_   = "";
	private String parsedValue_  = "";

	private float confidenceMeasureThres_ = 0.9f;
	
	private ECMAScript phypoEcmascript_ = null;
	private String phypoEventHandler_ = "";
	private boolean enablePhypo_ = false;

	public RecogInterpreter() {
		confidenceMeasureThres_ = Property.getAsFloat("DM.CondidenceMeasureThres", 0.9f);
		if (enablePhypo_) {
			try {
				phypoEcmascript_ = new ECMAScript();
			} catch (RuntimeError e) {
				logger_.print(e.toString());
			}
		}
	}
	
	
	public void setListener(RecogInterpreterListener listener) {
		recogListener_ = listener;
	}
	
	
	public void reset() {
		_resetShypo();
		frames_ = 0;
		msec_ = 0;
		if (enablePhypo_ && (phypoEventHandler_.length() > 0)) {
			String sc = phypoEventHandler_ + "('reset')";
			logger_.print(sc);
			try {
				phypoEcmascript_.evaluate(sc);
			} catch (RuntimeError e) {
				logger_.print(e.toString());
			}
		}
	}
	
	
	private void _resetShypo() {
		text_ = "";
		script_ = "";
	}
	
	
	public void setDialogName(String dialogName) {
		dialogName_ = dialogName;
	}
	
	
	public void resetSlotAlias() {
		slotAliases_ = new HashArray();
	}
	
	
	public void setSlotAlias(String arg) {
		Matcher m;
		if ((m = pattern4_.matcher(arg)).matches()) {
			slotAliases_.put(m.group(1), m.group(2));
		}
	}
	
	
	// 
	// to @SIM set SlotAlias = boolean:yes place:石川
	// to @SIM set SlotAlias = boolean confirm
	//
	private String _makeAssignCommand(String dialog, String slot, String value) {
		String d = dialog; 
		String s = slot; 
		String v = value;
		String key = slot + ":" + value;
		if (slotAliases_.has(key)) {
			String arg = (String)slotAliases_.get(key);
			Matcher m;
			if ((m = pattern1_.matcher(arg)).matches()) {
				s = m.group(1);
				v = m.group(2);
			}
		} else if (slotAliases_.has(slot)) {
			String arg = (String)slotAliases_.get(slot);
			s = arg;
		}
		String script = "";
		script += d + "." + s + "='" + v + "';";
		script += d + "." + s + "$.justfilled=true;";
		return script;
	}
	
	
	public String getText() {
		return text_;
	}
	
	
	public String getScript() {
		return script_;
	}
	
	
	/*
	 * handle prpgressive hypothesis (pass1)
	 * evaluate: onevent('assign','field1=さようなら')
	 */
	private void _parsePhypo(EventPHYPO phypo) {
		if (!enablePhypo_) return;
		String text = "";
		for (int i = 0, n = phypo.sizeWHYPO(); i < n; i++) {
			EventWHYPO whypo = phypo.getWHYPO(i);
			String word = whypo.getWORD();
			_parseToScript(word);
			if (phypoEventHandler_.length() > 0 && parsedSlot_.length() > 0) {
				String sc = phypoEventHandler_
				+ "('assign','" + parsedSlot_ + "=" + parsedValue_ + "')";
				logger_.print(sc);
				try {
					phypoEcmascript_.evaluate(sc);
				} catch (RuntimeError e) {
					logger_.print(e.toString());
				}
			}
			text += parsedString_;
			// dbg.print("text: "+text);
		}
		if ( phypoEventHandler_.length() > 0 && text.length() > 0) {
			String sc = phypoEventHandler_ + "('phypo','" + text + "')";
			logger_.print("phypo " + phypo.getSCORE() + " " + phypo.getFRAME());
			logger_.print(sc);
			try {
				phypoEcmascript_.evaluate(sc);
			} catch (RuntimeError e) {
				logger_.print(e.toString());
			}
		}
	}
	
	/* 
	 * handle sentence hypothesis (pass2)
	 * <SHYPO RANK="1" SCORE="-3783.279297" GRAM="2">    
	 * <WHYPO WORD="silB:" CLASSID="0" PHONE="silB" CM="1.000"/>    
	 * <WHYPO WORD="新大阪@source=新大阪" CLASSID="7" PHONE="sh i N o: s a k a" CM="0.999"/>    
	 * <WHYPO WORD="silE:" CLASSID="1" PHONE="silE" CM="1.000"/>  
	 * </SHYPO>
	 * @returns true if "accept pass2" required
	 */
	private boolean _parseShypo(EventSHYPO shypo) {
		boolean accept_pass2 = false;
		_resetShypo();
		String text = "";
		String items = "";
		for (int i = 0, n = shypo.sizeWHYPO(); i < n; i++) {
			EventWHYPO whypo = shypo.getWHYPO(i);
			float cm = whypo.getCM();
			String word = whypo.getWORD();
			boolean accept_word = false;
			// CM-based rejection
			if (cm > confidenceMeasureThres_) {
				String sc = _parseToScript(word);
				if (sc.length() > 0) {
					accept_word = true;
					accept_pass2 = true;
				}
				items += sc;
				text += parsedString_;
			} else {
				text += "???";
			}
			if (accept_word) {
				logger_.print("accept " + word + " cm " + cm, 1);
			} else {
				// skip "silB:" "silE:"
				if (!word.endsWith(":")) {
					logger_.print("ignore " + word + " cm " + cm, 1);
				}
			}
		}
		script_ = "$utterance='" + text + "';";
		script_ += items;
		text_ = text;
		return accept_pass2;
	}
	
	
	/*
	 * input word : おはよう@field1=おはよう
	 */
	private String _parseToScript(String word) {
		parsedString_ = "";
		parsedSlot_ = "";
		parsedValue_ = "";
		
		if (word == null) return ""; // no slot-value pair
		
		String label="", slot="", val="", item="";
		String wd = "";
		Matcher m1, m2, m3;
		if ((m1 = pattern1_.matcher(word)).matches()) {
			// label:wd
			logger_.print("parseToScript WHYPO match1 "+word, 2);
			label = m1.group(1);
			wd    = m1.group(2);
		} else if ((m2 = pattern2_.matcher(word)).matches()) {
			// wd@slot=val
			logger_.print("parseToScript WHYPO match2 "+word, 2);
			label = m2.group(1);
			wd    = m2.group(1);
			slot  = m2.group(2);
			val   = m2.group(3);
			item  = _makeAssignCommand(dialogName_, slot, val);
		} else if ((m3 = pattern3_.matcher(word)).matches()) {
			// wd@slot (assume wd=val)
			logger_.print("parseToScript WHYPO match3 "+word, 3);
			label = m3.group(1);
			wd    = m3.group(1);
			slot  = m3.group(2);
			val   = m3.group(1);
			item  = _makeAssignCommand(dialogName_, slot, val);
		} else {
			// dbg.print("nomatch "+word);
			label = word;
			wd    = word;
		}
		parsedString_ = wd;
		parsedSlot_   = slot;
		parsedValue_  = val;
		return item;
	}
	
	
	public void receiveInputParam(EventINPUTPARAM ei) {
		frames_ = ei.getFRAMES();
		msec_ = ei.getMSEC();
	}
	
	
	public void receiveRecogout(EventRECOGOUT ro) {
		IEventRECOGOUTChoice rc = ro.getContent();
		if (enablePhypo_ && (rc instanceof EventPHYPO)) {
			_parsePhypo((EventPHYPO)rc);
		} else if (rc instanceof EventRECOGOUTRef) {
			EventRECOGOUTRef rr = (EventRECOGOUTRef)rc;
			for (int i = 0, n = rr.sizeSHYPO(); i < n; i++) {
				EventSHYPO shypo = rr.getSHYPO(i);
				if (shypo.getPASS() == 1) {
					recogListener_.recogPass1Finished();
				} else if (shypo.getRANK() == 1) {
					if (_parseShypo(shypo)) {
						recogListener_.recogPass2Finished();
						recogListener_.requestAck();
					}
				}
			}
		}
	}
	
	
	public void receiveRecogMessage(String body) {
		String evstr = "<ev src=\"SRM\" type=\"INPUT\">" + body + "</ev>";
		StringReader reader = new StringReader(evstr);
		try {
			EventEv ev = new EventEv(reader);
			IEventEvChoice evc = ev.getContent();
			if (evc instanceof EventRECOGOUT) {
				receiveRecogout((EventRECOGOUT)evc);
			} else if (evc instanceof EventINPUT) {
				EventINPUT input = (EventINPUT)evc;
				String status = input.getSTATUS();
				if (status.equals("LISTEN")) {
					recogListener_.recogListenStarted();
					reset();
				} else if (status.equals("STARTREC")) {
					recogListener_.recogRecordStarted();
				} else if (status.equals("ENDREC")) {
					recogListener_.recogRecordFinished();
				}
			} else if (evc instanceof EventRECOGFAIL) {
				recogListener_.recogFailed();
			} else if (evc instanceof EventINPUTPARAM) {
				receiveInputParam((EventINPUTPARAM)ev.getContent());
			} else if (evc instanceof EventGRAMINFO) {
				String s = ((EventGRAMINFO)evc).getContentAsString();
				s = s.replaceAll("#", "\n#");
				s = s.replaceAll("Global:", "\nGlobal:");
				logger_.print( "GRAMINFO:" + s );
			}
		} catch (Exception e) {
			recogListener_.recogException(e, body);
		}
	}
	
	//  <native>to @SIM set Script = interval.js</native>
	//  <native>to @SIM set EventHandler = onevent</native>
	public void setScript(String file) {
		if (enablePhypo_) {
			try {
				Charset cs = Util.getSystemDefaultCharset();//Charset.forName(ENCODING);
				String sc = Util.loadFromFile(file, cs);
				phypoEcmascript_.evaluate(sc);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void setEventHandler(String func) {
		if (enablePhypo_) {
			phypoEventHandler_ = func;
		}
	}
	
}
