package com.shin1ogawa.entity;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertThat;

import javax.jdo.PersistenceManager;

import org.junit.Test;

import com.google.appengine.api.datastore.Blob;
import com.shin1ogawa.AbstractRelasionShipTest;

/**
 * {@link BlobEntity}のテストケース.
 * <p>
 * {@link BlobEntity#getBlob()}は{@code defaultFetchGroup="true"}していない。
 * </p>
 * 
 * @author shin1ogawa
 */
public class BlobEntityTest extends AbstractRelasionShipTest {

	@Test
	public void PersistenceManageを開いた状態でBlobを取得する() {
		BlobEntity entity = new BlobEntity();
		entity.setValue("value");
		entity.setBlob(new Blob("あいうえお".getBytes()));
		int blobSize = "あいうえお".getBytes().length;

		PersistenceManager pm = getFactory().getPersistenceManager();
		pm.makePersistent(entity);
		pm.close();

		pm = getFactory().getPersistenceManager();
		BlobEntity objectById = pm.getObjectById(BlobEntity.class, entity
				.getKey());
		assertThat(objectById, notNullValue());
		assertThat(objectById.getValue(), is(equalTo("value")));
		assertThat(objectById.getBlob(), notNullValue());
		assertThat(objectById.getBlob().getBytes().length,
				is(equalTo(blobSize)));
		pm.close();
	}

	@Test
	public void detachしてからBlobを取得しようとする() {
		BlobEntity entity = new BlobEntity();
		entity.setValue("value");
		entity.setBlob(new Blob("あいうえお".getBytes()));
		@SuppressWarnings("unused")
		int blobSize = "あいうえお".getBytes().length;

		PersistenceManager pm = getFactory().getPersistenceManager();
		pm.makePersistent(entity);
		pm.close();

		pm = getFactory().getPersistenceManager();
		BlobEntity objectById = pm.getObjectById(BlobEntity.class, entity
				.getKey());
		objectById = pm.detachCopy(objectById); // detach
		assertThat(objectById, notNullValue());
		assertThat(objectById.getValue(), is(equalTo("value")));
		assertThat(objectById.getBlob(), nullValue()); // nullが返ってくる。
		pm.close();
	}

	@Test
	public void PersitenceManagerを閉じてからBlobを取得しようとする() {
		BlobEntity entity = new BlobEntity();
		entity.setValue("value");
		entity.setBlob(new Blob("あいうえお".getBytes()));
		@SuppressWarnings("unused")
		int blobSize = "あいうえお".getBytes().length;

		PersistenceManager pm = getFactory().getPersistenceManager();
		pm.makePersistent(entity);
		pm.close();

		pm = getFactory().getPersistenceManager();
		BlobEntity objectById = pm.getObjectById(BlobEntity.class, entity
				.getKey());
		pm.close();
		assertThat(objectById, notNullValue());
		assertThat(objectById.getValue(), is(equalTo("value")));
		assertThat(objectById.getBlob(), nullValue()); // nullが返ってくる。
	}

	@Test
	public void PersitenceManagerをretrieveしてから閉じてBlobを取得しようとする() {
		BlobEntity entity = new BlobEntity();
		entity.setValue("value");
		entity.setBlob(new Blob("あいうえお".getBytes()));
		@SuppressWarnings("unused")
		int blobSize = "あいうえお".getBytes().length;

		PersistenceManager pm = getFactory().getPersistenceManager();
		pm.makePersistent(entity);
		pm.close();

		pm = getFactory().getPersistenceManager();
		BlobEntity objectById = pm.getObjectById(BlobEntity.class, entity
				.getKey());
		pm.retrieve(objectById);
		pm.close();
		assertThat(objectById, notNullValue());
		assertThat(objectById.getValue(), is(equalTo("value")));
		assertThat(objectById.getBlob(), notNullValue()); // 取得できている。
	}
}
