/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "thread_list_model.hxx"

#include <gtkmm/treemodel.h>
#include <gtkmm/treepath.h>
#include <boost/mpl/size.hpp>
#include <boost/mpl/for_each.hpp>
#include <boost/mpl/find.hpp>
#include <boost/mpl/pop_front.hpp>
#include <boost/foreach.hpp>
#include <boost/function.hpp>


namespace dialektos {

namespace model_column {

struct GetColumnData {
  GetColumnData(
      const ModelColumns& model_columns,
      size_t index,
      Glib::ValueBase& data)
  : index_(index), model_columns_(model_columns), data_(data) {
        assert(index < unsigned(boost::mpl::size<ModelColumns::type>::value));
      }

  template <typename TypeHolderT>
  void operator()(const TypeHolderT& /*t*/) {
    using namespace boost::mpl;
    if (index_ == unsigned(find<ModelColumns::type, TypeHolderT>::type::pos::value)) {
      Glib::Value<typename TypeHolderT::type> value;
      value.init(Glib::Value<typename TypeHolderT::type>::value_type());

      value.set(field<TypeHolderT>(model_columns_));
      data_.init(Glib::Value<typename TypeHolderT::type>::value_type());
      data_ = value;
    }
  }

  const size_t index_;
  const dialektos::ModelColumns& model_columns_;
  Glib::ValueBase& data_;
};

struct GetColumnGType {
  GetColumnGType(size_t index, GType& data) : index_(index), data_(data) {
        assert(index < unsigned(boost::mpl::size<ModelColumns::type>::value));
      }

  template <typename TypeHolderT>
  void operator()(const TypeHolderT& /*t*/) {
    using namespace boost::mpl;
    if (index_ == unsigned(find<ModelColumns::type, TypeHolderT>::type::pos::value)) {
      data_ = Glib::Value<typename TypeHolderT::type>::value_type();
    }
  }

  const size_t index_;
  GType& data_;
};

} // namespace model_column


Glib::RefPtr<ThreadListModel> ThreadListModel::create() {
  return Glib::RefPtr<ThreadListModel>(new ThreadListModel());
}

ThreadListModel::ThreadListModel():
  Glib::ObjectBase(typeid(ThreadListModel)),
  Glib::Object(),
  list_(), order_(), sort_function_(Compare<model_column::Number>()) {
}

ThreadListModel::~ThreadListModel() {}

void ThreadListModel::append(const ModelColumns& record) {
  const model_column::ID::type& id = model_column::field<model_column::ID>(record);
  OrderType::iterator it = order_.find(id);
  if (it != order_.end()) {
    list_[it->second] = record;
  } else {
    list_.push_back(record);
    order_[id] = list_.size()-1;

    const size_t index = list_.size() -1;
    iterator iter;
    iter.gobj()->stamp = 1;
    iter.gobj()->user_data = GINT_TO_POINTER(index);
    iter.gobj()->user_data2 = 0;
    iter.gobj()->user_data3 = 0;

    Gtk::TreeModel::Path path;
    path.append_index(index);
    row_inserted(path, iter);
  }
}

void ThreadListModel::set_buffer(
    const boost::unordered_map<model_column::ID::type, ModelColumns>& buffer) {

  // remove all rows
  for(size_t i = 0; i != list_.size(); i++) {
    Gtk::TreeModel::Path path;
    path.append_index(0);
    row_deleted(path);
  }

  list_ = StoreType();
  order_ = OrderType();

  typedef std::pair<model_column::ID::type, ModelColumns> PairType;
  BOOST_FOREACH(const PairType& pair, buffer) {
    const ModelColumns& cols = pair.second;
    append(cols);
  }

  sort();
}

void ThreadListModel::sort() {
  std::sort(list_.begin(), list_.end(), sort_function_);
  do_after_sort();
}

void ThreadListModel::do_after_sort() {
  // for informing to the tree view.
  std::vector<int> new_order;
  new_order.reserve(order_.size());

  // set new order.
  for (size_t i = 0; i != list_.size(); i++) {
    const model_column::ID::type& id =
      model_column::field<model_column::ID>(list_[i]);
    new_order.push_back(order_[id]);
    order_[id] = i;
  }

  // inform new order to the tree view.
  if (!new_order.empty())
    rows_reordered(Gtk::TreeModel::Path(), &(new_order[0]));
}

void ThreadListModel::get_buffer(
    boost::unordered_map<model_column::ID::type, ModelColumns>& buffer) const {
  buffer.clear();
  BOOST_FOREACH(const ModelColumns& record, list_) {
    const model_column::ID::type& id =
      model_column::field<model_column::ID>(record);
    buffer[id] = record;
  }
}

const ModelColumns& ThreadListModel::get_model_columns(const size_t row_index) const {
  assert(row_index < list_.size());
  return list_[row_index];
}

void ThreadListModel::update_row(const ModelColumns& record) {
  const model_column::ID::type& id = model_column::field<model_column::ID>(record);

  OrderType::iterator it = order_.find(id);
  if (it != order_.end()) {
    const size_t row_index = it->second;

    model_column::field<model_column::Title>(list_[row_index]) =
      model_column::field<model_column::Title>(record);
    model_column::field<model_column::LineCount>(list_[row_index]) =
      model_column::field<model_column::LineCount>(record);

    Gtk::TreePath path;
    path.append_index(row_index);
    iterator iter;
    iter.set_stamp(1);
    iter.gobj()->user_data = GINT_TO_POINTER(row_index);
    row_changed(path, iter);
  } else {
    append(record);
  }
}


GType ThreadListModel::get_column_type_vfunc(const int index) const {
  assert(index >= 0 && index < boost::mpl::size<ModelColumns::type>::value);

  GType value = 0;
  model_column::GetColumnGType functor(index, value);
  boost::mpl::for_each<ModelColumns::type>(functor);
  return value;
}

Gtk::TreeModelFlags ThreadListModel::get_flags_vfunc() const {
  return Gtk::TREE_MODEL_LIST_ONLY;
}

int ThreadListModel::get_n_columns_vfunc() const {
  return boost::mpl::size<ModelColumns::type>::value;
}

bool ThreadListModel::get_iter_vfunc(const Gtk::TreeModel::Path& path, iterator& iter) const {
  assert(path.get_depth() == 1); // no children.

  iter = iterator();

  if (path.empty()) return false;

  const size_t index = path.front();

  if (index >= list_.size()) return false;

  iter.set_stamp(1);
  iter.gobj()->user_data = GINT_TO_POINTER(index);

  return true;
}

Gtk::TreeModel::Path ThreadListModel::get_path_vfunc(const iterator& iter) const {
  const GtkTreeIter* gtktreeiter = iter.gobj();
  const size_t index = GPOINTER_TO_INT(gtktreeiter->user_data);

  Gtk::TreeModel::Path path;
  path.append_index(index);
  return path;
}

void ThreadListModel::get_value_vfunc(const iterator& iter, const int column,
    Glib::ValueBase& value) const {

  if (iter.get_stamp() != 1) return;
  if (column >= boost::mpl::size<model_column::List>::value) return;

  const size_t index = GPOINTER_TO_INT(iter.gobj()->user_data);
  if (index >= order_.size()) return;

  const ModelColumns& record = list_[index];

  model_column::GetColumnData functor(record, column, value);
  boost::mpl::for_each<model_column::List>(functor);
}

bool ThreadListModel::iter_children_vfunc(const iterator& parent,
    iterator& iter) const {
  return iter_nth_child_vfunc(parent, 0, iter);
}

bool ThreadListModel::iter_has_child_vfunc(const iterator& iter) const {
  return false;
}

bool ThreadListModel::iter_is_valid(const iterator& iter) const {
  return iter.gobj()->stamp == 1 && Gtk::TreeModel::iter_is_valid(iter);
}

int ThreadListModel::iter_n_children_vfunc(const iterator& iter) const {
  return 0;
}

int ThreadListModel::iter_n_root_children_vfunc() const {
  return list_.size();
}

bool ThreadListModel::iter_next_vfunc(const iterator& iter, iterator& iter_next) const {
  iter_next = iterator();

  if (iter.get_stamp() != 1) return false;

  size_t index = GPOINTER_TO_INT(iter.gobj()->user_data);
  ++index;

  if (index >= list_.size()) return false;

  iter_next.gobj()->stamp = 1;
  iter_next.gobj()->user_data = GINT_TO_POINTER(index);

  return true;
}

bool ThreadListModel::iter_nth_child_vfunc(const iterator& parent, int n,
    iterator& iter) const {
  iter = iterator();
  return false;
}

bool ThreadListModel::iter_nth_root_child_vfunc(int n, iterator& iter) const {
  if (unsigned(n) >= list_.size()) return false;

  const unsigned int row_index = n;
  iter = iterator();
  iter.set_stamp(1);
  iter.gobj()->user_data = GINT_TO_POINTER(row_index);
  return true;
}

bool ThreadListModel::iter_parent_vfunc(const iterator& child, iterator& iter) const {
  iter = iterator();
  return false;
}


} // namespace dialektos
