/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef THREAD_IDX_CACHE_HXX
#define THREAD_IDX_CACHE_HXX

#include <boost/serialization/access.hpp>
#include <boost/serialization/nvp.hpp>
#include <boost/filesystem.hpp>
#include <boost/unordered_map.hpp>
#include <boost/unordered_set.hpp>
#include <string>
#include <vector>


namespace dialektos {


struct DirectoryTimeStamp {
  std::string filename;
  std::time_t last_modified;
private:
  friend class boost::serialization::access;
  template <typename ArchiveType>
  void serialize(ArchiveType& ar, const unsigned int version) {
    ar & boost::serialization::make_nvp("Directory", filename);
    ar & boost::serialization::make_nvp("LastModified", last_modified);
  }
};


struct ThreadIdxCache {

  typedef std::string ThreadID;

  std::string id_;
  std::string title_;
  std::string idx_last_modified_;
  int line_count_;

  static std::vector<ThreadIdxCache> from_directory(
      const boost::filesystem::path&);

private:

  static std::vector<ThreadIdxCache> from_xml(const boost::filesystem::path&);
  static void to_xml(const boost::filesystem::path&,
      const std::vector<ThreadIdxCache>&);
  static void idx_dir_scan(const boost::filesystem::path&,
      boost::unordered_map<ThreadID, ThreadIdxCache>&);
  static void merge_idx(const boost::filesystem::path&,
      const boost::unordered_set<ThreadID>&,
      boost::unordered_map<ThreadID, ThreadIdxCache>&);
  static void remove_deleted_ids(
      boost::unordered_map<ThreadID, ThreadIdxCache>&,
      const std::vector<ThreadID>& deleted_ids);
  static std::vector<ThreadID> get_deleted_ids(
      const boost::unordered_map<ThreadID, ThreadIdxCache>&,
      const boost::unordered_set<ThreadID>& exist_ids,
      const std::string& prefix);
  static boost::unordered_set<ThreadID> get_exist_ids(
        const boost::filesystem::path&);
  static std::vector<DirectoryTimeStamp> directory_timestamp_from_xml(
      const boost::filesystem::path&);
  static void directory_timestamp_to_xml(
      const boost::filesystem::path&, const std::vector<DirectoryTimeStamp>&);


  friend class boost::serialization::access;
  template <typename ArchiveType>
  void serialize(ArchiveType& ar, const unsigned int version) {
    ar & id_;
    ar & title_;
    ar & idx_last_modified_;
    ar & line_count_;
  }
};


} // namespace dialektos


#endif
