/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "text_view_pointer_trackable.hxx"

#include <gdkmm/cursor.h>
#include <boost/foreach.hpp>
#include <boost/bind.hpp>
#include <boost/ref.hpp>
#include <functional>
#include <typeinfo>
#include "text_element_plain.hxx"
#include "text_element_anchor.hxx"
#include "text_element_id.hxx"
#include "text_element_res_num.hxx"
#include "text_view_drawing_set.hxx"
#include "text_line.hxx"


namespace dialektos {


namespace text_view {


PointerTrackable::PointerTrackable(): Layoutable(),
  pressed_element_(0), pressed_index_(0),
  hovered_element_(0), hovered_index_(0),
  pressed_anchor_(0) {
  add_events(Gdk::POINTER_MOTION_MASK);
  add_events(Gdk::BUTTON_PRESS_MASK);
  add_events(Gdk::BUTTON_RELEASE_MASK);
}

PointerTrackable::~PointerTrackable() {}

Glib::ustring PointerTrackable::get_selected_text() const {
  if (!pressed_element_ || !hovered_element_) return "";

  GetSelectedSet set;
  set.start_element = pressed_element_;
  set.start_index = pressed_index_;
  set.end_element = hovered_element_;
  set.end_index = hovered_index_;

  std::pair<gdouble, gdouble> pressed_xy = pressed_element_->get_xy();
  std::pair<gdouble, gdouble> hovered_xy = hovered_element_->get_xy();

  if ((pressed_element_ == hovered_element_
      && pressed_index_ > hovered_index_) ||
      (pressed_xy.second == hovered_xy.second
          && pressed_xy.first > hovered_xy.first) ||
          (pressed_xy.second > hovered_xy.second)) {
    std::swap(set.start_element, set.end_element);
    std::swap(set.start_index, set.end_index);
  }

  LineListType::const_iterator it_start = std::find_if(
      line_list_.begin(), line_list_.end(),
      boost::bind(&TextLine::includes, _1, boost::ref(*set.start_element)));
  if (it_start == line_list_.end()) return "";

  LineListType::const_iterator it_end = std::find_if(
      line_list_.begin(), line_list_.end(),
      boost::bind(&TextLine::includes, _1, boost::ref(*set.end_element)));
  if (it_end == line_list_.end()) return "";
  ++it_end;

  Glib::ustring selected;
  for (LineListType::const_iterator it = it_start;;) {
    selected += it->get_selected_text(set);
    ++it;
    if (it == it_end) break;
    selected += "\n";
  }
  return selected;
}

void PointerTrackable::on_anchor_click_event(const text_element::Anchor&) {}
void PointerTrackable::on_anchor_hovered_event(const text_element::Anchor&) {}

bool PointerTrackable::on_button_press_event(GdkEventButton* event) {
  if (event->button == 1) {
    return on_left_button_press_event(event);
  } else if (event->button == 3) {
    return on_right_button_press_event(event);
  }
  return false;
}

bool PointerTrackable::on_button_release_event(GdkEventButton* event) {

  if (event->button != 1)
    return false;

  const text_element::Anchor* anchor = pressed_anchor_;
  pressed_anchor_ = 0;

  if (anchor) {
    const text_element::Plain* elem
      = get_text_element(event->x, event->y + adjustment_.get_value());
    if (anchor == elem && typeid(*anchor) == typeid(text_element::Anchor)) {
      // emit the event only when element is Anchor.
      // do not emit when ID and ResNum.
      on_anchor_click_event(*anchor);
    }
  }

  return true;
}

bool PointerTrackable::on_motion_notify_event(GdkEventMotion* event) {

  set_cursor(event->x, event->y);

  if (event->state & Gdk::BUTTON1_MASK) {
    if (pressed_anchor_)
      return true;

    // get the element under the pointer
    set_hovered(event->x, event->y);

    queue_draw();

    if (event->y > get_height()) {
      adjustment_.set_value(adjustment_.get_value() + 10);
    } else if (event->y < 0) {
      adjustment_.set_value(adjustment_.get_value() - 10);
    }

    return true;
  }

  using text_element::ID;
  using text_element::ResNum;
  using text_element::Anchor;

  const text_element::Plain* pressed_element =
    get_text_element(event->x, event->y + adjustment_.get_value());

  if (const Anchor* elem = dynamic_cast<const Anchor*>(pressed_element)) {
    on_anchor_hovered_event(*elem);
    return true;
  }

  return false;
}

bool PointerTrackable::on_left_button_press_event(GdkEventButton* event) {
  using text_element::ID;
  using text_element::ResNum;
  using text_element::Anchor;

  const text_element::Plain* pressed_element =
    get_text_element(event->x, event->y + adjustment_.get_value());

  pressed_anchor_ = dynamic_cast<const Anchor*>(pressed_element);

  if (pressed_anchor_) {
    // do not modify the selection.
    return true;
  }

  hovered_element_ = 0;
  hovered_index_ = 0;

  // get pressed element
  gdouble x = event->x;
  gdouble y_adj = event->y + adjustment_.get_value();
  pressed_element_ = get_nearest_text_element(x, y_adj);

  if (pressed_element_) {
    // get index of the cursor position
    pressed_index_ = pressed_element_->xy_to_index(x, y_adj);
  }

  queue_draw();

  return true;
}

bool PointerTrackable::on_right_button_press_event(GdkEventButton* event) {
  return false;
}


void PointerTrackable::set_hovered(gdouble x, gdouble y) {
  gdouble y_adj = y + adjustment_.get_value();
  const text_element::Plain* element = get_nearest_text_element(x, y_adj);
  if (!element) return;

  hovered_element_ = element;
  hovered_index_ = element->xy_to_index(x, y_adj);
}

void PointerTrackable::set_cursor(const gdouble x, const gdouble y) {
  const gdouble y_adj = y + adjustment_.get_value();
  const text_element::Plain* element = get_text_element(x, y_adj);
  if (!element) {
    Gdk::Cursor cursor(Gdk::LEFT_PTR);
    get_window()->set_cursor(cursor);
  } else {
    Gdk::Cursor cursor(element->get_cursor_type());
    get_window()->set_cursor(cursor);
  }
}



} // namespace text_view


} // namespace dialektos
