/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "text_element_plain.hxx"

#include <pangomm/layout.h>
#include <pangomm/attributes.h>
#include <pangomm/attrlist.h>
#include <pangomm/context.h>
#include <pangomm/font.h>
#include <pangomm/rectangle.h>
#include <pangomm/item.h>
#include <pangomm/glyphstring.h>
#include <glibmm/refptr.h>
#include <glibmm/ustring.h>
#include <gdkmm/gc.h>
#include <gdkmm/cursor.h>
#include <gdkmm/rectangle.h>
#include <boost/foreach.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/lambda/lambda.hpp>
#include <boost/array.hpp>
#include "text_view_drawing_set.hxx"
#include "text_element_char_size_cache.hxx"


namespace dialektos {

namespace text_element {


/* static data members */
CharSizeCache Plain::char_size_cache;


double Plain::get_approximate_char_height(const Pango::FontMetrics& metrics) {
  return double(
      metrics.get_ascent() +
      metrics.get_descent())
      / Pango::SCALE * 1.2;
}

void Plain::trim_right() {
  using namespace boost::lambda;

  boost::trim_right_if(str_, _1 == ' ');
}

Glib::ustring Plain::get_selected_text(
    const text_view::GetSelectedSet& set) const {
  size_t start_index = 0;
  size_t end_index = 0;
  get_selection_start_end_index(set.start_element, set.start_index,
      set.end_element, set.end_index, start_index, end_index);
  return str_.substr(start_index, end_index-start_index);
}

void Plain::layout(text_view::LayoutSet& set) {

  /* caching for drawing */
  x_ = set.x;
  y_ = set.y;
  x_start_ = set.x_start;
  x_end_ = set.x_end;

  BOOST_FOREACH(const Pango::Item& item, items_) {
    Glib::RefPtr<Pango::Font> font = item.get_analysis().get_font();
    const int font_id = reinterpret_cast<long int>(font->gobj());
    BOOST_FOREACH(const gunichar& uch, item.get_segment(str_)) {
      const double width = char_size_cache.get_char_width(uch, item, font_id);
      if (set.x + width > x_end_) {
        set.x = x_start_;
        set.y += get_approximate_char_height(metrics_);
      }
      set.x += width;
    }
  }
}

void Plain::draw(text_view::DrawingSet& set) const {
  size_t start_index = 0;
  size_t end_index = 0;
  get_selection_start_end_index(set.start_element, set.start_index,
      set.end_element, set.end_index, start_index, end_index);

  double x = x_;
  double y = y_;

  size_t str_count = 0;
  size_t str_start = 0;

  BOOST_FOREACH(Pango::Item item, items_) {
    const Glib::ustring sub = item.get_segment(str_);
    double draw_start_x = x;
    double draw_start_y = y;
    Glib::ustring text;
    bool need_draw = false;

    BOOST_FOREACH(const gunichar& uch, sub) {
      const double width = char_size_cache.get_char_width(uch, item);
      if (x + width > x_end_) {
        need_draw = true;
        x = x_start_;
        y += get_approximate_char_height(metrics_);
      }

      if (str_count == start_index && str_count != end_index)
        need_draw = true;
      if (str_count == end_index && str_count != start_index)
        need_draw = true;

      if (need_draw) {

        if (!text.empty())
          do_draw(set, item, item.shape(text), draw_start_x, draw_start_y,
              str_start >= start_index && str_count <= end_index);

        str_start = str_count;
        draw_start_x = x;
        draw_start_y = y;
        text.clear();
        need_draw = false;
      }

      x += width;
      ++str_count;
      text += uch;
    }

    if (!text.empty())
      do_draw(set, item, item.shape(text), draw_start_x, draw_start_y,
          str_start >= start_index && str_count <= end_index);

  }
}


void Plain::do_draw(text_view::DrawingSet& set,
    const Pango::Item& item, const Pango::GlyphString& glyphs,
    double x, double y, bool in_selection) const {

  // draw a rectangle for a selected region background.
  if (in_selection) {
    set.window->draw_rectangle(
        set.style->get_base_gc(Gtk::STATE_SELECTED),
        true, std::ceil(x), y - set.adj_value,
        std::ceil(double(glyphs.get_width()) / Pango::SCALE),
        (metrics_.get_ascent() + metrics_.get_descent()) / Pango::SCALE);
  }

  do_draw_glyphs(set, item, glyphs, x, y, in_selection);
}

void Plain::do_draw_glyphs(text_view::DrawingSet& set,
    const Pango::Item& item, const Pango::GlyphString& glyphs,
    double x, double y, bool in_selection) const {

  Glib::RefPtr<const Gdk::GC> gc = set.style->get_text_gc(
      in_selection ? Gtk::STATE_SELECTED : Gtk::STATE_NORMAL);

  set.window->draw_glyphs(gc, item.get_analysis().get_font(),
      std::ceil(x),
      y - set.adj_value + metrics_.get_ascent() / Pango::SCALE, glyphs);
}

Pango::FontMetrics Plain::get_metrics() const {
  return metrics_;
}


bool Plain::is_xy_on_this(const gdouble x, const gdouble y) const {

  gdouble x_pos = x_;
  gdouble y_pos = y_;

  BOOST_FOREACH(Pango::Item item, items_) {
    BOOST_FOREACH(const gunichar& uch, item.get_segment(str_)) {
      const double w = char_size_cache.get_char_width(uch, item);

      if (x_pos + w > x_end_) {
        // begin new line
        x_pos = x_start_;
        y_pos += get_approximate_char_height(metrics_);
      }

      if (x >= x_pos && x < x_pos + w &&
          y >= y_pos && y < y_pos + get_approximate_char_height(metrics_)) {
        return true;
      }

      x_pos += w;
    }
  }

  return false;
}

bool Plain::is_xy_near_to_this(const gdouble x, const gdouble y) const {
  gdouble x_pos = x_;
  gdouble y_pos = y_;

  BOOST_FOREACH(Pango::Item item, items_) {
    BOOST_FOREACH(const gunichar& uch, item.get_segment(str_)) {
      const double w = char_size_cache.get_char_width(uch, item);

      if (x_pos + w > x_end_) {
        // begin new line

        if (x >= x_pos && y >= y_pos &&
            y < y_pos + get_approximate_char_height(metrics_)) {
          // (x, y) is on the spaces caused by wrapping
          return true;
        }

        x_pos = x_start_;
        y_pos += get_approximate_char_height(metrics_);
      }

      if (x >= x_pos && x < x_pos + w &&
          y >= y_pos && y < y_pos + get_approximate_char_height(metrics_)) {
        return true;
      }

      x_pos += w;
    }
  }

  return false;
}


Gdk::Rectangle Plain::xy_to_rectangle(gdouble x, gdouble y) const {
  gdouble x_pos = x_;
  gdouble y_pos = y_;
  BOOST_FOREACH(Pango::Item item, items_) {
    BOOST_FOREACH(const gunichar& uch, item.get_segment(str_)) {
      const double w = char_size_cache.get_char_width(uch, item);

      if (x_pos + w > x_end_) {
        // begin new line
        x_pos = x_start_;
        y_pos += get_approximate_char_height(metrics_);
      }

      if (x >= x_pos && x < x_pos + w &&
          y >= y_pos && y < y_pos + get_approximate_char_height(metrics_))
        return Gdk::Rectangle(
            x_pos, y_pos, w, get_approximate_char_height(metrics_));

      x_pos += w;
    }
  }

  return Gdk::Rectangle(
      x_pos, y_pos, 0, get_approximate_char_height(metrics_));
}


int Plain::xy_to_index(const gdouble x, const gdouble y) const {

  gdouble x_pos = x_;
  gdouble y_pos = y_;
  int index = 0;

  BOOST_FOREACH(Pango::Item item, items_) {
    BOOST_FOREACH(const gunichar& uch, item.get_segment(str_)) {
      const double w = char_size_cache.get_char_width(uch, item);

      if (x_pos + w > x_end_) {
        // begin new line
        if (x >= x_pos &&
            y >= y_pos && y < y_pos + get_approximate_char_height(metrics_)) {
          // (x, y) is on the spaces caused by wrapping
          return index;
        }

        x_pos = x_start_;
        y_pos += get_approximate_char_height(metrics_);
      }

      if (x >= x_pos && x < x_pos + w &&
          y >= y_pos && y < y_pos + get_approximate_char_height(metrics_)) {
        if (x <= x_pos + w/2) {
          // left of this character
          return index;
        } else {
          // right of this character
          return index+1;
        }
      }

      x_pos += w;
      ++index;
    }
  }
  return index;
}

Gdk::CursorType Plain::get_cursor_type() const {
  return Gdk::XTERM;
}

void Plain::itemize(const Glib::RefPtr<Pango::Context>& context,
    const Pango::FontMetrics& metrics) {
  Pango::AttrList list = Pango::AttrList();
  Pango::AttrInt attr = bold_ ?
      Pango::Attribute::create_attr_weight(Pango::WEIGHT_BOLD):
      Pango::Attribute::create_attr_weight(Pango::WEIGHT_NORMAL);
  list.insert(attr);
  if (!str_.empty()) items_ = context->itemize(str_, list);
  metrics_ = metrics;
}

void Plain::get_selection_start_end_index(
    const Plain* const start_element, const size_t start_index,
    const Plain* const end_element, const size_t end_index,
    size_t& start, size_t& end) const {

  start = 0;
  end = 0;
  if (start_element && end_element) {
    if (start_element == this && end_element == this) {
      start = start_index;
      end = end_index;
    } else if (start_element == this) {
      start = start_index;
      end = str_.size();
    } else if (end_element == this) {
      end = end_index;
    } else if (start_element->y_ < y_ && end_element->y_ > y_) {
      end = str_.size();
    } else if (start_element->y_ == y_ && end_element->y_ == y_ &&
        start_element->x_ < x_ && end_element->x_ > x_) {
      end = str_.size();
    } else if (start_element->y_ == y_ && end_element->y_ != y_ &&
        start_element->x_ < x_) {
      end = str_.size();
    } else if (end_element->y_ == y_ && start_element->y_ != y_ &&
        end_element->x_ > x_) {
      end = str_.size();
    }
  }

}


} // namespace text_element

} // namespace dialektos
