/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "http_header.hxx"

#include <boost/xpressive/xpressive.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string.hpp>
#include <string>
#include <map>


namespace dialektos {

namespace http {


Header::Header() : map_() {
}

Header::Header(const Header& rhs) : map_(rhs.map_) {
}

Header::~Header() {}

void Header::set_header_from_line(const std::string& _line) {
  using namespace boost::xpressive;
  const sregex regex = (s1=+set[alnum | '-']) >> ":" >> _s >> (s2=-+_);

  const std::string line = boost::algorithm::trim_right_copy(_line);
  smatch what;
  if (regex_match(line, what, regex)) {
    const std::string name = what[1];
    const std::string value = what[2];
    set_header(name, value);
  }
}

void Header::set_header(const std::string& name, const std::string& value) {
  if (!value.empty()) map_[name] =  value;
}

void Header::set_host(const std::string& value) {
  set_header("Host", value);
}

void Header::set_if_modified_since(const std::string& value) {
  set_header("If-Modified-Since", value);
}

void Header::set_if_none_match(const std::string& value) {
  set_header("If-None-Match", value);
}

void Header::set_range(const int value) {
  const std::string text = boost::lexical_cast<std::string>(value);
  set_header("Range", "bytes=" + text + "-");
}

std::string Header::get_header(const std::string& name) const {
  MapType::const_iterator it = map_.find(name);
  if (it != map_.end()) return it->second;
  return "";
}

std::string Header::get_host() const {
  return get_header("Host");
}

std::string Header::get_last_modified() const {
  return get_header("Last-Modified");
}

std::string Header::get_etag() const {
  return get_header("ETag");
}

std::string Header::get_range() const {
  return get_header("Range");
}


} // namespace http

} // namespace dialektos
