/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "http_client.hxx"

#include <boost/asio.hpp>
#include <boost/bind.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/foreach.hpp>
#include <boost/format.hpp>
#include <iostream>
#include <istream>
#include <ostream>
#include <string>
#include <map>
#include "http_response.hxx"
#include "http_status_line.hxx"


namespace dialektos {

namespace http {


AsyncClient::AsyncClient(
    boost::asio::io_service& io_service, const std::string& uri,
    const Header& header) :
      resolver_(io_service),
      socket_(io_service),
      request_(),
      response_(),
      content_(),
      http_version_(),
      status_code_(),
      status_message_(),
      response_header_(),
      err_() {

  const std::string host = header.get_host();
  assert(!host.empty());

  std::ostream stream(&request_);

  stream << boost::format("GET %1% HTTP/1.0\r\n") % uri;

  typedef std::map<std::string, std::string>::value_type PairType;
  BOOST_FOREACH(const PairType& pair, header) {
    stream << boost::format("%1%: %2%\r\n") % pair.first % pair.second;
  }
  stream << "\r\n" << std::flush;

  boost::asio::ip::tcp::resolver::query query(host, "http");
  resolver_.async_resolve(query,
      boost::bind(&AsyncClient::handle_resolve, this,
        boost::asio::placeholders::error,
        boost::asio::placeholders::iterator));
}

void AsyncClient::handle_resolve(const boost::system::error_code& err,
    boost::asio::ip::tcp::resolver::iterator endpoint_iterator) {
  if (err) {
    err_ = err;
    return;
  }

  boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;
  socket_.async_connect(endpoint,
      boost::bind(&AsyncClient::handle_connect, this,
          boost::asio::placeholders::error, ++endpoint_iterator));
}

void AsyncClient::handle_connect(const boost::system::error_code& err,
    boost::asio::ip::tcp::resolver::iterator endpoint_iterator) {
  if (!err) {
    boost::asio::async_write(socket_, request_,
        boost::bind(&AsyncClient::handle_write_request, this,
          boost::asio::placeholders::error));
  } else if (endpoint_iterator != boost::asio::ip::tcp::resolver::iterator()) {
    socket_.close();
    boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;
    socket_.async_connect(endpoint,
        boost::bind(&AsyncClient::handle_connect, this,
          boost::asio::placeholders::error, ++endpoint_iterator));
  } else {
    err_= err;
  }
}

void AsyncClient::handle_write_request(const boost::system::error_code& err) {
  if (err) {
    err_ = err;
    return;
  }

  boost::asio::async_read_until(socket_, response_, "\r\n",
      boost::bind(&AsyncClient::handle_read_status_line, this,
          boost::asio::placeholders::error));
}

void AsyncClient::handle_read_status_line(
    const boost::system::error_code& err) {
  if (err) {
    err_ = err;
    return;
  }

  std::istream stream(&response_);

  stream >> http_version_;
  stream >> status_code_;
  std::getline(stream, status_message_);
  boost::algorithm::trim(status_message_);

  if (!stream || !boost::algorithm::starts_with(http_version_, "HTTP/")) {
    std::cerr << "response is not HTTP" << std::endl;
    return;
  }

  boost::asio::async_read_until(socket_, response_, "\r\n\r\n",
      boost::bind(&AsyncClient::handle_read_headers, this,
          boost::asio::placeholders::error));
}

void AsyncClient::handle_read_headers(const boost::system::error_code& err) {
  if (err) {
    err_ = err;
    return;
  }

  std::istream stream(&response_);
  std::string line;
  while (std::getline(stream, line)) {
    boost::algorithm::trim_right(line);
    if (line.empty()) break;
    response_header_.set_header_from_line(line);
  }

  if (response_.size() > 0)
    content_ << &response_ << std::flush;

  boost::asio::async_read(socket_, response_,
      boost::asio::transfer_at_least(1),
      boost::bind(&AsyncClient::handle_read_content, this,
          boost::asio::placeholders::error));
}

void AsyncClient::handle_read_content(const boost::system::error_code& err)
{
  if (!err) {
    content_ << &response_ << std::flush;

    boost::asio::async_read(socket_, response_,
        boost::asio::transfer_at_least(1),
        boost::bind(&AsyncClient::handle_read_content, this,
          boost::asio::placeholders::error));
  } else if (err != boost::asio::error::eof) {
    err_ = err;
  }
}

Response AsyncClient::get_response() const {
  std::stringstream ss;
  ss << boost::format("%1% %2% %3%")
    % http_version_ % status_code_ % status_message_;
  std::string status_line = ss.str();

  StatusLine status(http_version_, status_code_, status_message_, status_line);
  return Response(status, response_header_, content_.str());
}


} // namespace http

} // namespace dialektos
