/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "bookmark_item.hxx"

#include <boost/archive/xml_iarchive.hpp>
#include <boost/archive/xml_oarchive.hpp>
#include <boost/serialization/nvp.hpp>
#include <boost/serialization/vector.hpp>
#include <boost/xpressive/xpressive.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/filesystem/operations.hpp>
#include <sstream>
#include <fstream>
#include <vector>
#include <string>
#include "convert/cp932.hxx"
#include "misc.hxx"


namespace dialektos {

namespace bookmark {


std::vector<Item> from_html(const std::string& html) {
  using namespace boost::xpressive;

  const sregex regex = as_xpr("<A HREF=")
    >> (s1=as_xpr("http://") >> +~set[as_xpr('>')|as_xpr(' ')])
    >> *~as_xpr('>') >> '>'
    >> (s2=+~as_xpr('<')) >> as_xpr("</A>") >> *_;
  const sregex regex_category = "<BR><BR><B>"
    >> (s1=+~as_xpr('<')) >> "</B><BR>" >> *_;

  std::vector<Item> list;
  std::stringstream ss;
  ss << html;
  std::string current_category;
  std::string line;
  while (std::getline(ss, line)) {
    boost::algorithm::trim_right(line);
    const std::string utf8 = convert::cp932(line);

    if (utf8.empty()) {
      current_category.clear();
      continue;
    }

    smatch what;

    if (regex_match(utf8, what, regex_category)) {
      current_category = what[1];
      continue;
    }

    if (regex_match(utf8, what, regex)) {
      Item item;
      item.uri = what[1];
      item.name = what[2];
      item.categories.push_back("bbsmenu");
      if (!current_category.empty())
        item.categories.push_back(current_category);
      list.push_back(item);
    }
  }

  return list;
}

std::vector<Item> from_xml(const boost::filesystem::path& xml) {
  std::vector<Item> items;

  if (!boost::filesystem::exists(xml) || !boost::filesystem::is_regular(xml))
    return items;

  std::ifstream ifs(xml.file_string().c_str());
  try {
    boost::archive::xml_iarchive ia(ifs);
    ia >> boost::serialization::make_nvp("Bookmarks", items);
  } catch (const boost::archive::archive_exception& e) {
    std::cerr << "bookmark::from_xml(): " << e.what() << " "
    << xml << std::endl;
  }
  return items;
}

void to_xml(const boost::filesystem::path& xml,
    const std::vector<Item>& items) {
  if (!misc::create_directories(xml.parent_path())) return;
  std::ofstream ofs(xml.file_string().c_str());
  try {
    boost::archive::xml_oarchive oa(ofs);
    oa << boost::serialization::make_nvp("Bookmarks", items);
  } catch (const boost::archive::archive_exception& e) {
    std::cerr << "bookmark::to_xml(): " << e.what() << " " << xml << std::endl;
  }
}


} // namespace bookmark

} // namespace dialektos
