/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef BOARD_WINDOW_HXX
#define BOARD_WINDOW_HXX

#include <glibmm/refptr.h>
#include <gtkmm/treeview.h>
#include <gtkmm/scrolledwindow.h>
#include <boost/scoped_ptr.hpp>
#include <boost/foreach.hpp>
#include <memory>
#include "application_framework.hxx"
#include "thread_list_model.hxx"
#include "http_get.hxx"
#include "board_subject_item.hxx"
#include "board_subject_idx.hxx"
#include "board_window_state.hxx"


namespace dialektos {

namespace bbs_detail {
  class Base;
}

struct ThreadIdx;

class ThreadWindow;

class BoardWindow: public dialektos::ApplicationFrameWork {
public:
  static void create(std::auto_ptr<bbs_detail::Base> bbs);
  virtual ~BoardWindow();
  virtual void save_state() const;
  void on_informed_from(const bbs_detail::Base&, const ThreadIdx&);
protected:
  BoardWindow(std::auto_ptr<bbs_detail::Base> bbs);

  void on_action_file_open();

  virtual void on_action_edit_copy();
  virtual void on_action_view_refresh();
  virtual void on_action_view_stop();
private:
  virtual bool is_same(const bbs_detail::Base& bbs) const;
  virtual std::string get_uri() const;

  void on_row_activated(const Gtk::TreeModel::Path&,
      Gtk::TreeViewColumn*);
  void load();

  void merge_subject_txt(
      const std::vector<SubjectItem>&,
      boost::unordered_map<std::string, ModelColumns>&);
  void unmerge_subject_txt(
      boost::unordered_map<std::string, ModelColumns>&);

  void on_http_get_end(bool);
  void on_refresh_end(const http::StatusLine&, const http::Header&);

  void save_content(const http::Response&);

  template <typename ColumnType>
  void on_column_clicked(ColumnType& column);

  template <typename ColumnType>
  void add_column(const BoardWindowState&);

  void inform_to_threads(
      boost::unordered_map<std::string, ModelColumns>&) const;
  void inform_to_thread(ThreadWindow&, int res_num) const;

  Gtk::TreeView tree_view_;
  boost::scoped_ptr<bbs_detail::Base> bbs_;
  Gtk::ScrolledWindow scrolled_;
  Glib::RefPtr<ThreadListModel> tree_model_;
  boost::scoped_ptr<http::GetInThread> http_getter_;
  SubjectIdx idx_;
};

template <typename ColumnType>
void BoardWindow::on_column_clicked(ColumnType& column) {
  typedef typename ColumnType::sort_model_column_type SortColumn;
  BOOST_FOREACH(Gtk::TreeViewColumn* kolumn, tree_view_.get_columns()){
    if (kolumn == &column) {
      if (kolumn->get_sort_indicator()) {
        if (kolumn->get_sort_order() == Gtk::SORT_ASCENDING)
          kolumn->set_sort_order(Gtk::SORT_DESCENDING);
        else kolumn->set_sort_order(Gtk::SORT_ASCENDING);
      }
      kolumn->set_sort_indicator(true);
    } else {
      if (kolumn->get_sort_indicator())
        kolumn->set_sort_order(Gtk::SORT_ASCENDING);
      kolumn->set_sort_indicator(false);
    }
  }
  tree_model_->set_sort_function<SortColumn>(
      column.get_sort_order() == Gtk::SORT_ASCENDING);
  tree_model_->sort();
}

template <typename ColumnType>
void BoardWindow::add_column(const BoardWindowState& state) {
  ColumnType* p = Gtk::manage(new ColumnType);
  tree_view_.append_column(*p);
  p->signal_clicked().connect(sigc::bind<ColumnType&>(
      sigc::mem_fun(*this, &BoardWindow::on_column_clicked<ColumnType>),
      *p));

  if (state.sort.column == p->get_id()) {
    typedef typename ColumnType::sort_model_column_type SortColumn;
    p->set_sort_indicator(true);
    p->set_sort_order(
        state.sort.ascendant ? Gtk::SORT_ASCENDING : Gtk::SORT_DESCENDING);
    tree_model_->set_sort_function<SortColumn>(state.sort.ascendant);
  }

}


} // namespace dialektos

#endif
