/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef BOARD_VIEW_COLUMN_HXX
#define BOARD_VIEW_COLUMN_HXX

#include <gtkmm/treeviewcolumn.h>
#include <glibmm/ustring.h>
#include <gtkmm/cellrenderer.h>
#include <boost/mpl/find.hpp>
#include "thread_list_model.hxx"


namespace dialektos {

namespace view_column {


class Base : public Gtk::TreeViewColumn {
public:
  Base(const Glib::ustring& title) : Gtk::TreeViewColumn(title) {
    set_min_width(15);
    set_reorderable(true);
    set_resizable(true);
    set_sizing(Gtk::TREE_VIEW_COLUMN_FIXED);
    set_clickable(true);
//    set_fixed_width(30); // specify in a derived class.
  }
  virtual ~Base() {}
  virtual std::string get_id() const = 0;
};

template <typename DerivedT>
class ViewColumnBase : public Base {
public:
  ViewColumnBase(const Glib::ustring& title) : Base(title) {
    Gtk::CellRendererText* cell = Gtk::manage(new Gtk::CellRendererText);
    pack_start(*cell, true);

    typedef typename DerivedT::render_cell_functor FunctorType;
    set_cell_data_func(*cell, FunctorType());
  }
  virtual ~ViewColumnBase() {}
};


namespace {
template <typename TypeHolderT, typename ValueType> struct DoRenderCell;

template <typename TypeHolderT>
struct DoRenderCell<TypeHolderT, std::string> :
  public std::binary_function<Gtk::CellRenderer*,
    Gtk::TreeModel::iterator, void> {
  void operator()(
      Gtk::CellRenderer* cell,
      const Gtk::TreeModel::iterator& iter) const {
    typename TypeHolderT::type value;
    iter->get_value(boost::mpl::find<model_column::List,
        TypeHolderT>::type::pos::value, value);

    cell->set_property("text", value);
  }
};
template <typename TypeHolderT>
struct DoRenderCell<TypeHolderT, int> :
  public std::binary_function<Gtk::CellRenderer*,
    Gtk::TreeModel::iterator, void> {
  void operator()(
      Gtk::CellRenderer* cell,
      const Gtk::TreeModel::iterator& iter) const {
    typename TypeHolderT::type value;
    iter->get_value(boost::mpl::find<model_column::List,
        TypeHolderT>::type::pos::value, value);
    if (value > 0) cell->set_property("text", value);
    else cell->set_property("text", Glib::ustring(""));
  }
};
template <typename TypeHolderT>
struct DoRenderCell<TypeHolderT, double> :
  public std::binary_function<Gtk::CellRenderer*,
    Gtk::TreeModel::iterator, void> {
  void operator()(
      Gtk::CellRenderer* cell,
      const Gtk::TreeModel::iterator& iter) const {
    typename TypeHolderT::type value;
    iter->get_value(boost::mpl::find<model_column::List,
        TypeHolderT>::type::pos::value, value);
    if (value > 0) cell->set_property("text", value);
    else cell->set_property("text", Glib::ustring(""));
  }
};

template <typename TypeHolderT>
struct RenderCell :
  public DoRenderCell<TypeHolderT, typename TypeHolderT::type> {};

} // anonymous namespace


struct RenderCellStartTime :
  public std::binary_function<Gtk::CellRenderer*,
    Gtk::TreeModel::iterator, void> {
  void operator()(
      Gtk::CellRenderer* cell,
      const Gtk::TreeModel::iterator& iter) const;
};


class StartTime: public ViewColumnBase<StartTime> {
public:
  typedef RenderCellStartTime render_cell_functor;
  typedef model_column::ID sort_model_column_type;
  StartTime() : ViewColumnBase<StartTime>("Since") {
    set_fixed_width(50);
  }
  virtual ~StartTime() {}
  virtual std::string get_id() const { return "start_time"; };
};

class Average: public ViewColumnBase<Average> {
public:
  typedef RenderCell<model_column::Average> render_cell_functor;
  typedef model_column::Average sort_model_column_type;
  Average() : ViewColumnBase<Average>("Average") {
    set_fixed_width(50);
  }
  virtual ~Average() {}
  virtual std::string get_id() const { return "average"; };
};


class Title: public ViewColumnBase<Title> {
public:
  typedef RenderCell<model_column::Title> render_cell_functor;
  typedef model_column::Title sort_model_column_type;
  Title() : ViewColumnBase<Title>("Title") {
    set_fixed_width(200);
  }
  virtual ~Title() {}
  virtual std::string get_id() const { return "title"; };
};


class ResNum: public ViewColumnBase<ResNum> {
public:
  typedef RenderCell<model_column::ResNum> render_cell_functor;
  typedef model_column::ResNum sort_model_column_type;
  ResNum() : ViewColumnBase<ResNum>("Res") {
    set_fixed_width(30);
  }
  virtual ~ResNum() {}
  virtual std::string get_id() const { return "res_num"; };
};


class Number: public ViewColumnBase<Number> {
public:
  typedef RenderCell<model_column::Number> render_cell_functor;
  typedef model_column::Number sort_model_column_type;
  Number() : ViewColumnBase<Number>("No.") {
    set_fixed_width(30);
  }
  virtual ~Number() {}
  virtual std::string get_id() const { return "number"; };
};

class LineCount: public ViewColumnBase<LineCount> {
public:
  typedef RenderCell<model_column::LineCount> render_cell_functor;
  typedef model_column::LineCount sort_model_column_type;
  LineCount() : ViewColumnBase<LineCount>("Read") {
    set_fixed_width(30);
  }
  virtual ~LineCount() {}
  virtual std::string get_id() const { return "line_count"; };
};


} // namespace view_column


} // namespace dialektos


#endif
