/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "bbs_detail_base.hxx"

#include <glibmm/convert.h>
#include <glibmm/error.h>
#include <boost/filesystem.hpp>
#include <boost/format.hpp>
#include <boost/xpressive/xpressive.hpp>
#include <boost/lambda/lambda.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/lexical_cast.hpp>
#include <fstream>
#include <vector>
#include <string>
#include <iostream>
#include <sstream>
#include <cstdlib>
#include "dat_parser.hxx"
#include "html_parser.hxx"
#include "buffer_builder.hxx"
#include "board_subject_item.hxx"
#include "http_header.hxx"
#include "convert/cp932.hxx"


namespace dialektos {

namespace bbs_detail {


const std::string Base::bbs_name = "other";
const std::string Base::encoding = "cp932";


Base::Base(const std::string& uri, const std::string& host,
    const std::string& board, const std::string& thread) :
      uri_(uri), host_(host), board_(board), thread_(thread) {
}

Base::Base(const Base& rhs) :
  uri_(rhs.uri_), host_(rhs.host_), board_(rhs.board_), thread_(rhs.thread_) {
}

Base* Base::clone() const {
  return do_clone();
}

Base* Base::do_clone() const {
  return new Base(*this);
}

Base::~Base() {}

bool Base::is_board() const {
  return thread_.empty();
}

bool Base::is_thread() const {
  return !thread_.empty();
}

std::string Base::get_board_dir_path() const {
  std::string homedir = std::getenv("HOME");
  boost::filesystem::path dir(homedir);
  dir = dir / ".dialektos" / "logs" / get_bbs_name() / host_ / board_;
  return dir.file_string();
}

std::string Base::get_thread_file_path() const {
  boost::filesystem::path dir(get_board_dir_path());
  std::string filename = thread_ + ".dat";
  dir = dir / "dat" / filename;
  return dir.file_string();
}

std::string Base::get_thread_idx_path() const {
  const boost::filesystem::path dir(get_board_idx_dir_path());
  const boost::filesystem::path xml =
    dir / thread_.substr(0, 3) / (thread_ + ".xml");
  return xml.file_string();
}

std::string Base::get_thread_state_path() const {
  const boost::filesystem::path dir(get_board_dir_path());
  const boost::filesystem::path xml = dir / "states" / (thread_ + ".xml");
  return xml.file_string();
}

std::string Base::get_thread_dat_uri() const {
  return "http://" + host_ + "/" + board_ + "/dat/" + thread_ + ".dat";
}

std::string Base::get_thread_uri() const {
  assert(!thread_.empty());
  std::stringstream ss;
  ss << boost::format("http://%1%/test/read.cgi/%2%/%3%/")
    % host_ % board_ % thread_ << std::flush;
  return ss.str();
}

std::string Base::get_another_thread_uri(const std::string& thread) const {
  return "http://" + host_ + "/test/read.cgi/" + board_ + "/" + thread + "/";
}

std::string Base::get_board_subject_path() const {
  boost::filesystem::path sbj(get_board_dir_path());
  sbj = sbj / "subject.txt";
  return sbj.file_string();
}

std::string Base::get_board_state_path() const {
  boost::filesystem::path xml(get_board_dir_path());
  xml = xml / "boardstate.xml";
  return xml.file_string();
}

std::string Base::get_board_subject_uri() const {
  std::stringstream ss;
  ss << boost::format("http://%1%/%2%/%3%")
    % host_ % board_ % "subject.txt" << std::flush;
  return ss.str();
}

std::string Base::get_board_uri() const {
  std::stringstream ss;
  ss << boost::format("http://%1%/%2%/") % host_ % board_ << std::flush;
  return ss.str();
}

std::string Base::get_board_subject_idx_path() const {
  boost::filesystem::path dir(get_board_dir_path());
  dir = dir / "subject.xml";
  return dir.file_string();
}

std::string Base::get_board_idx_dir_path() const {
  boost::filesystem::path dir(get_board_dir_path());
  dir = dir / "idx";
  return dir.file_string();
}

std::string Base::get_board_id() const {
  return board_;
}

const std::string& Base::get_encoding() const {
  return encoding;
}

const std::string& Base::get_bbs_name() const {
  return bbs_name;
}

void Base::load_thread_from_string(const std::string& dat,
    text_view::Layoutable& output) const {

  int resnum = output.get_res_num()+1;

  std::stringstream ss;
  ss << dat << std::flush;

  std::string line;
  Glib::IConv iconv("UTF-8", get_encoding());
  while (std::getline(ss, line)) {
    dialektos::BufferBuilder builder(output, resnum);
    try {
      dialektos::run_parser<
      dialektos::DatHtmlParserDriver,
      dialektos::HtmlParserFunctions>(
          builder,
          get_encoding() == "cp932" ?
              convert::cp932(line) : iconv.convert(line),
          resnum);
//      iconv.reset();
    } catch (const Glib::ConvertError& e) {
      builder.new_line(1);
      builder.add_res_num(resnum, false);
      builder.add_text(e.what(), true, std::string(""));
    }
    ++resnum;
  }

  output.set_res_num(resnum-1);
}

void Base::load_thread(text_view::Layoutable& output) const {
  std::string filepath = get_thread_file_path();

  std::ifstream ifs(filepath.c_str());
  std::stringstream ss;
  ss << ifs.rdbuf() << std::flush;
  load_thread_from_string(ss.str(), output);
}

void Base::load_subject(std::vector<SubjectItem>& output) const {
  std::ifstream ifs(get_board_subject_path().c_str());
  std::stringstream ss;
  ss << ifs.rdbuf();
  load_subject_from_string(ss.str(), output);
}

void Base::load_subject_from_string(const std::string& subject,
    std::vector<SubjectItem>& output) const {

  using namespace boost::xpressive;
  const sregex regex = (s1=repeat<9,10>(_d)) >> ".dat" >> *_s >> "<>"
  >> (s2=-+_) >> '(' >> *_s >> (s3=+_d) >> *_s >> ')';

  std::vector<SubjectItem> list;

  Glib::IConv iconv("UTF-8", get_encoding());
  size_t count = 1;

  std::stringstream ss;
  ss << subject;
  std::string line;
  while (std::getline(ss, line)) {
    boost::algorithm::trim_right(line);
    try {
      const std::string utf8 =
        get_encoding() == "cp932" ? convert::cp932(line) :
          iconv.convert(line);
      smatch what;
      if (regex_match(utf8, what, regex)) {
        using namespace boost::lambda;
        using boost::lambda::_1;
        const std::string id = what[1];
        std::string title = what[2];
        boost::algorithm::trim_if(title, _1 == ' ');
        SubjectItem item(count, id, title,
            boost::lexical_cast<int>(what[3]));
        list.push_back(item);
        ++count;
      }
    } catch (const Glib::ConvertError& e) {
      std::cerr << "convert error!! - " << e.what() << " " << line << std::endl;
    }
  }

  output = list;
}

http::Header Base::get_board_subject_request_header() const {
  http::Header request_header;
  request_header.set_host(host_);
  return request_header;
}

bool Base::operator==(const bbs_detail::Base& rhs) const {
  if (this == &rhs) return true;
  // same uri is not need.
  return host_ == rhs.host_ && board_ == rhs.board_ && thread_ == rhs.thread_;
}

std::string Base::get_title_from_string(const std::string& dat) const {
  using namespace boost::xpressive;
  const sregex regex = bos >> -repeat<4>(*(~_n) >> "<>")
    >> (s1=*(~_n)) >> _n >> -*_;

  smatch what;
  if (regex_match(dat, what, regex)) {
    try {
      return get_encoding() == "cp932" ? convert::cp932(what[1]) :
        Glib::convert(what[1], "UTF-8", get_encoding());
    } catch (const Glib::ConvertError& e) {
      std::cerr << "convert error!! - " << e.what() << " "
        << what[1] << std::endl;
    }
  }
  return "";
}

bool Base::belongs_to(const Base& rhs) const {
  assert(is_thread() && rhs.is_board());

  return get_bbs_name() == rhs.get_bbs_name() &&
  host_ == rhs.host_ && board_ == rhs.board_;

}

std::string Base::get_thread_id() const {
  return thread_;
}

http::Header Base::get_thread_dat_request_header() const {
  http::Header request_header;
  request_header.set_host(host_);

  if (boost::filesystem::exists(get_thread_file_path())) {
    const size_t file_size = boost::filesystem::file_size(get_thread_file_path());
    if (file_size > 0) request_header.set_range(file_size);
  }
  return request_header;
}

//Base* Base::judge(const std::string& uri) {
//}

} // namespace bbs_detail

} // namespace dialektos
