/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "application_window.hxx"

#include <gtkmm/main.h>
#include <boost/lambda/lambda.hpp>
#include <boost/foreach.hpp>
#include <boost/archive/xml_iarchive.hpp>
#include <boost/archive/xml_oarchive.hpp>
#include <boost/serialization/nvp.hpp>
#include <boost/serialization/vector.hpp>
#include <boost/filesystem.hpp>
#include <boost/bind.hpp>
#include <boost/ref.hpp>
#include <fstream>
#include <iostream>
#include <algorithm>
#include <cstdlib>
#include "uri_opener.hxx"
#include "misc.hxx"


namespace dialektos {

/* static data members */
boost::ptr_vector<ApplicationWindow> ApplicationWindow::windows;
history::Data ApplicationWindow::histories;

/* static member function */
ApplicationWindow* ApplicationWindow::find(const bbs_detail::Base& bbs) {
  boost::ptr_vector<ApplicationWindow>::iterator it =
      std::find_if(windows.begin(), windows.end(),
          boost::bind(&ApplicationWindow::is_same, _1, boost::ref(bbs)));
  return it != windows.end() ? &*it : 0;
}

void ApplicationWindow::load() {
  std::string homedir = std::getenv("HOME");
  boost::filesystem::path dir(homedir);
  boost::filesystem::path session = dir / ".dialektos" / "session.xml";

  histories.from_xml(dir / ".dialektos" / "history.xml");
  std::vector<std::string> uris;

  if (boost::filesystem::exists(session) &&
      boost::filesystem::is_regular(session)) {
    std::ifstream ifs(session.file_string().c_str());
    try {
      boost::archive::xml_iarchive ia(ifs);
      ia >> boost::serialization::make_nvp("Session", uris);
    } catch (const boost::archive::archive_exception& e) {
      std::cerr << "ApplicationWindow::load(): " << e.what() << std::endl;
    }
  }

  BOOST_FOREACH(const std::string& uri, uris) {
    uri_opener::open(uri);
  }

  if (uris.empty()) {
    uri_opener::open("http://dubai.2ch.net/morningcoffee/");
  }

  histories.set_writable();
}

void ApplicationWindow::save() {
  std::vector<std::string> uris;
  BOOST_FOREACH(const ApplicationWindow& window, windows) {
    uris.push_back(window.get_uri());
  }

  std::string homedir = std::getenv("HOME");
  boost::filesystem::path dir(homedir);
  boost::filesystem::path session = dir / ".dialektos" / "session.xml";

  if (!misc::create_directories(session.parent_path())) return;
  std::ofstream ofs(session.file_string().c_str());
  try {
    boost::archive::xml_oarchive oa(ofs);
    oa << boost::serialization::make_nvp("Session", uris);
  } catch (const boost::archive::archive_exception& e) {
    std::cerr << "save(): " << e.what() << std::endl;
  }

  histories.to_xml(dir / ".dialektos" / "history.xml");
}

void ApplicationWindow::close_windows(
    const std::vector<ApplicationWindow*>& wins) {
  BOOST_FOREACH(ApplicationWindow* window, wins) {
    using namespace boost::lambda;
    using boost::lambda::_1;
    window->save_state();
    windows.erase_if(&_1 == window);
  }

  if (!windows.empty()) save();
  if (windows.empty()) Gtk::Main::quit();
}

//bool ApplicationWindow::is_same(const bbs_detail::Base& bbs) const {
//  return false;
//}

ApplicationWindow::ApplicationWindow(): Gtk::Window() {}

ApplicationWindow::~ApplicationWindow() {}

void ApplicationWindow::regist(ApplicationWindow* window) {
  windows.push_back(window);
  save();
}

bool ApplicationWindow::on_delete_event(GdkEventAny* /*event*/) {
  using namespace boost::lambda;
  using boost::lambda::_1;
  save_state();
  windows.erase_if(&_1 == this);
  if (!windows.empty()) save();
  if (windows.empty()) Gtk::Main::quit();
  return false;
}


} // namespace dialektos
