/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "application_framework.hxx"

#include <glibmm/error.h>
#include <glibmm/ustring.h>
#include <gtkmm/widget.h>
#include <gtkmm/main.h>
#include <gtkmm/stock.h>
#include <gtkmm/menu.h>
#include <gtkmm/uimanager.h>
#include <gtkmm/actiongroup.h>
#include <boost/foreach.hpp>
#include <iostream>
#include "bookmark_window.hxx"
#include "uri_opener.hxx"
#include "bbs_detail_judge.hxx"
#include "bbs_detail_base.hxx"


namespace dialektos {


ApplicationFrameWork::ApplicationFrameWork() :
  ApplicationWindow(),
  vbox_(),
  statusbar_(),
  ui_manager_(Gtk::UIManager::create()),
  action_group_(Gtk::ActionGroup::create()),
  menubar_(),
  toolbar_(),
  popupmenu_(),
  history_menu_() {

  ApplicationWindow::add(vbox_);
  build_menubar();
  statusbar_.set_no_show_all(true);
}

ApplicationFrameWork::~ApplicationFrameWork() {}

void ApplicationFrameWork::add(Gtk::Widget& child) {
  vbox_.pack_start(*menubar_, Gtk::PACK_SHRINK);
  vbox_.pack_start(*toolbar_, Gtk::PACK_SHRINK);
  vbox_.pack_start(child, true, true);
  vbox_.pack_start(statusbar_, Gtk::PACK_SHRINK);
}
void ApplicationFrameWork::build_menubar() {
  // File menu
  action_group_->add(Gtk::Action::create("MenuFile", "_File"));
  action_group_->add(Gtk::Action::create("FileQuit", Gtk::Stock::QUIT),
      sigc::mem_fun(*this, &ApplicationFrameWork::on_action_file_quit));
  action_group_->add(Gtk::Action::create("FileClose", Gtk::Stock::CLOSE),
      sigc::mem_fun(*this, &ApplicationFrameWork::on_action_file_close));

  // Edit menu
  action_group_->add(Gtk::Action::create("MenuEdit", "_Edit"));
  action_group_->add(Gtk::Action::create("EditCopy", Gtk::Stock::COPY),
      sigc::mem_fun(*this, &ApplicationFrameWork::on_action_edit_copy));

  // View menu
  action_group_->add(Gtk::Action::create("MenuView", "_View"));
  action_group_->add(Gtk::Action::create("ViewRefresh", Gtk::Stock::REFRESH),
      sigc::mem_fun(*this, &ApplicationFrameWork::on_action_view_refresh));
  action_group_->add(Gtk::Action::create("ViewStop", Gtk::Stock::STOP),
      sigc::mem_fun(*this, &ApplicationFrameWork::on_action_view_stop));
  action_group_->add(Gtk::Action::create("ViewMenubar", "Toggle _Menubar"),
      sigc::mem_fun(*this, &ApplicationFrameWork::on_action_view_menubar));
  action_group_->add(Gtk::Action::create("ViewToolbar", "Toggle _Toolbar"),
      sigc::mem_fun(*this, &ApplicationFrameWork::on_action_view_toolbar));
  action_group_->add(Gtk::Action::create("ViewStatusbar", "Toggle _Statusbar"),
      sigc::mem_fun(*this, &ApplicationFrameWork::on_action_view_statusbar));

  action_group_->add(Gtk::Action::create("MenuHistory", "Hi_story"));

  // Bookmark menu
  action_group_->add(Gtk::Action::create("MenuBookmark", "_Bookmark"));
  action_group_->add(Gtk::Action::create("BookmarkShow", "_Show Bookmarks"),
      sigc::mem_fun(*this, &ApplicationFrameWork::on_action_bookmark_show));

  ui_manager_->insert_action_group(action_group_);
  add_accel_group(ui_manager_->get_accel_group());

  Glib::ustring ui_info =
    "<ui>"
    "  <menubar name='MenuBar'>"
    "    <menu action='MenuFile'>"
    "      <menuitem action='FileQuit'/>"
    "      <menuitem action='FileClose'/>"
    "    </menu>"
    "    <menu action='MenuEdit'>"
    "      <menuitem action='EditCopy'/>"
    "    </menu>"
    "    <menu action='MenuView'>"
    "      <menuitem action='ViewRefresh'/>"
    "      <menuitem action='ViewStop'/>"
    "      <separator/>"
    "      <menuitem action='ViewMenubar'/>"
    "      <menuitem action='ViewToolbar'/>"
    "      <menuitem action='ViewStatusbar'/>"
    "    </menu>"
    "    <menu action='MenuHistory'/>"
    "    <menu action='MenuBookmark'>"
    "      <menuitem action='BookmarkShow'/>"
    "    </menu>"
    "  </menubar>"
    "  <toolbar name='ToolBar'>"
    "    <toolitem action='ViewRefresh'/>"
    "    <toolitem action='ViewStop'/>"
    "  </toolbar>"
    "  <popup name='MenuPopup'>"
    "    <menuitem action='ViewRefresh'/>"
    "    <menuitem action='ViewStop'/>"
    "    <menuitem action='EditCopy'/>"
    "    <separator/>"
    "    <menuitem action='ViewMenubar'/>"
    "  </popup>"
    "</ui>";

  ui_manager_->add_ui_from_string(ui_info);

  menubar_ = ui_manager_->get_widget("/MenuBar");
  menubar_->set_no_show_all(true);
  menubar_->hide();
  toolbar_ = ui_manager_->get_widget("/ToolBar");
  toolbar_->set_no_show_all(true);
  toolbar_->hide();
  popupmenu_ =
    dynamic_cast<Gtk::Menu*>(ui_manager_->get_widget("/MenuPopup"));

  history_menu_ = dynamic_cast<Gtk::MenuItem*>(
      ui_manager_->get_widget("/MenuBar/MenuHistory"));
  history_menu_->show();
  history_menu_->get_submenu()->signal_show().connect(
      sigc::mem_fun(*this, &ApplicationFrameWork::set_history_menus));
}

void ApplicationFrameWork::on_action_file_quit() {
  save();
  BOOST_FOREACH(const ApplicationWindow& window, windows)
    window.save_state();
  windows.clear();
  Gtk::Main::quit();
}

void ApplicationFrameWork::on_action_file_close() {
  std::vector<ApplicationWindow*> closes;
  closes.push_back(this);
  close_windows(closes);
}

void ApplicationFrameWork::on_action_edit_copy() {
  std::cout << "edit copy activated" << std::endl;
}

void ApplicationFrameWork::on_action_view_refresh() {
  std::cout << "view refresh activated" << std::endl;
}

void ApplicationFrameWork::on_action_view_stop() {
  std::cout << "view refresh stop" << std::endl;
}

void ApplicationFrameWork::on_action_view_menubar() {
  if (menubar_->is_visible()) menubar_->hide(); else menubar_->show();
}

void ApplicationFrameWork::on_action_view_toolbar() {
  if (toolbar_->is_visible()) toolbar_->hide(); else toolbar_->show();
}

void ApplicationFrameWork::on_action_view_statusbar() {
  if (statusbar_.is_visible()) statusbar_.hide(); else statusbar_.show();
}

void ApplicationFrameWork::on_action_bookmark_show() {
  BookmarkWindow::create();
}

void ApplicationFrameWork::on_child_button_press(GdkEventButton* event) {
  if (event->type == GDK_BUTTON_PRESS && event->button == 3) {
    popupmenu_->popup(event->button, event->time);
  }
}

void ApplicationFrameWork::on_history_menu_item_activate(
    const std::string& uri) {
  uri_opener::open(uri);
}

void ApplicationFrameWork::on_history_menu_items_activate(
    const std::vector<std::string>& uris) const {
  BOOST_FOREACH(const std::string& uri, uris) uri_opener::open(uri);
}

void ApplicationFrameWork::on_history_menu_items_open_close_activate(
    const std::vector<std::string>& uris) const {

  std::vector<ApplicationWindow*> closes;

  BOOST_FOREACH(ApplicationWindow& window, windows)
    if (std::find(uris.begin(), uris.end(), window.get_uri()) == uris.end())
      closes.push_back(&window);

  on_history_menu_items_activate(uris);
  close_windows(closes);
}

void ApplicationFrameWork::set_history_menus() {
  Gtk::Menu* sub = history_menu_->get_submenu();
  sub->items().clear();
  std::vector<history::Item> recents = histories.get_recent();

  typedef std::map<std::string, Gtk::Menu*> SubMenuType;
  SubMenuType subs;

  typedef std::map<std::string, std::vector<std::string> > UrisType;
  UrisType uris;

  BOOST_FOREACH(const history::Item& hitem, recents) {
    std::auto_ptr<bbs_detail::Base> bbs = bbs_detail::judge(hitem.uri_);
    if (!bbs->is_thread()) continue;

    const std::string board = bbs->get_board_id();
    SubMenuType::iterator it = subs.find(board);
    Gtk::Menu* subsub;
    if (it != subs.end()) {
      subsub = it->second;
      if (subsub->items().size() == 10) continue;
    }
    else {
      Gtk::MenuItem* subitem = Gtk::manage(new Gtk::MenuItem(board));
      sub->append(*subitem);
      subitem->show();
      subsub = Gtk::manage(new Gtk::Menu());
      subitem->set_submenu(*subsub);
      subs.insert(std::make_pair(board, subsub));

      // menut item for threads list
      Gtk::MenuItem* menuitem =
          Gtk::manage(new Gtk::MenuItem(bbs->get_board_uri()));
      subsub->append(*menuitem);
      menuitem->show();
      menuitem->signal_activate().connect(sigc::bind(
          sigc::mem_fun(*this,
              &ApplicationFrameWork::on_history_menu_item_activate),
              bbs->get_board_uri()));
    }

    Glib::ustring label = hitem.name_;
    Gtk::MenuItem* item = Gtk::manage(new Gtk::MenuItem(
        label.size() > 40 ? label.substr(0, 40) + "..." : label));
    item->show();
    item->signal_activate().connect(sigc::bind(
        sigc::mem_fun(*this,
            &ApplicationFrameWork::on_history_menu_item_activate),
        hitem.uri_));
    subsub->append(*item);
    uris[board].push_back(hitem.uri_);
  }

  BOOST_FOREACH(SubMenuType::value_type pair, subs) {
    Gtk::MenuItem* item = Gtk::manage(new Gtk::MenuItem("Open All"));
    pair.second->append(*item);
    item->show();
    item->signal_activate().connect(sigc::bind(
        sigc::mem_fun(*this,
            &ApplicationFrameWork::on_history_menu_items_activate),
            uris[pair.first]));

    item = Gtk::manage(new Gtk::MenuItem("Open All & close others"));
    pair.second->append(*item);
    item->show();
    item->signal_activate().connect(sigc::bind(
        sigc::mem_fun(*this,
            &ApplicationFrameWork::on_history_menu_items_open_close_activate),
            uris[pair.first]));
  }
}


} // namespace dialektos
