﻿/*
 *	Fugashi Project
 *
 *	This software is distributed under a BSD-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Drawing.Drawing2D;

namespace Fugashi
{
    public partial class BitSwitch : UserControl
    {
        #region fields
        private bool state;
        private bool isExponential;
        private static Bitmap normal;
        private static Bitmap exponent;
        private static Bitmap hover;
        #endregion

        #region constructor
        public BitSwitch()
        {
            InitializeComponent();
            State = false;
            if (BorderColor.IsEmpty)
            {
                BorderColor = Color.FromArgb(0xD5, 0xDF, 0xE5);
            }
            UpdateBackgroundImage();
        }
        #endregion

        #region properties

        [Category("Behavior")]
        public event System.EventHandler StateChanged;


        [Category("Data")]
        [DefaultValue(typeof(bool), "True")]
        public bool State
        {
            get { return state; }
            set
            {
                SetStateWithoutEvent(value);
                if (StateChanged != null)
                {
                    StateChanged(this, new EventArgs());
                }
            }
        }

        [Category("Appearance")]
        [DefaultValue(typeof(Color), "LightSteelBlue")]
        public Color BorderColor
        {
            get;
            set;
        }

        [CategoryAttribute("Data")]
        [DefaultValue(typeof(bool), "False")]
        public bool IsExponential
        {
            get
            {
                return isExponential;
            }
            set
            {
                isExponential = value;
                UpdateBackgroundImage();
            }
        }

        #endregion

        #region event handlers
        private void label_Click(object sender, EventArgs e)
        {
            Focus();
            State = !state;
        }

        private void label_MouseLeave(object sender, EventArgs e)
        {
            BackgroundImage = IsExponential ? exponent : normal;
            Update();
        }

        private void label_MouseEnter(object sender, EventArgs e)
        {
            BackgroundImage = hover;
            Update();
        }

        private void label_Paint(object sender, PaintEventArgs e)
        {
            if (!Enabled)
            {
                e.Graphics.FillRectangle(
                    Brushes.LightGray, 0, 0, Width, Height);
            }
            using (Pen p = new Pen(BorderColor, 1))
            {
                int width = Tag != null ? Width - 1 : Width;
                e.Graphics.DrawRectangle(p, 0, 0, width, Height - 1);
            }
        }

        private void BitSwitch_SizeChanged(object sender, EventArgs e)
        {
            if (Width <= MinimumSize.Width || Height <= MinimumSize.Height)
            {
                return;
            }
            UpdateBackgroundImage();
        }

        #endregion

        #region private methods

        private void CreateNormalBackgroundImage()
        {
            if (normal is Bitmap && normal.Size == Size)
            {
                return;
            }
            normal = CreateBackGroundImage(Color.White, Color.Lavender,
                                            Color.Gainsboro, Color.Lavender);
        }

        private void CreateHoverBackgroundImage()
        {
            if (hover is Bitmap && hover.Size == Size)
            {
                return;
            }
            hover = CreateBackGroundImage(Color.LemonChiffon, Color.Gold,
                                            Color.Orange, Color.DarkOrange);
        }

        private void CreateExponentBackgroundImage()
        {
            if (exponent is Bitmap && exponent.Size == Size)
            {
                return;
            }
            exponent = CreateBackGroundImage(Color.LemonChiffon, Color.Bisque,
                                            Color.NavajoWhite, Color.Bisque);
        }

        private Bitmap CreateBackGroundImage(
                            Color hilight, Color light, Color face, Color dark)
        {
            if (Width < MinimumSize.Width || Height < MinimumSize.Height)
            {
                return null;
            }
            Bitmap bmp = new Bitmap(Width, Height);
            Graphics g = Graphics.FromImage(bmp);
            int border = (int)(Height * 0.48);
            Rectangle upper = new Rectangle(0, 0, Width, border);
            Rectangle lower = new Rectangle(0, border, Width, Height);
            LinearGradientBrush brushL =
                                new LinearGradientBrush(upper, hilight, light,
                                                LinearGradientMode.Vertical);
            LinearGradientBrush brushD =
                                    new LinearGradientBrush(lower, face, dark,
                                                LinearGradientMode.Vertical);
            Pen edge = new Pen(hilight, 1);
            g.FillRectangle(brushL, upper);
            g.FillRectangle(brushD, lower);
            g.DrawRectangle(edge, 1, 1, Width - 1, Height - 3);

            brushL.Dispose();
            brushD.Dispose();
            g.Dispose();
            return bmp;
        }

        public void SetStateWithoutEvent(bool state)
        {
            this.state = state;
            label.Text = state ? "1" : "0";
            label.ForeColor = state ? Color.DarkSlateGray : Color.SteelBlue;
            label.Font = new Font(label.Font,
                                state ? FontStyle.Bold : FontStyle.Regular);
        }

        private void UpdateBackgroundImage()
        {
            bool mouseOver = BackgroundImage == hover;
            CreateNormalBackgroundImage();
            CreateHoverBackgroundImage();
            CreateExponentBackgroundImage();
            if (!mouseOver)
            {
                BackgroundImage = IsExponential ? exponent : normal;
            }
            else
            {
                BackgroundImage = hover;
            }
        }

        #endregion
    }
}
