/*
 * 
 * XNua 0.1 Based Massively on Lua2IL 0.5
 * Lua2IL 0.5.0
 * LuaTypes.cs - Types for the XNua runtime
 * Copyright 2003-2005 Fabio Mascarenhas
 *                2007 Dean Calver
 * 
 */

using System;
using System.Collections;
using System.Diagnostics;
using System.Text;
using System.Reflection;

namespace XNua
{
    public struct LuaValue
    {
        public double N;
        public LuaReference O;

        public LuaValue(double n, LuaReference o)
        {
            this.N = n;
            this.O = o;
        }

        public LuaValue(double n)
            : this(n, null)
        {
        }

        public LuaValue(string s)
            : this(0, new LuaString(s))
        {
        }

        public LuaValue(LuaReference o)
            : this(0, o)
        {
        }

        public static implicit operator LuaValue(double n)
        {
            return new LuaValue(n);
        }
        public static implicit operator LuaValue(string s)
        {
            return new LuaValue(s);
        }
        public static implicit operator LuaValue(LuaReference o)
        {
            return new LuaValue(o);
        }

        public bool IsReference
        {
            get
            {
                return O != null;
            }
        }

        /// <summary>
        /// helper function when ur in C# to access a table value
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public LuaValue this[string index]
        {
            get
            {
                if (IsReference)
                {
                    return O[index];
                }
                else
                {
                    throw new RuntimeException("the value cannot be indexed");
                }
            }
            set
            {
                if (IsReference)
                {
                    O[index] = value;
                }
                else
                {
                    throw new RuntimeException("the value cannot be indexed");
                }
            }
        }

        public double ToNumber()
        {
            if (O == null)
                return N;
            else
                return O.ToNumber();
        }

        public string ToStr()
        {
            if (O == null)
                return N.ToString();
            else
                return O.ToString();
        }

        public bool ToBoolean()
        {
            if (O == null)
                return true;
            else
                return O.ToBoolean();
        }

        public LuaReference Metatable
        {
            get
            {
                if (O == null)
                    return NilClass.Instance;
                else
                    return O.Metatable;
            }
            set
            {
                throw new RuntimeException("value cannot have a metatable");
            }
        }

        public object CLRObject
        {
            get
            {
                if (O == null)
                    return N;
                else
                    return O.CLRObject;
            }
        }

        public bool Equals(LuaValue v)
        {
            if (v.O == null && O == null)
                return v.N == N;
            else if (v.O != null && O != null)
                return O.Equals(v.O);
            else
                return false;
        }

        public override bool Equals(object o)
        {
            if (o == null) return false;
            try
            {
                LuaValue v = (LuaValue)o;
                if (v.O == null && O == null)
                    return v.N == N;
                else if (v.O != null && O != null)
                    return O.Equals(v.O);
                else
                    return false;
            }
            catch (InvalidCastException)
            {
                return false;
            }
        }

        public bool LessThan(LuaValue v)
        {
            if (v.O == null && O == null)
                return N < v.N;
            else if (O == null)
                throw new RuntimeException("trying to compare incompatible values");
            else
                return O.LessThan(v.O);
        }

        public bool GreaterThan(LuaValue v)
        {
            if (v.O == null && O == null)
                return N > v.N;
            else if (O == null)
                throw new RuntimeException("trying to compare incompatible values");
            else
                return O.GreaterThan(v.O);
        }

        public override int GetHashCode()
        {
            if (O == null)
                return N.GetHashCode();
            else
                return O.GetHashCode();
        }

        public override string ToString()
        {
            if (O == null)
                return N.ToString();
            else
                return O.ToString();
        }

        public static bool operator ==(LuaValue v1, LuaValue v2) // raw equals
        {
            if (v1.O == null)
                return v2.O == null && v1.N == v2.N;
            else if (v1.O.Tag == LuaTypes.LUA_TSTRING && v2.O.Tag == LuaTypes.LUA_TSTRING)
                return ((LuaString)v1.O).s == ((LuaString)v2.O).s;
            else
                return v1.O == v2.O;
        }

        public static bool operator !=(LuaValue v1, LuaValue v2) // raw unequals
        {
            if (v1.O == null)
                return v2.O == null && v1.N != v2.N;
            else if (v1.O.Tag == LuaTypes.LUA_TSTRING && v2.O.Tag == LuaTypes.LUA_TSTRING)
                return ((LuaString)v1.O).s != ((LuaString)v2.O).s;
            else
                return v1.O != v2.O;
        }

        /// <summary>
        /// c api function 
        /// </summary>
        /// <param name="idx"></param>
        /// <returns></returns>
        public LuaTypes type
        {
            get
            {
                return IsReference ? O.Tag : LuaTypes.LUA_TNUMBER;
            }
        }
    }

    public abstract class LuaReference
    {
        public LuaTypes Tag;
        public uint Hash;

        public virtual object CLRObject
        {
            get
            {
                return this;
            }
        }

        public virtual LuaValue this[LuaState L, LuaValue index]
        {
            get
            {
                throw new RuntimeException("the value cannot be indexed");
            }
            set
            {
                throw new RuntimeException("the value cannot be indexed");
            }
        }

        public virtual LuaValue this[LuaValue index]
        {
            get
            {
                throw new RuntimeException("the value cannot be indexed");
            }
            set
            {
                throw new RuntimeException("the value cannot be indexed");
            }
        }

        /// <summary>
        /// helper function when ur in C# to access a table value
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public virtual LuaValue this[string index]
        {
            get
            {
                throw new RuntimeException("the value cannot be indexed");
            }
            set
            {
                throw new RuntimeException("the value cannot be indexed");
            }
        }

        public virtual bool ToBoolean()
        {
            return true;
        }

        public virtual double ToNumber()
        {
            return double.NaN;
        }

        public virtual string ToStr()
        {
            // not sure why we have ToStr as well as ToSTring but i've made
            // the base ToStr just call ToString so we get something...
            return this.ToString();
        }

        public virtual LuaReference Metatable
        {
            get
            {
                return NilClass.Instance;
            }
            set
            {
                throw new RuntimeException("value cannot have a metatable");
            }
        }

        public virtual bool LessThan(LuaReference o)
        {
            return false;
        }

        public virtual bool GreaterThan(LuaReference o)
        {
            return false;
        }

        public virtual bool Equals(LuaReference r)
        {
            return (this == r);
        }

        public virtual int Call(LuaState L, int nResults, int firstArg)
        {
            LuaReference callFunction;
            if (((Metatable != null) && (Metatable != NilClass.Instance)) && ((callFunction = Metatable[MetaMethods.Call].O) != null) &&
                (callFunction != NilClass.Instance))
            {
                for (int i = L.Stack.Top; i > L.Stack.Base + firstArg; i--)
                    L.Stack.Values[i] = L.Stack.Values[i - 1];
                L.Stack.Top++;
                return callFunction.Call(L, nResults, firstArg);
            }
            else
                throw new RuntimeException("trying to call a non-function");
        }

    }

    public class LuaString : LuaReference
    {
        public string s;

        public LuaString(string s)
        {
            this.s = s;
            Tag = LuaTypes.LUA_TSTRING;
            Hash = (uint)s.GetHashCode();
        }

        public override object CLRObject
        {
            get
            {
                return s;
            }
        }

        public override bool Equals(LuaReference r)
        {
            if ((r != null) && (r.Tag == LuaTypes.LUA_TSTRING))
            {
                return s == (string)((LuaString)r).CLRObject;
            }
            else return false;
        }

        public override bool Equals(object o)
        {
            try
            {
                return s == ((string)((LuaString)o).CLRObject);
            }
            catch (InvalidCastException)
            {
                return false;
            }
        }

        public override int GetHashCode()
        {
            return s.GetHashCode();
        }

        public override string ToString()
        {
            return s;
        }

        public override string ToStr()
        {
            return s;
        }

        public override double ToNumber()
        {
            try
            {
                return double.Parse(s);
            }
            catch
            {
                return double.NaN;
            }
        }

        public override bool LessThan(LuaReference r)
        {
            try
            {
                return s.CompareTo(((LuaString)r).CLRObject) < 0;
            }
            catch (InvalidCastException)
            {
                throw new RuntimeException("trying to compare incompatible values");
            }
        }

        public override bool GreaterThan(LuaReference r)
        {
            try
            {
                return s.CompareTo(((LuaString)r).CLRObject) > 0;
            }
            catch (InvalidCastException)
            {
                throw new RuntimeException("trying to compare incompatible values");
            }
        }
    }

    public class NilClass : LuaReference
    {
        public static NilClass Instance = new NilClass();

        private NilClass()
        {
            Tag = LuaTypes.LUA_TNIL;
            Hash = (uint)GetHashCode();
        }

        public override object CLRObject
        {
            get
            {
                return null;
            }
        }

        public override bool ToBoolean()
        {
            return false;
        }

        public override string ToString()
        {
            return "nil";
        }
    }

    public class TrueClass : LuaReference
    {
        public static TrueClass Instance = new TrueClass();

        private TrueClass()
        {
            Tag = LuaTypes.LUA_TBOOLEAN;
            Hash = (uint)GetHashCode();
        }

        public override object CLRObject
        {
            get
            {
                return true;
            }
        }

        public override string ToString()
        {
            return "true";
        }
    }

    public class FalseClass : LuaReference
    {
        public static FalseClass Instance = new FalseClass();

        private FalseClass()
        {
            Tag = LuaTypes.LUA_TBOOLEAN;
            Hash = (uint)GetHashCode();
        }

        public override object CLRObject
        {
            get
            {
                return false;
            }
        }

        public override bool ToBoolean()
        {
            return false;
        }

        public override string ToString()
        {
            return "false";
        }
    }
    public class WeakHashComparer : IEqualityComparer
    {
        public static readonly WeakHashComparer Instance = new WeakHashComparer();

        bool IEqualityComparer.Equals(object x, object y)
        {
            LuaValue vx = (LuaValue)x;
            if (y is WeakReference)
            {
                object target = ((WeakReference)y).Target;
                if (target != null)
                {
                    LuaValue vy = (LuaValue)target;
                    return vx.Equals(vy);
                }
                else return false;
            }
            else
            {
                return x.Equals(y);
            }
        }

        int IEqualityComparer.GetHashCode(object obj)
        {
            if (obj is WeakReference)
            {
                object target = ((WeakReference)obj).Target;
                if (target != null)
                    return target.GetHashCode();
                else
                    return 0;
            }
            else return obj.GetHashCode();
        }

    }

    public class LuaHashComparer : IEqualityComparer
    {
        public static readonly LuaHashComparer Instance = new LuaHashComparer();

        bool IEqualityComparer.Equals(object x, object y)
        {
            LuaValue vx = (LuaValue)x;
            LuaValue vy = (LuaValue)y;
            if (vx.O == null)
                return (((vy.O == null) && (vx.N == vy.N)));
            else if (vy.O != null)
            {
                LuaReference r1 = (LuaReference)vx.O;
                LuaReference r2 = (LuaReference)vy.O;
                if (r1.Tag == r2.Tag)
                {
                    if (r1.Tag == LuaTypes.LUA_TSTRING)
                        return (((LuaString)r1).s == ((LuaString)r2).s);
                    else if (r1.Tag == LuaTypes.LUA_TTABLE || r1.Tag == LuaTypes.LUA_TUSERDATA)
                        return r1.Equals(r2);
                    else return (r1 == r2);
                }
                else return false;
            }
            else return false;
        }
        int IEqualityComparer.GetHashCode(object obj)
        {
            LuaValue v = (LuaValue)obj;
            return v.GetHashCode();
        }
    }

    public struct MetaMethods
    {
        public static readonly LuaValue Mode = new LuaValue(new LuaString("__mode"));
        public static readonly LuaValue Add = new LuaValue(new LuaString("__add"));
        public static readonly LuaValue Sub = new LuaValue(new LuaString("__sub"));
        public static readonly LuaValue Mul = new LuaValue(new LuaString("__mul"));
        public static readonly LuaValue Div = new LuaValue(new LuaString("__div"));
        public static readonly LuaValue Pow = new LuaValue(new LuaString("__pow"));
        public static readonly LuaValue Unm = new LuaValue(new LuaString("__unm"));
        public static readonly LuaValue Concat = new LuaValue(new LuaString("__concat"));
        public static readonly LuaValue Eq = new LuaValue(new LuaString("__eq"));
        public static readonly LuaValue Lt = new LuaValue(new LuaString("__lt"));
        public static readonly LuaValue Le = new LuaValue(new LuaString("__le"));
        public static readonly LuaValue Index = new LuaValue(new LuaString("__index"));
        public static readonly LuaValue NewIndex = new LuaValue(new LuaString("__newindex"));
        public static readonly LuaValue Call = new LuaValue(new LuaString("__call"));
        public static readonly LuaValue GC = new LuaValue(new LuaString("__gc"));
        public static readonly LuaValue Tostring = new LuaValue(new LuaString("__tostring"));
    }

    public class LuaOldTable : LuaReference
    {
        bool weakKeys;
        bool weakValues;
        LuaReference meta;

        public Hashtable hash;

        public LuaOldTable()
        {
            weakKeys = false;
            weakValues = false;
            meta = NilClass.Instance;
            hash = new Hashtable(LuaHashComparer.Instance);
            Tag = LuaTypes.LUA_TTABLE;
            Hash = (uint)GetHashCode();
        }

        ~LuaOldTable()
        {
            LuaReference gcFunction;
            if (meta != null && meta != NilClass.Instance && ((gcFunction = meta[MetaMethods.GC].O) != null) &&
                (gcFunction != NilClass.Instance))
            {
                LuaState L = new LuaState();
                L.Stack.Values[L.Stack.Top++].O = this;
                gcFunction.Call(L, 0, 0);
            }
        }

        void SetMode(bool weakKeys, bool weakValues)
        {
            Hashtable oldHash = hash;
            bool oldWeakKeys = this.weakKeys;
            bool oldWeakValues = this.weakValues;
            this.weakKeys = weakKeys;
            this.weakValues = weakValues;
            if (weakKeys)
                hash = new Hashtable(WeakHashComparer.Instance);
            else
                hash = new Hashtable(LuaHashComparer.Instance);
            IDictionaryEnumerator entries = oldHash.GetEnumerator();
            while (entries.MoveNext())
            {
                DictionaryEntry entry = entries.Entry;
                if (oldWeakKeys && oldWeakValues)
                    this[(LuaValue)((WeakReference)entry.Key).Target] = (LuaValue)((WeakReference)entry.Value).Target;
                else if (oldWeakKeys)
                    this[(LuaValue)((WeakReference)entry.Key).Target] = (LuaValue)entry.Value;
                else if (oldWeakValues)
                    this[(LuaValue)entry.Key] = (LuaValue)((WeakReference)entry.Value).Target;
                else
                    this[(LuaValue)entry.Key] = (LuaValue)entry.Value;
            }
        }

        public override LuaReference Metatable
        {
            get
            {
                return meta;
            }
            set
            {
                meta = value;
                string mode = (string)meta[MetaMethods.Mode].CLRObject;
                if (mode != null)
                {
                    bool weakk = (mode.IndexOf('k') != -1);
                    bool weakv = (mode.IndexOf('k') != -1);
                    SetMode(weakk, weakv);
                }
            }
        }

        public override object CLRObject
        {
            get
            {
                return hash;
            }
        }

        public override LuaValue this[LuaState L, LuaValue index]
        {
            get
            {
                LuaValue v = this[index];
                if (v.O == NilClass.Instance && meta != NilClass.Instance)
                {
                    LuaReference metaIndex = meta[MetaMethods.Index].O;
                    if (metaIndex != NilClass.Instance && metaIndex != null)
                    {
                        if (metaIndex.Tag == LuaTypes.LUA_TFUNCTION)
                        {
                            L.Stack.Values[L.Stack.Top].O = metaIndex;
                            L.Stack.Values[L.Stack.Top + 1].O = this;
                            L.Stack.Values[L.Stack.Top + 2] = index;
                            L.Stack.Check(3);
                            L.Stack.Top += 3;
                            metaIndex.Call(L, 1, L.Stack.Top - 2);
                            L.Stack.Top--;
                            v = L.Stack.Values[L.Stack.Top];
                        }
                        else v = metaIndex[L, index];
                    }
                    else v.O = NilClass.Instance;
                }
                return v;
            }
            set
            {
                if (meta == NilClass.Instance)
                {
                    this[index] = value;
                }
                else
                {
                    LuaValue v = this[index];
                    if (v.O != NilClass.Instance)
                        this[index] = value;
                    else
                    {
                        LuaReference metaNewIndex = meta[MetaMethods.NewIndex].O;
                        if (metaNewIndex == NilClass.Instance || metaNewIndex == null)
                            this[index] = value;
                        else if (metaNewIndex.Tag == LuaTypes.LUA_TFUNCTION)
                        {
                            L.Stack.Values[L.Stack.Top].O = metaNewIndex;
                            L.Stack.Values[L.Stack.Top + 1].O = this;
                            L.Stack.Values[L.Stack.Top + 2] = index;
                            L.Stack.Values[L.Stack.Top + 3] = value;
                            L.Stack.Check(4);
                            L.Stack.Top += 4;
                            metaNewIndex.Call(L, 1, L.Stack.Top - 3);
                        }
                        else metaNewIndex[L, index] = value;
                    }
                }
            }
        }

        public override LuaValue this[LuaValue index]
        {
            get
            {
                object o = hash[index];
                if (o == null)
                    return new LuaValue(NilClass.Instance);
                if (weakValues)
                {
                    o = ((WeakReference)o).Target;
                    if (o != null)
                        return (LuaValue)o;
                    else
                    {
                        hash.Remove(index);
                        return new LuaValue(NilClass.Instance);
                    }
                }
                else
                    return (LuaValue)o;
            }
            set
            {
                if (weakKeys && weakValues)
                    hash[new WeakReference(index)] = new WeakReference(value);
                else if (weakKeys)
                    hash[new WeakReference(index)] = value;
                else if (weakValues)
                    hash[index] = new WeakReference(value);
                else
                    hash[index] = value;
            }
        }

        public override string ToString()
        {
            return "table: <" + hash.ToString() + ">";
        }

        public IDictionaryEnumerator GetEnumerator()
        {
            return hash.GetEnumerator();
        }
    }

    public abstract class LuaFunction : LuaReference
    {
        LuaState standalone;
        public LuaReference globals;

        public LuaFunction(LuaReference globals)
        {
            this.globals = globals;
            Tag = LuaTypes.LUA_TFUNCTION;
            Hash = (uint)GetHashCode();
        }

        /// <summary>
        /// bit rubbish but make it easier to kick things off on .Net compact dynamic thing-me-bob
        /// </summary>
        public LuaFunction()
        {
            this.globals = null;
            Tag = LuaTypes.LUA_TFUNCTION;
            Hash = (uint)GetHashCode();
        }

        public override int Call(LuaState L, int nResults, int firstArg)
        {
            CallInfo callInfo = new CallInfo();
            PreCall(L, firstArg, ref callInfo);
            int results;
            Execute(L, out results);
            int firstRes = L.Stack.Top - results;
            PosCall(L, nResults, firstRes, callInfo.LastBase);
            //if(nResults>0) L.Stack.Top=callInfo.MaxTop;
            return results;
        }

        public int Call(object[] argv, object[] results)
        {
            return Call(argv, null, results);
        }

        public int Call(object[] argv, Type[] returnTypes, object[] results)
        {
            lock (this)
            {
                if (standalone == null) standalone = new LuaState();
            }

            return Call(standalone, argv, returnTypes, results);
        }

        public int Call(LuaValue[] argv, object[] result)
        {
            lock (this)
            {
                if (standalone == null) standalone = new LuaState();
            }
            return Call(standalone, argv, result);
        }
        public int Call(LuaState L, object[] argv, object[] result)
        {
            return Call(L, argv, null, result);
        }

        public int Call(LuaState L, object[] argv, Type[] returnTypes, object[] results)
        {
            L.Stack.Check(argv.Length);
            for (int i = 0; i < argv.Length; i++)
                TypeChecker.GetToLuaConv(argv[i].GetType())(L, i, argv[i]);
            L.Stack.Top = argv.Length;
            int nResults = this.Call(L, -1, 0);
            //object[] results = new object[nResults];
            for (int i = 0; i < results.Length && i < nResults; i++)
                if (returnTypes != null)
                    results[i] = TypeChecker.GetToCLRConv(returnTypes[i])(ref L.Stack.Values[L.Stack.Top - nResults + i]);
                else
                    results[i] = L.Stack.Values[L.Stack.Top - nResults + i].CLRObject;
            return nResults;
        }

        public int Call(LuaState L, LuaValue[] argv, object[] results)
        {
            L.Stack.Check(argv.Length);
            for (int i = 0; i < argv.Length; i++)
                L.Stack.Values[i] = argv[i];
            L.Stack.Top = argv.Length;
            int nResults = this.Call(L, -1, 0);
            //object[] results = new object[nResults];
            for (int i = 0; i < results.Length; i++)
                results[i] = L.Stack.Values[L.Stack.Top - nResults + i].CLRObject;
            return nResults;
        }

        public void PreCall(LuaState L, int firstArg, ref CallInfo callInfo)
        {
            callInfo.LastBase = L.Stack.Base;
            L.Stack.Check(LuaState.MIN_STACK);
            L.Stack.Base = L.Stack.Base + firstArg;
            callInfo.MaxTop = L.Stack.Top + LuaState.MIN_STACK;
        }

        public void PosCall(LuaState L, int wanted, int firstResult, int lastBase)
        {
            int res = L.Stack.Base - 1;
            if (res < 0)
                res = 0;
            L.Stack.Base = lastBase;
            while (wanted != 0 && firstResult < L.Stack.Top)
            {
                L.Stack.Values[res] = L.Stack.Values[firstResult];
                res++;
                firstResult++;
                wanted--;
            }
            while (wanted > 0)
            {
                L.Stack.Values[res].O = NilClass.Instance;
                wanted--;
                res++;
            }
            L.Stack.Top = res;
        }
        //FI[o[[h֐ĂяoɗO𓊂鏈ԓIɍsĂ
        //      sfalseԂ`ɐ؂ւBs͗O𓊂Ƃ
        public abstract bool Execute(LuaState L, out int result);

        public int PCall(LuaState L, int nResults, int firstArg)
        {
            try
            {
                return Call(L, nResults, firstArg);
            }
            catch (RuntimeException)
            {
                return -1;
            }
        }

        public override string ToString()
        {
            return "function: " + GetType().ToString();
        }
    }

    public abstract class LuaClosure : LuaFunction
    {
        public UpValue[] UpVals;
        public CompiledPrototype proto;

        public LuaClosure(LuaReference globals)
            : base(globals)
        {
            proto = null;
        }

        public override bool Execute(LuaState L, out int result)
        {
            throw new Exception("not supported");
        }

        //public abstract int Execute(LuaState L,ref CallInfo callInfo);

        public new void PreCall(LuaState L, int firstArg, ref CallInfo callInfo)
        {
            callInfo.LastBase = L.Stack.Base;
            if (proto.IsVararg)
                AdjustVarargs(L, firstArg);
            L.Stack.Check(proto.MaxStack);
            L.Stack.Base = L.Stack.Base + firstArg;
            callInfo.MaxTop = L.Stack.Base + proto.MaxStack;
            while (L.Stack.Top < callInfo.MaxTop)
            {
                L.Stack.Values[L.Stack.Top].O = NilClass.Instance;
                L.Stack.Top++;
            }
        }

        public void AdjustVarargs(LuaState L, int stkBase)
        {
            int actual = L.Stack.Top - L.Stack.Base;
            if (actual < proto.NParams)
            {
                L.Stack.Check(proto.NParams - actual);
                for (; actual < proto.NParams; ++actual)
                {
                    L.Stack.Values[L.Stack.Top].O = NilClass.Instance;
                    L.Stack.Top++;
                }
            }
            actual -= proto.NParams;
            LuaTable tab = new LuaTable();
            for (int i = 0; i < actual; i++)
                tab[L, new LuaValue(i + 1)] = L.Stack.Values[L.Stack.Top - actual + 1];
            tab[L, new LuaValue("n")] = new LuaValue(actual);
            L.Stack.Top -= actual;
            L.Stack.Values[L.Stack.Top].O = tab;
            L.Stack.Check(1);
            L.Stack.Top++;
        }
        public void OpMove(LuaState L, int RA, int RB)
        {
            L.Stack.Values[L.Stack.Base + RA] = L.Stack.Values[L.Stack.Base + RB];
        }

        public void OpLoadK(LuaState L, int RA, int RBx)
        {
            L.Stack.Values[L.Stack.Base + RA] = proto.Constants[RBx];
        }

        public void OpLoadBool(LuaState L, int RA, int RB)
        {
            if (RB != 0)
                L.Stack.Values[L.Stack.Base + RA].O = TrueClass.Instance;
            else
                L.Stack.Values[L.Stack.Base + RA].O = FalseClass.Instance;
        }

        public void OpLoadNil(LuaState L, int RA, int RB)
        {
            for (int i = RA; i <= RB; i++)
                L.Stack.Values[L.Stack.Base + i].O = NilClass.Instance;
        }

        public void OpGetUpVal(LuaState L, int RA, int RB)
        {
            L.Stack.Values[L.Stack.Base + RA] = UpVals[RB].Value;
        }

        public void OpGetGlobal(LuaState L, int RA, int RBx)
        {
            L.Stack.Values[L.Stack.Base + RA] = globals[L, proto.Constants[RBx]];
        }

        public void OpGetTable(LuaState L, int RA, int RB, int RC)
        {
            LuaValue[] stack = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RC : RC - LuaState.MAX_LOCAL_STACK);
            L.Stack.Values[L.Stack.Base + RA] = L.Stack.Values[L.Stack.Base + RB].O[L, stack[idx]];
        }

        public void OpSetGlobal(LuaState L, int RA, int RBx)
        {
            globals[L, proto.Constants[RBx]] = L.Stack.Values[L.Stack.Base + RA];
        }

        public void OpSetUpVal(LuaState L, int RA, int RB)
        {
            UpVals[RB].Value = L.Stack.Values[L.Stack.Base + RA];
        }

        public void OpSetTable(LuaState L, int RA, int RB, int RC)
        {
            LuaValue[] stack1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RB : RB - LuaState.MAX_LOCAL_STACK);
            LuaValue[] stack2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RC : RC - LuaState.MAX_LOCAL_STACK);
            L.Stack.Values[L.Stack.Base + RA].O[L, stack1[idx1]] = stack2[idx2];
        }

        public void OpNewTable(LuaState L, int RA, int RB, int RC)
        {
            L.Stack.Values[L.Stack.Base + RA].O = new LuaTable();
        }

        public void OpSelf(LuaState L, int RA, int RB, int RC)
        {
            L.Stack.Values[L.Stack.Base + RA + 1] = L.Stack.Values[L.Stack.Base + RB];
            OpGetTable(L, RA, RB, RC);
        }

        public void Arith(LuaState L, int RA, LuaValue[] stack1, int idx1, LuaValue[] stack2, int idx2, LuaOpCode op)
        {
            double b, c;
            if (((b = stack1[idx1].ToNumber()) != double.NaN) &&
                ((c = stack2[idx2].ToNumber()) != double.NaN))
            {
                switch (op)
                {
                    case LuaOpCode.OP_ADD:
                        {
                            L.Stack.Values[L.Stack.Base + RA].N = b + c;
                            L.Stack.Values[L.Stack.Base + RA].O = null;
                            break;
                        }
                    case LuaOpCode.OP_SUB:
                        {
                            L.Stack.Values[L.Stack.Base + RA].N = b - c;
                            L.Stack.Values[L.Stack.Base + RA].O = null;
                            break;
                        }
                    case LuaOpCode.OP_MUL:
                        {
                            L.Stack.Values[L.Stack.Base + RA].N = b * c;
                            L.Stack.Values[L.Stack.Base + RA].O = null;
                            break;
                        }
                    case LuaOpCode.OP_DIV:
                        {
                            L.Stack.Values[L.Stack.Base + RA].N = b / c;
                            L.Stack.Values[L.Stack.Base + RA].O = null;
                            break;
                        }
                    case LuaOpCode.OP_POW:
                        {
                            L.Stack.Values[L.Stack.Base + RA].N = Math.Pow(b, c);
                            L.Stack.Values[L.Stack.Base + RA].O = null;
                            break;
                        }
                    default:
                        throw new RuntimeException("Invalid operation to Arith");
                }
            }
            else
            {
                CallMeta(L, stack1, idx1, stack2, idx2, op);
                L.Stack.Values[L.Stack.Base + RA] = L.Stack.Values[L.Stack.Top];
            }
        }

        public LuaReference GetMetaMethod(LuaReference meta, LuaOpCode op)
        {
            if (meta != NilClass.Instance)
            {
                switch (op)
                {
                    case LuaOpCode.OP_ADD:
                        return meta[MetaMethods.Add].O;
                    case LuaOpCode.OP_SUB:
                        return meta[MetaMethods.Sub].O;
                    case LuaOpCode.OP_MUL:
                        return meta[MetaMethods.Mul].O;
                    case LuaOpCode.OP_DIV:
                        return meta[MetaMethods.Div].O;
                    case LuaOpCode.OP_POW:
                        return meta[MetaMethods.Pow].O;
                    case LuaOpCode.OP_UNM:
                        return meta[MetaMethods.Unm].O;
                    case LuaOpCode.OP_CONCAT:
                        return meta[MetaMethods.Concat].O;
                    case LuaOpCode.OP_EQ:
                        return meta[MetaMethods.Eq].O;
                    case LuaOpCode.OP_LE:
                        return meta[MetaMethods.Le].O;
                    case LuaOpCode.OP_LT:
                        return meta[MetaMethods.Lt].O;
                    default:
                        return NilClass.Instance;
                }
            }
            else return NilClass.Instance;
        }

        public void CallMeta(LuaState L, LuaValue[] stack1, int idx1, LuaValue[] stack2, int idx2, LuaOpCode op)
        {
            LuaReference meta = GetMetaMethod(stack1[idx1].Metatable, op);
            if (meta == NilClass.Instance || meta == null)
                meta = GetMetaMethod(stack2[idx2].Metatable, op);
            L.Stack.Values[L.Stack.Top].O = meta;
            L.Stack.Values[L.Stack.Top + 1] = stack1[idx1];
            L.Stack.Values[L.Stack.Top + 2] = stack2[idx2];
            L.Stack.Check(3);
            L.Stack.Top += 3;
            meta.Call(L, 1, L.Stack.Top - 2);
            L.Stack.Top--;
        }

        public bool CallCompMeta(LuaState L, LuaReference o1, LuaReference o2, LuaOpCode op)
        {
            LuaReference meta1 = o1.Metatable;
            if (meta1 == NilClass.Instance || meta1 == null)
                return false;
            LuaReference meta2 = o2.Metatable;
            if (meta2 == NilClass.Instance || meta2 == null)
                return false;
            LuaReference metaFunc1 = GetMetaMethod(meta1, op);
            if (metaFunc1 == NilClass.Instance || metaFunc1 == null)
                return false;
            if (!meta1.Equals(meta2))
            {
                LuaReference metaFunc2 = GetMetaMethod(meta2, op);
                if (!metaFunc1.Equals(metaFunc2)) return false;
            }
            L.Stack.Values[L.Stack.Top].O = metaFunc1;
            L.Stack.Values[L.Stack.Top + 1].O = o1;
            L.Stack.Values[L.Stack.Top + 2].O = o2;
            L.Stack.Check(3);
            L.Stack.Top += 3;
            metaFunc1.Call(L, 1, L.Stack.Top - 2);
            L.Stack.Top--;
            return L.Stack.Values[L.Stack.Top].ToBoolean();
        }


        public void OpAdd(LuaState L, int RA, int RB, int RC)
        {
            LuaValue[] stack1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RB : RB - LuaState.MAX_LOCAL_STACK);
            LuaValue[] stack2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RC : RC - LuaState.MAX_LOCAL_STACK);
            if (stack1[idx1].O == null && stack2[idx2].O == null)
            {
                L.Stack.Values[L.Stack.Base + RA].N = stack1[idx1].N + stack2[idx2].N;
                L.Stack.Values[L.Stack.Base + RA].O = null;
            }
            else
                Arith(L, RA, stack1, idx1, stack2, idx2, LuaOpCode.OP_ADD);
        }

        public void OpSub(LuaState L, int RA, int RB, int RC)
        {
            LuaValue[] stack1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RB : RB - LuaState.MAX_LOCAL_STACK);
            LuaValue[] stack2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RC : RC - LuaState.MAX_LOCAL_STACK);
            if (stack1[idx1].O == null && stack2[idx2].O == null)
            {
                L.Stack.Values[L.Stack.Base + RA].N = stack1[idx1].N - stack2[idx2].N;
                L.Stack.Values[L.Stack.Base + RA].O = null;
            }
            else
                Arith(L, RA, stack1, idx1, stack2, idx2, LuaOpCode.OP_SUB);
        }

        public void OpMul(LuaState L, int RA, int RB, int RC)
        {
            LuaValue[] stack1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RB : RB - LuaState.MAX_LOCAL_STACK);
            LuaValue[] stack2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RC : RC - LuaState.MAX_LOCAL_STACK);
            if (stack1[idx1].O == null && stack2[idx2].O == null)
            {
                L.Stack.Values[L.Stack.Base + RA].N = stack1[idx1].N * stack2[idx2].N;
                L.Stack.Values[L.Stack.Base + RA].O = null;
            }
            else
                Arith(L, RA, stack1, idx1, stack2, idx2, LuaOpCode.OP_MUL);
        }

        public void OpDiv(LuaState L, int RA, int RB, int RC)
        {
            LuaValue[] stack1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RB : RB - LuaState.MAX_LOCAL_STACK);
            LuaValue[] stack2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RC : RC - LuaState.MAX_LOCAL_STACK);
            if (stack1[idx1].O == null && stack2[idx2].O == null)
            {
                L.Stack.Values[L.Stack.Base + RA].N = stack1[idx1].N / stack2[idx2].N;
                L.Stack.Values[L.Stack.Base + RA].O = null;
            }
            else
                Arith(L, RA, stack1, idx1, stack2, idx2, LuaOpCode.OP_DIV);
        }

        public void OpPow(LuaState L, int RA, int RB, int RC)
        {
            LuaValue[] stack1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RB : RB - LuaState.MAX_LOCAL_STACK);
            LuaValue[] stack2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RC : RC - LuaState.MAX_LOCAL_STACK);
            if (stack1[idx1].O == null && stack2[idx2].O == null)
            {
                L.Stack.Values[L.Stack.Base + RA].N = Math.Pow(stack1[idx1].N, stack2[idx2].N);
                L.Stack.Values[L.Stack.Base + RA].O = null;
            }
            else
                Arith(L, RA, stack1, idx1, stack2, idx2, LuaOpCode.OP_POW);
        }

        public void OpUnm(LuaState L, int RA, int RB)
        {
            LuaValue[] stack1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RB : RB - LuaState.MAX_LOCAL_STACK);
            if (stack1[idx1].ToNumber() != double.NaN)
            {
                L.Stack.Values[L.Stack.Base + RA].N = -stack1[idx1].N;
                L.Stack.Values[L.Stack.Base + RA].O = null;
            }
            else
            {
                CallMeta(L, stack1, idx1, LuaStack.NilStack, 0, LuaOpCode.OP_UNM);
                L.Stack.Values[L.Stack.Base + RA] = L.Stack.Values[L.Stack.Top];
                //L.Stack.Values[L.Stack.Base+RA].O=stack1[idx1].O.Minus();
            }
        }

        public void OpNot(LuaState L, int RA, int RB)
        {
            if (L.Stack.Values[L.Stack.Base + RB].O == NilClass.Instance ||
                L.Stack.Values[L.Stack.Base + RB].O == FalseClass.Instance)
                L.Stack.Values[L.Stack.Base + RA].O = TrueClass.Instance;
            else
                L.Stack.Values[L.Stack.Base + RA].O = FalseClass.Instance;
        }

        public void OpConcat(LuaState L, int RA, int RB, int RC)
        {
            int total = RC - RB + 1;
            int last = RC;
            do
            {
                int top = L.Stack.Base + last + 1;
                int n = 2;
                if (L.Stack.Values[top - 2].ToStr() == null ||
                    L.Stack.Values[top - 1].ToStr() == null)
                {
                    CallMeta(L, L.Stack.Values, top - 2, L.Stack.Values, top - 1, LuaOpCode.OP_CONCAT);
                    L.Stack.Values[top - 2].O = L.Stack.Values[L.Stack.Top].O;
                }
                else
                {
                    string s;
                    int size = 0;
                    while (n < total && (s = L.Stack.Values[top - n - 1].ToStr()) != null)
                    {
                        size += s.Length;
                        n++;
                    }
                    StringBuilder sb = new StringBuilder(size);
                    for (int i = n; i > 0; i--) sb.Append(L.Stack.Values[top - i].ToStr());
                    L.Stack.Values[top - n].O = new LuaString(sb.ToString());
                }
                total -= n - 1;
                last -= n - 1;
            } while (total > 1);
            L.Stack.Values[L.Stack.Base + RA].O = L.Stack.Values[L.Stack.Base + RB].O;
        }

        public bool EqualVal(LuaState L, LuaReference o1, LuaReference o2)
        {
            if (o1 == null || o2 == null || o1.Tag != o2.Tag) return false;
            if (o1.Equals(o2)) return true;
            return CallCompMeta(L, o1, o2, LuaOpCode.OP_EQ);
        }

        public bool LessThanVal(LuaState L, LuaReference o1, LuaReference o2)
        {
            if (o1 == null || o2 == null || o1.Tag != o2.Tag)
                throw new RuntimeException("Order error: incompatible values");
            if (o1.LessThan(o2)) return true;
            return CallCompMeta(L, o1, o2, LuaOpCode.OP_LT);
        }

        public bool LessEqualVal(LuaState L, LuaReference o1, LuaReference o2)
        {
            if (o1 == null || o2 == null || o1.Tag != o2.Tag)
                throw new RuntimeException("Order error: incompatible values");
            if (!o1.GreaterThan(o2)) return true;
            return CallCompMeta(L, o1, o2, LuaOpCode.OP_LE)
                || !CallCompMeta(L, o2, o1, LuaOpCode.OP_LT);
        }

        public bool OpEq(LuaState L, int RA, int RB, int RC)
        {
            LuaValue[] stack1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RB : RB - LuaState.MAX_LOCAL_STACK);
            LuaValue[] stack2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RC : RC - LuaState.MAX_LOCAL_STACK);
            if (stack1[idx1].O == null && stack2[idx2].O == null)
                return (stack1[idx1].N == stack2[idx2].N) != (RA != 0);
            else
                return (EqualVal(L, stack1[idx1].O, stack2[idx2].O)) != (RA != 0);
        }

        public bool OpLt(LuaState L, int RA, int RB, int RC)
        {
            LuaValue[] stack1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RB : RB - LuaState.MAX_LOCAL_STACK);
            LuaValue[] stack2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RC : RC - LuaState.MAX_LOCAL_STACK);
            if (stack1[idx1].O == null && stack2[idx2].O == null)
                return (stack1[idx1].N < stack2[idx2].N) != (RA != 0);
            else
                return LessThanVal(L, stack1[idx1].O, stack2[idx2].O) != (RA != 0);
        }

        public bool OpLe(LuaState L, int RA, int RB, int RC)
        {
            LuaValue[] stack1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx1 = (RB < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RB : RB - LuaState.MAX_LOCAL_STACK);
            LuaValue[] stack2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Values : proto.Constants);
            int idx2 = (RC < LuaState.MAX_LOCAL_STACK ? L.Stack.Base + RC : RC - LuaState.MAX_LOCAL_STACK);
            if (stack1[idx1].O == null && stack2[idx2].O == null)
                return (stack1[idx1].N <= stack2[idx2].N) != (RA != 0);
            else
                return LessEqualVal(L, stack1[idx1].O, stack2[idx2].O) != (RA != 0);
        }

        public bool OpTest(LuaState L, int RA, int RB, int RC)
        {
            if ((L.Stack.Values[L.Stack.Base + RB].O == NilClass.Instance ||
                L.Stack.Values[L.Stack.Base + RB].O == FalseClass.Instance) && (RC != 0))
            {
                return true;
            }
            else
            {
                L.Stack.Values[L.Stack.Base + RA] = L.Stack.Values[L.Stack.Base + RB];
                return false;
            }
        }

        public void OpCall(LuaState L, int RA, int RB, int RC)
        {
            LuaReference fun = L.Stack.Values[L.Stack.Base + RA].O;
            if (fun != null)
            {
                if (RB != 0)
                    L.Stack.Top = L.Stack.Base + RA + RB;
                fun.Call(L, RC - 1, RA + 1);
            }
            else
                throw new RuntimeException("trying to call a numeric value");
        }

        public int OpReturn(LuaState L, int RA, int RB, ref CallInfo callInfo)
        {
            if (RB != 0)
                L.Stack.Top = L.Stack.Base + RA + RB - 1;
            if (L.OpenUpvalues != null)
            {
                OpClose(L, 0);
            }
            PosCall(L, callInfo.NResults, L.Stack.Base + RA, callInfo.LastBase);
            if (callInfo.NResults >= 0) L.Stack.Top = callInfo.MaxTop;
            return RB - 1;
        }

        public bool OpForLoop(LuaState L, int RA)
        {
            if (L.Stack.Values[L.Stack.Base + RA].O != null)
                throw new Exception("'for' initial value must be a number");
            if (L.Stack.Values[L.Stack.Base + RA + 1].O != null)
                throw new Exception("'for' limit value must be a number");
            if (L.Stack.Values[L.Stack.Base + RA + 2].O != null)
                throw new Exception("'for' step value must be a number");
            double step, idx, limit;
            step = L.Stack.Values[L.Stack.Base + RA + 2].N;
            idx = L.Stack.Values[L.Stack.Base + RA].N + step;
            limit = L.Stack.Values[L.Stack.Base + RA + 1].N;
            if (step > 0 ? idx <= limit : idx >= limit)
            {
                L.Stack.Values[L.Stack.Base + RA].N = idx;
                return true;
            }
            else return false;
        }

        public bool OpTForLoop(LuaState L, int RA, int RC)
        {
            int nvar = RC + 1;
            int callBase = L.Stack.Base + RA + nvar + 2;
            int fun = L.Stack.Base + RA;
            L.Stack.Values[callBase] = L.Stack.Values[fun];
            L.Stack.Values[callBase + 1] = L.Stack.Values[fun + 1];
            L.Stack.Values[callBase + 2] = L.Stack.Values[fun + 2];
            L.Stack.Top = callBase + 3;
            L.Stack.Values[fun].O.Call(L, nvar, callBase + 1);
            fun = L.Stack.Base + RA + 2;
            callBase = fun + nvar;
            do
            {
                nvar--;
                L.Stack.Values[fun + nvar] = L.Stack.Values[callBase + nvar];
            } while (nvar > 0);
            return L.Stack.Values[fun].O == NilClass.Instance;
        }

        public void OpTForPrep(LuaState L, int RA)
        {
            if (L.Stack.Values[L.Stack.Base + RA].O.Tag == LuaTypes.LUA_TTABLE)
            {
                L.Stack.Values[L.Stack.Base + RA + 1].O = L.Stack.Values[L.Stack.Base + RA].O;
                L.Stack.Values[L.Stack.Base + RA].O = new Pairs(L.Globals, (LuaTable)L.Stack.Values[L.Stack.Base + RA].O);
            }
        }

        public void OpSetList(LuaState L, int RA, int bc, int n)
        {
            LuaTable table = (LuaTable)L.Stack.Values[L.Stack.Base + RA].O;
            bc &= ~(LuaState.LFIELDS_PER_FLUSH - 1);
            for (; n > 0; n--)
                table[new LuaValue((double)bc + n)] = L.Stack.Values[L.Stack.Base + RA + n];
        }

        public void OpClose(LuaState L, int RA)
        {
            UpValue upval;
            int level = L.Stack.Base + RA;
            UpValue up = L.OpenUpvalues;
            while (up != null)
            {
                up = up.Next;
            }
            while ((upval = L.OpenUpvalues) != null && upval.Level >= level)
            {
                upval.Close();
                L.OpenUpvalues = upval.Next;
            }
        }

        public void OpMoveClosure(LuaState L, int RA, LuaClosure cl)
        {
            L.Stack.Values[L.Stack.Base + RA].O = cl;
        }
    }


    public class MethodProxy : LuaFunction, IMemberWrapper
    {
        MethodBase method;
        ParameterInfo[] paramInfo;
        ConvToCLR[] converters;
        object[] methodParams;
        ConvToLua[] returnConv;
        int[] outParams;
        int retVals;
        InvokeMethod invoker;

        delegate object InvokeMethod(object target, object[] argv);

        object InvokeMeth(object target, object[] argv)
        {
            return method.Invoke(target, argv);
        }

        object InvokeCons(object target, object[] argv)
        {
            return ((ConstructorInfo)method).Invoke(argv);
        }

        public MethodProxy(LuaReference globals, MethodBase method)
            : base(globals)
        {
            this.method = method;
            paramInfo = method.GetParameters();
            converters = new ConvToCLR[paramInfo.Length];
            methodParams = new object[paramInfo.Length];
            int nOut = 0;
            for (int i = 0; i < converters.Length; i++)
            {
                converters[i] = TypeChecker.GetToCLRConv(paramInfo[i].ParameterType);
                if (paramInfo[i].ParameterType.IsByRef) nOut++;
            }
            returnConv = new ConvToLua[nOut + 1];
            outParams = new int[nOut];
            Type returnType = (method.IsConstructor ? method.ReflectedType : ((MethodInfo)method).ReturnType);
            returnConv[0] = TypeChecker.GetToLuaConv(returnType);
            for (int i = 0, j = 1; i < paramInfo.Length; i++)
                if (paramInfo[i].ParameterType.IsByRef)
                {
                    returnConv[j] = TypeChecker.GetToLuaConv(paramInfo[i].ParameterType.GetElementType());
                    outParams[j - 1] = i;
                    j++;
                }
            if (returnType.Equals(typeof(void)))
                retVals = nOut;
            else
                retVals = nOut + 1;
            invoker = (method.IsConstructor ? new InvokeMethod(this.InvokeCons) :
                new InvokeMethod(this.InvokeMeth));
        }

        public override bool Execute(LuaState L, out int result)
        {
            int j = L.Stack.Base;
            int top = L.Stack.Top;
            LuaValue[] values = L.Stack.Values;
            object target = null;
            if (!method.IsStatic && !method.IsConstructor && j < top)
            {
                target = values[j].CLRObject;
                j++;
            }
            for (int i = 0; i < methodParams.Length; i++)
            {
                if (paramInfo[i].Attributes != ParameterAttributes.Out ||
                    paramInfo[i].Attributes == ParameterAttributes.In)
                {
                    if (j < top)
                    {
                        object param = converters[i](ref values[j]);
                        if (param == null && values[j].O != NilClass.Instance)
                        {
                            //throw new ArgumentException("Invalid parameter type", paramInfo[i].Name);
                            result = 0;
                            return false;
                        }
                        methodParams[i] = param;

                    }
                    else
                    {
                        //throw new TargetParameterCountException("Too few parameters to method");
                        result = 0;
                        return false;
                    }
                    j++;
                }
                else
                {
                    methodParams[i] = null;
                }
            }
            if (j < top)
            {
                //throw new TargetParameterCountException("Too many parameters to method call");
                result = 0;
                return false;
            }
            object returnValue = invoker(target, methodParams);
            returnConv[0](L, top, returnValue);
            top++;
            for (int i = 1; i < returnConv.Length; i++)
            {
                returnConv[i](L, top, methodParams[outParams[i - 1]]);
                top++;
            }
            L.Stack.Top = top;
            result = retVals;
            return true;
        }

        public LuaValue GetValue(object wrapped)
        {
            return new LuaValue(this);
        }
        public void SetValue(object wrapped, LuaValue val)
        {
            throw new Exception("Methods cannot be set");
        }
    }

    public class OverloadedMethodProxy : LuaFunction, IMemberWrapper
    {
        MethodProxy[] methods;
        MethodProxy lastMatch;

        public OverloadedMethodProxy(LuaReference globals, MethodBase[] methods)
            : base(globals)
        {
            this.methods = new MethodProxy[methods.Length];
            for (int i = 0; i < methods.Length; i++)
                this.methods[i] = new MethodProxy(globals, methods[i]);
            lastMatch = null;
        }

        public override bool Execute(LuaState L, out int result)
        {
            bool success = false;
            if (lastMatch != null)
            {
                try
                {
                    success = lastMatch.Execute(L, out result);
                }
                catch (TargetInvocationException tie)
                {
                    throw tie.GetBaseException();
                }
                /*catch(Exception) 
                {
                    lastMatch=null;
                }*/
                if (success)
                    return true;
                lastMatch = null;
            }
            return ExecuteAll(L, out result);
        }

        bool ExecuteAll(LuaState L, out int result)
        {
            foreach (MethodProxy m in methods)
            {
                bool success = false;
                try
                {
                    lastMatch = m;
                    success = m.Execute(L, out result);
                }
                catch (TargetInvocationException tie)
                {
                    throw tie.GetBaseException();
                }
                /*catch(Exception) 
                {
                    lastMatch=null;
                }*/
                if (success)
                    return true;
                lastMatch = null;
            }
            throw new Exception("No method match found");
        }

        public LuaValue GetValue(object wrapped)
        {
            return new LuaValue(this);
        }

        public void SetValue(object wrapped, LuaValue val)
        {
            throw new Exception("Methods cannot be set");
        }

    }

    public abstract class LuaWrapper : LuaReference
    {
        protected object wrapped;
        protected Type wrappedType;
        protected Hashtable members;
        protected BindingFlags flags;

        public override object CLRObject
        {
            get
            {
                return wrapped;
            }
        }

        public override LuaValue this[LuaState L, LuaValue index]
        {
            get
            {
                return this[index];
            }
            set
            {
                this[index] = value;
            }
        }

        public override LuaValue this[LuaValue index]
        {
            get
            {
                if (index.O == null)
                    return new LuaValue(NilClass.Instance);
                string memberName = index.O.ToStr();
                object o = members[memberName];
                if (o != null)
                    return (LuaValue)((IMemberWrapper)o).GetValue(wrapped);
                IMemberWrapper member = GetMember(memberName);
                if (member == null)
                    return new LuaValue(NilClass.Instance);
                members[memberName] = member;
                return member.GetValue(wrapped);
            }
            set
            {
                if (index.O == null)
                    throw new Exception("Member does not exist");
                string memberName = index.O.ToString();
                object o = members[memberName];
                if (o != null)
                {
                    ((IMemberWrapper)o).SetValue(wrapped, value);
                    return;
                }
                IMemberWrapper member = GetMember(memberName);
                if (member == null)
                    throw new Exception("Member does not exist");
                members[memberName] = member;
                member.SetValue(wrapped, value);
            }
        }

        IMemberWrapper GetMember(string memberName)
        {
            MemberInfo[] members = wrappedType.GetMember(memberName, BindingFlags.Public | flags);

            // count fields, properties and methods
            int numMethods = 0;
            foreach (MemberInfo member in members)
            {
                if (member.MemberType == MemberTypes.Field)
                {
                    return new FieldProxy((FieldInfo)member);
                }
                if (member.MemberType == MemberTypes.Property)
                {
                    PropertyInfo property = (PropertyInfo)member;
                    if (property.GetIndexParameters().Length == 0)
                        return new PropertyProxy(property);
                }
#if LUA2IL_CANEMIT
                if (member.MemberType == MemberTypes.Event)
                {
                    return new EventProxyFactory((EventInfo)member);
                }
#endif
                if (member.MemberType == MemberTypes.Method)
                    numMethods++;
            }

            if (numMethods == 1)
            {
                foreach (MemberInfo member in members)
                {
                    if (member.MemberType == MemberTypes.Method)
                        return new MethodProxy(null, (MethodBase)member);
                }
            }
            else if (numMethods > 1)
            {
                MethodInfo[] methods = new MethodInfo[numMethods];
                int i = 0;
                foreach (MemberInfo member in members)
                {
                    if (member.MemberType == MemberTypes.Method)
                        methods[i++] = (MethodInfo)member;
                }

                return new OverloadedMethodProxy(null, (MethodInfo[])methods);
            }
            return null;
        }

        public override bool ToBoolean()
        {
            return (wrapped != null);
        }

        public override string ToString()
        {
            return wrapped.ToString();
        }

        public override bool Equals(LuaReference r)
        {
            return (r != null) && (r.Tag == LuaTypes.LUA_TUSERDATA) && (wrapped.Equals(((LuaWrapper)r).CLRObject));
        }

        public override bool Equals(object o)
        {
            if (o == null) return false;
            try
            {
                return wrapped.Equals(((LuaWrapper)o).CLRObject);
            }
            catch (InvalidCastException)
            {
                return false;
            }
        }

        public override int GetHashCode()
        {
            return wrapped.GetHashCode();
        }
    }

    public class LuaObjectWrapper : LuaWrapper
    {
        static Hashtable typeCache;

        static LuaObjectWrapper()
        {
            typeCache = new Hashtable();
        }

        public LuaObjectWrapper(object wrapped)
        {
            this.wrapped = wrapped;
            wrappedType = wrapped.GetType();
            lock (typeCache)
            {
                members = (Hashtable)typeCache[wrappedType];
                if (members == null)
                {
                    members = new Hashtable();
                    typeCache[wrappedType] = members;
                }
            }
            flags = BindingFlags.Instance;
            Tag = LuaTypes.LUA_TUSERDATA;
        }
    }

    public class LuaTypeWrapper : LuaWrapper
    {
        static Hashtable typeCache;
        LuaFunction constructor;

        static LuaTypeWrapper()
        {
            typeCache = new Hashtable();
        }

        public LuaTypeWrapper(Type wrapped)
        {
            this.wrapped = wrapped;
            wrappedType = wrapped;
            lock (typeCache)
            {
                members = (Hashtable)typeCache[wrappedType];
                if (members == null)
                {
                    members = new Hashtable();
                    typeCache[wrappedType] = members;
                }
            }
            flags = BindingFlags.Static;
            ConstructorInfo[] cons = wrappedType.GetConstructors();
            if (cons.Length == 1)
                constructor = new MethodProxy(null, cons[0]);
            else
                constructor = new OverloadedMethodProxy(null, cons);
            Tag = LuaTypes.LUA_TUSERDATA;
        }

        public override int Call(LuaState L, int nResults, int firstArg)
        {
            return constructor.Call(L, nResults, firstArg);
        }
    }

    public interface IMemberWrapper
    {
        LuaValue GetValue(object wrapped);
        void SetValue(object wrapped, LuaValue val);
    }

    public class FieldProxy : IMemberWrapper
    {
        FieldInfo field;
        ConvFromCLR getConv;
        ConvToCLR setConv;

        public FieldProxy(FieldInfo field)
        {
            this.field = field;
            getConv = TypeChecker.GetFromCLRConv(field.FieldType);
            setConv = TypeChecker.GetToCLRConv(field.FieldType);
        }

        public LuaValue GetValue(object wrapped)
        {
            return getConv(field.GetValue(wrapped));
        }

        public void SetValue(object wrapped, LuaValue val)
        {
            field.SetValue(wrapped, setConv(ref val));
        }
    }

    public class PropertyProxy : IMemberWrapper
    {
        PropertyInfo property;
        ConvFromCLR getConv;
        ConvToCLR setConv;

        public PropertyProxy(PropertyInfo property)
        {
            this.property = property;
            getConv = TypeChecker.GetFromCLRConv(property.PropertyType);
            setConv = TypeChecker.GetToCLRConv(property.PropertyType);
        }

        public LuaValue GetValue(object wrapped)
        {
            return getConv(property.GetValue(wrapped, null));
        }

        public void SetValue(object wrapped, LuaValue val)
        {
            property.SetValue(wrapped, setConv(ref val), null);
        }
    }
#if LUA2IL_CANEMIT
	public class EventProxyFactory : IMemberWrapper
	{
		EventInfo eventInfo;

		public EventProxyFactory(EventInfo eventInfo) 
		{
			this.eventInfo=eventInfo;
		}

		public LuaValue GetValue(object wrapped)
		{
			return new LuaValue(new LuaObjectWrapper(new EventProxy(wrapped,eventInfo)));
		}

		public void SetValue(object wrapped,LuaValue val)
		{
			throw new Exception("events cannot be set.");
		}
	}

	public class EventProxy 
	{
		object target;
		EventInfo eventInfo;

		public EventProxy(object target,EventInfo eventInfo) 
		{
			this.target=target;
			this.eventInfo=eventInfo;
		}

		public Delegate Add(LuaFunction fun) 
		{
			Delegate handler=CodeGeneration.Instance.GetDelegate(eventInfo.EventHandlerType,
				fun);
			eventInfo.AddEventHandler(target,handler);
			return handler;
		}

		public void Remove(Delegate handler) 
		{
			eventInfo.RemoveEventHandler(target,handler);
		}
	}
#endif
    public class LuaDelegate
    {
        public Type[] returnTypes;
        public LuaFunction function;
        object[] returnValues = new object[256];//߂lmۗp

        public LuaDelegate()
        {
            function = null;
            returnTypes = null;
        }

        public object callFunction(object[] args, object[] inArgs, int[] outArgs)
        {
            object returnValue;
            int iRefArgs;
            Array.Clear(returnValues, 0, returnValues.Length);
            function.Call(inArgs, returnTypes, returnValues);
            if (returnTypes[0].Equals(typeof(void)))
            {
                returnValue = null;
                iRefArgs = 0;
            }
            else
            {
                returnValue = returnValues[0];
                iRefArgs = 1;
            }
            // Sets the value of out and ref parameters (from
            // the values returned by the Lua function).
            for (int i = 0; i < outArgs.Length; i++)
            {
                args[outArgs[i]] = returnValues[iRefArgs];
                iRefArgs++;
            }
            return returnValue;
        }
    }


}
