/*
 * 
 * XNua 0.1 Based Massively on Lua2IL 0.5
 * Lua2IL 0.5.0
 * Exceptions.cs - XNua exceptions
 * Copyright 2003-2005 Fabio Mascarenhas
 *                2007 Dean Calver
 * 
 */

using System;
using System.Collections.Generic;
using System.Text;
using System.Collections;
using System.Reflection;
using System.Diagnostics;

namespace XNua
{
    public class LuaBaseLib
    {
        /// <summary>
        /// A helper function that installs all the lua
        /// standard libraries into the global state
        /// </summary>
        /// <param name="L">State to have base functions added</param>
        static public void InstallStdLib(LuaState L)
        {
            LuaReference Globals = L.Globals;

            InstallBaseLib(L);

            InstallMathLib(L);

            InstallTableLib(L);

            InstallOsLib(L);

            InstallCoroutineLib(L);

            Globals["load_assembly"] = new LoadAssembly(Globals);
            Globals["import_type"] = new ImportType(Globals);
#if XBOX360
            Globals["_XBOX360"] = TrueClass.Instance;
#else
            Globals["_XBOX360"] = FalseClass.Instance;
#endif

        }
        /// <summary>
        /// Lua doesn't really operate without these
        /// </summary>
        /// <param name="L">LuaState</param>
        static public void InstallBaseLib(LuaState L)
        {
            LuaReference Globals = L.Globals;

            Globals["_G"] = Globals;
            Globals["next"] = new Next(Globals);
            Globals["pairs"] = new PairsFactory(Globals);
            Globals["ipairs"] = new IpairsFactory(Globals);
            Globals["rawequal"] = new RawEqual(Globals);
            Globals["rawget"] = new RawGet(Globals);
            Globals["rawset"] = new RawSet(Globals);
            Globals["tostring"] = new LuaToString(Globals);
            Globals["tonumber"] = new LuaToNumber(Globals);
            Globals["type"] = new LuaType(Globals);

            Globals["max"] = new MethodProxy(Globals,
                typeof(Math).GetMethod("Max", new Type[] { typeof(int), typeof(int) }));
            Globals["printf"] = new MethodProxy(Globals,
                typeof(Console).GetMethod("WriteLine", new Type[] { typeof(int) }));
            Globals["print"] = new LuaPrint(Globals);
            Globals["assert"] = new LuaAssert(Globals);
        }

        /// <summary>
        /// Os level Lua function, only clock actually works properly...
        /// </summary>
        /// <param name="L">LuaState</param>
        public static void InstallOsLib(LuaState L)
        {
            LuaReference Globals = L.Globals;
            LuaValue os = new LuaValue(new LuaTable());

            Globals["os"] = os;
            os["clock"] = new LuaValue(new Clock(Globals));
            os["time"] = new LuaValue(new Time(Globals));
        }
        /// <summary>
        /// Math functions, most done, a few still not done
        /// </summary>
        /// <param name="L">LuaState</param>
        public static void InstallMathLib(LuaState L)
        {
            LuaReference Globals = L.Globals;

            LuaValue math = new LuaTable();
            Globals["math"] = math;
            math["abs"] = new Abs(Globals);
            math["acos"] = new ACos(Globals);
            math["asin"] = new ASin(Globals);
            math["atan"] = new ATan(Globals);
            math["atan2"] = new ATan2(Globals);
            math["ceil"] = new Ceil(Globals);
            math["cos"] = new Cos(Globals);
            math["deg"] = new Deg(Globals);
            math["exp"] = new Exp(Globals);
            math["floor"] = new Floor(Globals);
            math["log"] = new Log(Globals);
            math["log10"] = new Log10(Globals);
            math["max"] = new Max(Globals);
            math["min"] = new Min(Globals);
            math["mod"] = new Modulo(Globals);
            math["pow"] = new Pow(Globals);
            math["rad"] = new Rad(Globals);
            math["sin"] = new Sin(Globals);
            math["sqrt"] = new Sqrt(Globals);
            math["tan"] = new Tan(Globals);
        }

        /// <summary>
        /// Coroutines, not tested much 
        /// </summary>
        /// <param name="L">LuaState</param>
        public static void InstallCoroutineLib(LuaState L)
        {
            LuaReference Globals = L.Globals;
            Globals["coroutine"] = new LuaTable();
            LuaValue coro = Globals["coroutine"];
            coro["create"] = new CoCreate(Globals);
            coro["resume"] = new CoResume(Globals);
            coro["yield"] = new CoYield(Globals);
            coro["wrap"] = new CoWrap(Globals);
        }
        /// <summary>
        /// Tables library
        /// </summary>
        /// <param name="L">LuaState</param>
        public static void InstallTableLib(LuaState L)
        {
            LuaReference Globals = L.Globals;
            Globals["table"] = new LuaTable();
            LuaValue tab = Globals["table"];
            tab["setn"] = new TableSetN(Globals);
            tab["getn"] = new TableGetN(Globals);
            tab["insert"] = new TableInsert(Globals);
            tab["remove"] = new TableRemove(Globals);
        }
    }
    /// <summary>
    /// If the first Lua parameters is nil or false
    /// print out the rest of the parameters
    /// </summary>
    public class LuaAssert : LuaFunction
    {
        public LuaAssert(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            // require one parameter
            Debug.Assert(L.Stack.Base < L.Stack.Top);

            // if the first parameter evaluates to false or nil
            if (L.Stack[L.Stack.Base].O == NilClass.Instance ||
                L.Stack[L.Stack.Base].O == FalseClass.Instance)
            {
                if (L.Stack.Base + 1 < L.Stack.Top)
                {
                    for (int i = L.Stack.Base + 1; i < L.Stack.Top; i++)
                    {
                        Console.Write(L.Stack[i].ToString() + "\t");
                    }
                    Console.WriteLine();
                }
                else
                {
                    Console.WriteLine("assertion failed!");
                }
            }
            result = 0;
            return true;
        }
    }
    /// <summary>
    /// gives a string type name for a parameter lua stack
    /// </summary>
    public class LuaType : LuaFunction
    {
        public LuaType(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = L.Stack.typename(index);
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// print out each parameter on the lua stack
    /// </summary>
    public class LuaPrint : LuaFunction
    {
        public LuaPrint(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            for (int i = L.Stack.Base; i < L.Stack.Top; i++)
            {
                System.Diagnostics.Trace.Write(L.Stack[i].ToString() + "\t");
            }
            System.Diagnostics.Trace.WriteLine("");
            result = 0;
            return true;
        }
    }

    /// <summary>
    /// current time used for timing
    /// </summary>
    public class Clock : LuaFunction
    {
        public Clock(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            double now = ((double)DateTime.Now.Minute * 60) + ((double)DateTime.Now.Second)
                + ((double)DateTime.Now.Millisecond / 1000);
            L.Stack[L.Stack.Top++] = now;
            result = 1;
            return true;
        }
    }

    /// <summary>
    /// partial implemention of os.time
    /// </summary>
    public class Time : LuaFunction
    {
        public Time(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            // parameter to be filled is optional and must be a table
            if (L.Stack[L.Stack.Base].O != null && L.Stack[L.Stack.Base].O is LuaTable)
            {
                LuaTable inTable = (LuaTable)L.Stack[L.Stack.Base].O;
                if (inTable["year"].O != NilClass.Instance &&
                    inTable["month"].O != NilClass.Instance &&
                    inTable["day"].O != NilClass.Instance)
                {
                }
                else
                {
                    // local time in seconds
                    double now = ((double)DateTime.Now.ToFileTime() * 1e7);
                    L.Stack[L.Stack.Top] = now;
                    L.Stack.Top++;

                }

            }
            else
            {
            }
            result = 1;
            return true;
        }
    }

    /// <summary>
    /// Math.abs
    /// </summary>
    public class Abs : LuaFunction
    {
        public Abs(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Abs(L.Stack[index].ToNumber());
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// Math.acos
    /// </summary>
    public class ACos : LuaFunction
    {
        public ACos(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Acos(L.Stack[index].ToNumber());
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// math.asin
    /// </summary>
    public class ASin : LuaFunction
    {
        public ASin(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Asin(L.Stack[index].ToNumber());
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// math.atan
    /// </summary>
    public class ATan : LuaFunction
    {
        public ATan(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Atan(L.Stack[index].ToNumber());
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// math.atan2
    /// </summary>
    public class ATan2 : LuaFunction
    {
        public ATan2(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Atan2(L.Stack[index].ToNumber(), L.Stack[index + 1].ToNumber());
            result = 1;
            return true;
        }
    }

    /// <summary>
    /// math.ceil
    /// </summary>
    public class Ceil : LuaFunction
    {
        public Ceil(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Ceiling(L.Stack[index].ToNumber());
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// math.cos
    /// </summary>
    public class Cos : LuaFunction
    {
        public Cos(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Cos(L.Stack[index].ToNumber());
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// math.deg
    /// </summary>
    public class Deg : LuaFunction
    {
        public Deg(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = L.Stack[index].ToNumber() / (Math.PI / 180.0f);
            result = 1;
            return true;
        }
    }

    /// <summary>
    /// math.exp
    /// </summary>
    public class Exp : LuaFunction
    {
        public Exp(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Exp(L.Stack[index].ToNumber());
            result = 1;
            return true;
        }
    }

    /// <summary>
    /// math.floor
    /// </summary>
    public class Floor : LuaFunction
    {
        public Floor(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Floor(L.Stack[index].ToNumber());
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// math.log
    /// </summary>
    public class Log : LuaFunction
    {
        public Log(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Log(L.Stack[index].ToNumber());
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// math.log10
    /// </summary>
    public class Log10 : LuaFunction
    {
        public Log10(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Log10(L.Stack[index].ToNumber());
            result = 1;
            return true;
        }
    }

    /// <summary>
    /// math.max
    /// </summary>
    public class Max : LuaFunction
    {
        public Max(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            double val = L.Stack[L.Stack.Base].ToNumber();
            for (int i = L.Stack.Base + 1; i < L.Stack.Top; i++)
            {
                val = Math.Max(val, L.Stack[i].ToNumber());
            }
            L.Stack[top] = val;

            result = 1;
            return true;
        }
    }
    /// <summary>
    /// math.min
    /// </summary>
    public class Min : LuaFunction
    {
        public Min(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            double val = L.Stack[L.Stack.Base].ToNumber();
            for (int i = L.Stack.Base + 1; i < L.Stack.Top; i++)
            {
                val = Math.Min(val, L.Stack[i].ToNumber());
            }
            L.Stack[top] = val;

            result = 1;
            return true;
        }
    }

    /// <summary>
    /// math.mod
    /// </summary>
    public class Modulo : LuaFunction
    {
        public Modulo(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = L.Stack[index].ToNumber() % L.Stack[index + 1].ToNumber();
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// math.pow
    /// </summary>
    public class Pow : LuaFunction
    {
        public Pow(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Pow(L.Stack[index].ToNumber(), L.Stack[index + 1].ToNumber());
            result = 1;
            return true;
        }
    }

    /// <summary>
    /// math.rad - degrees to radian
    /// </summary>
    public class Rad : LuaFunction
    {
        public Rad(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = L.Stack[index].ToNumber() * (Math.PI / 180.0f);
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// math.sin
    /// </summary>
    public class Sin : LuaFunction
    {
        public Sin(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Sin(L.Stack[index].ToNumber());
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// math.sqrt
    /// </summary>
    public class Sqrt : LuaFunction
    {
        public Sqrt(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Sqrt(L.Stack[index].ToNumber());
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// math.tan
    /// </summary>
    public class Tan : LuaFunction
    {
        public Tan(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            L.Stack[top] = Math.Tan(L.Stack[index].ToNumber());
            result = 1;
            return true;
        }
    }

    /// <summary>
    /// Load a CLR assembly for use by Lua
    /// </summary>
    public class LoadAssembly : LuaFunction
    {
        public static ArrayList Assemblies;

        public LoadAssembly(LuaReference globals)
            : base(globals)
        {
            Assemblies = ArrayList.Synchronized(new ArrayList());
        }

        public override bool Execute(LuaState L, out int result)
        {
            string assemblyName = L.Stack[L.Stack.Base].ToString();
            Assembly assembly = null;
            try
            {
                assembly = Assembly.Load(assemblyName);
            }
            catch (Exception)
            {
#if !XBOX360
                try
                {
                    //assembly = Assembly.LoadWithPartialName(assemblyName);
                    assembly = Assembly.Load(assemblyName);
                }
                catch (Exception)
                {
                    result = 0;
                    return false;
                }
#else
                result= 0;
                return false;
#endif
            }
            if (Assemblies.IndexOf(assembly) == -1)
                Assemblies.Add(assembly);
            result = 0;
            return true;
        }
    }
    /// <summary>
    /// Import a clr type into Lua
    /// </summary>
    public class ImportType : LuaFunction
    {
        public ImportType(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            string className = L.Stack[L.Stack.Base].ToString();
            foreach (Assembly assembly in LoadAssembly.Assemblies)
            {
                Type klass = assembly.GetType(className);
                if (klass != null)
                {
                    L.Stack[L.Stack.Top++] = new LuaTypeWrapper(klass);
                    result = 1;
                    return true;
                }
            }
            L.Stack.Values[L.Stack.Top++] = NilClass.Instance;
            result = 1;
            return true;
        }
    }

    /// <summary>
    /// Next iterator
    /// </summary>
    public class Next : LuaFunction
    {
        public Next(LuaReference globals)
            : base(globals)
        {
        }
        public override bool Execute(LuaState L, out int result)
        {
            Debug.Assert(L.Stack[L.Stack.Base].IsReference);
            Debug.Assert(L.Stack[L.Stack.Base].O is LuaTable);
            // first param is the table
            LuaTable table = (LuaTable)L.Stack[L.Stack.Base].O;
            // second param if it exist is the current index
            // TODO check for nill
            int tableindex = (int)L.Stack[L.Stack.Base + 1].ToNumber();
            Node[] tmp = null;
            if (table.GetNext(ref tmp, ref tableindex))
            {
                L.Stack[L.Stack.Top - 1] = tmp[tableindex].key;
                L.Stack[L.Stack.Top++] = tmp[tableindex].val;
                result = 2;
                return true;
            }
            else
            {
                result = 0;
                return true;
            }
        }
    }

    /// <summary>
    /// pairs, creates the pairs closure function
    /// </summary>
    public class PairsFactory : LuaFunction
    {
        public PairsFactory(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            //            Debug.Assert(L.Stack[index].IsReference && L.Stack[index].O.Tag == LuaTypes.LUA_TTABLE);
            L.Stack[top] = new Pairs(L.Globals, (LuaTable)L.Stack[index].O);
            result = 1;
            return true;
        }
    }

    /// <summary>
    /// ipairs - creates the ipairs closure
    /// </summary>
    public class IpairsFactory : LuaFunction
    {
        public IpairsFactory(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;
            Debug.Assert(L.Stack[index].IsReference && L.Stack[index].O.Tag == LuaTypes.LUA_TTABLE);
            L.Stack[top] = new Ipairs(L.Globals, (LuaTable)L.Stack[index].O);
            result = 1;
            return true;
        }
    }

    /// <summary>
    /// pairs - iterator for pairs
    /// </summary>
    public class Pairs : LuaFunction
    {
        Node[] currentStructure;
        int currentIndex;
        LuaTable table;

        public Pairs(LuaReference globals, LuaTable table)
            : base(globals)
        {
            currentStructure = null;
            currentIndex = -1;
            this.table = table;
        }

        public override bool Execute(LuaState L, out int result)
        {
            if (table.GetNext(ref currentStructure, ref currentIndex))
            {
                L.Stack[L.Stack.Top - 1] = currentStructure[currentIndex].key;
                L.Stack[L.Stack.Top++] = currentStructure[currentIndex].val;
                result = 2;
                return true;
            }
            else
            {
                result = 0;
                return true;
            }
        }
    }
    /// <summary>
    /// iterator for ipairs
    /// </summary>
    public class Ipairs : LuaFunction
    {
        Node[] currentStructure;
        int currentIndex;
        LuaTable table;

        public Ipairs(LuaReference globals, LuaTable table)
            : base(globals)
        {
            currentStructure = null;
            currentIndex = -1;
            this.table = table;
        }

        public override bool Execute(LuaState L, out int result)
        {
            if (table.GetNextInt(ref currentStructure, ref currentIndex))
            {
                L.Stack[L.Stack.Top - 1] = currentStructure[currentIndex].key;
                L.Stack[L.Stack.Top++] = currentStructure[currentIndex].val;
                result = 2;
                return true;
            }
            else
            {
                result = 0;
                return true;
            }
        }
    }

    /// <summary>
    /// rawequal
    /// </summary>
    public class RawEqual : LuaFunction
    {
        public RawEqual(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int top = L.Stack.Top - 1;
            L.Stack[top] = L.Stack.rawequal(1, 2) ? (LuaReference)TrueClass.Instance : (LuaReference)FalseClass.Instance;
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// rawget
    /// </summary>
    public class RawGet : LuaFunction
    {
        public RawGet(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            L.Stack.rawget(1);
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// rawset
    /// </summary>
    public class RawSet : LuaFunction
    {
        public RawSet(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            L.Stack.rawset(1);
            result = 0;
            return true;
        }
    }
    /// <summary>
    /// tostring
    /// </summary>
    public class LuaToString : LuaFunction
    {
        public LuaToString(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int top = L.Stack.Top - 1;
            L.Stack[top] = L.Stack.tostring(1);
            result = 1;
            return true;
        }
    }
    /// <summary>
    /// tonumber
    /// </summary>
    public class LuaToNumber : LuaFunction
    {
        public LuaToNumber(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int top = L.Stack.Top - 1;
            L.Stack[top] = L.Stack.tonumber(1);
            result = 1;
            return true;
        }
    }

    /// <summary>
    /// table.setn
    /// </summary>
    public class TableSetN : LuaFunction
    {
        public TableSetN(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int index = L.Stack.Base;
            int top = L.Stack.Top - 1;

            double n = L.Stack[index + 1].ToNumber();
            L.Stack.aux_setn(1, (int)n);
            result = 0;
            return true;
        }
    }
    /// <summary>
    /// table.getn
    /// </summary>
    public class TableGetN : LuaFunction
    {
        public TableGetN(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int top = L.Stack.Top - 1;
            int n = L.Stack.aux_getn(1);
            L.Stack[top] = new LuaValue((double)n);
            result = 1;
            return true;
        }

    }
    /// <summary>
    /// table.insert
    /// </summary>
    public class TableInsert : LuaFunction
    {
        public TableInsert(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int v = L.Stack.gettop();
            int n = L.Stack.aux_getn(1) + 1;
            int pos;
            if (v == 2)
                pos = n;
            else
            {
                pos = L.Stack.checkint(2);
                if (pos > n)
                {
                    n = pos;
                }
                v = 3;
            }
            L.Stack.aux_setn(1, n);
            while (--n >= pos)
            {
                L.Stack.rawgeti(1, n);
                L.Stack.rawseti(1, n + 1);
            }
            L.Stack.pushvalue(v);
            L.Stack.rawseti(1, pos);
            result = 0;
            return true;
        }
    }

    /// <summary>
    /// table.remove
    /// </summary>
    public class TableRemove : LuaFunction
    {
        public TableRemove(LuaReference globals)
            : base(globals)
        {
        }

        public override bool Execute(LuaState L, out int result)
        {
            int n = L.Stack.aux_getn(1);
            int pos = L.Stack.optint(2, n);
            if (n <= 0)
            {
                result = 0;
                return true;
            }
            L.Stack.aux_setn(1, n - 1);
            L.Stack.rawgeti(1, pos);
            for (; pos < n; pos++)
            {
                L.Stack.rawgeti(1, pos + 1);
                L.Stack.rawseti(1, pos);
            }
            L.Stack.pushnil();
            L.Stack.rawseti(1, n);
            result = 1;
            return true;
        }
    }
}
