/*
 * 
 * XNua 0.1 Based Massively on Lua2IL 0.5
 * Lua2IL 0.5.0
 * CoRoutines.cs - Coroutine support functions
 * Copyright 2003-2005 Fabio Mascarenhas
 * 
 */

using System;
using System.Collections;
using System.Threading;

namespace XNua
{
	public class Semaphore
	{
		private int    _rescount;
		private int    _count;
		private object _syncobj;
        private AutoResetEvent autoResetEvent;

		public Semaphore( int count )
		{
			if ( count <= 0 )
				throw new ArgumentOutOfRangeException( "count", "Must be greater zero" );
			_syncobj = new object();
			_rescount = _count = count;
            autoResetEvent = new AutoResetEvent(false);
		}

		public void Acquire()
		{
			lock( _syncobj )
			{
                while (_count == 0)
                {
                    Monitor.Exit(_syncobj);
                    autoResetEvent.WaitOne();
                    Monitor.Enter(_syncobj);
                }
				--_count;
			}
		}

		public void Release()
		{
			lock( _syncobj )
			{
				if ( _count == _rescount )
					throw new ApplicationException(
						"Exceeds Semaphore limit" );
				++_count;
				Monitor.Exit( _syncobj );
                autoResetEvent.WaitOne();
                Monitor.Enter(_syncobj);
			}
		}
	}
	
	public class LuaCoroutine : LuaReference 
	{
		LuaFunction mainFunction;
		Thread coThread;
		bool dead,error;

		LuaState coState;

		public LuaCoroutine(LuaFunction func) 
		{
			mainFunction=func;
			dead=false;
			error=false;
			Semaphore resumeHandle=new Semaphore(1);
			resumeHandle.Acquire();
			Semaphore yieldHandle=new Semaphore(1);
			yieldHandle.Acquire();
			coState=new LuaState();
			coState.Stack.Values[coState.Stack.Top++].O=func;
			coState.ResumeHandle=resumeHandle;
			coState.YieldHandle=yieldHandle;
			coThread=new Thread(new ThreadStart(this.Start));
			coThread.Start();
		}

		public void Start() 
		{
			coState.ResumeHandle.Acquire();
			try 
			{
				int nRes=mainFunction.Call(coState,-1,1);
			} 
			catch(RuntimeException e) 
			{
				error=true;
				coState.Stack.Values[coState.Stack.Base].O=new LuaObjectWrapper(e);
				coState.Stack.Top=coState.Stack.Base+1;
			}
			dead=true;
			coState.YieldHandle.Release();
		}

		public int Resume(LuaState L) 
		{
			if(!dead) 
			{
				int nArgs=L.Stack.Top-L.Stack.Base-1;
				L.Stack.Top-=nArgs;
				for(int i=0;i<nArgs;i++) 
				{
					coState.Stack.Values[coState.Stack.Top]=
						L.Stack.Values[L.Stack.Top+i];
					coState.Stack.Top++;
				}
				coState.ResumeArgs = nArgs;
				coState.ResumeHandle.Release();
				coState.YieldHandle.Acquire();
				int nRes=coState.Stack.Top-coState.Stack.Base;
				L.Stack.Check(nRes);
				coState.Stack.Top-=nRes;
				L.Stack.Values[L.Stack.Top++].O=(error ? (LuaReference)FalseClass.Instance : (LuaReference)TrueClass.Instance);
				for(int i=0;i<nRes;i++) 
				{
					L.Stack.Values[L.Stack.Top++]=
						coState.Stack.Values[coState.Stack.Top+i];
				}
				return nRes+1;
			} 
			else 
			{
				L.Stack.Values[L.Stack.Top++].O=FalseClass.Instance;
				L.Stack.Values[L.Stack.Top++].O=new LuaObjectWrapper(new Exception("cannot resume dead coroutine"));
				return 2;
			}
		}

		public static int Yield(LuaState co) 
		{
			co.YieldHandle.Release();
			co.ResumeHandle.Acquire();
			return co.ResumeArgs;
		}
	}

	public class CoCreate : LuaFunction 
	{
		public CoCreate(LuaReference globals) : base(globals)
		{
		}

		public override bool Execute(LuaState L, out int result) 
		{
			LuaFunction func=(LuaFunction)L.Stack.Values[L.Stack.Base].O;
			LuaCoroutine co=new LuaCoroutine(func);
			L.Stack.Values[L.Stack.Top++].O=co;
            result = 1;
			return true;
		}
	}

	public class CoResume : LuaFunction 
	{
		public CoResume(LuaReference globals) : base(globals)
		{
		}

		public override bool Execute(LuaState L, out int result) 
		{
			LuaCoroutine co=(LuaCoroutine)L.Stack.Values[L.Stack.Base].O;
			result = co.Resume(L);
            return true;
		}
	}

	public class CoYield : LuaFunction 
	{
		public CoYield(LuaReference globals) : base(globals)
		{
		}

		public override bool Execute(LuaState L,out int result) 
		{
			result = LuaCoroutine.Yield(L);
            return true;
		}
	}

	public class CoWrapper : LuaFunction 
	{
		LuaCoroutine coWrapped;

		public CoWrapper(LuaReference globals,LuaCoroutine coWrapped) : base(globals)
		{
			this.coWrapped=coWrapped;
		}

		public override bool Execute(LuaState L, out int result) 
		{
			int retVals=coWrapped.Resume(L);
            if (L.Stack.Values[L.Stack.Top - retVals].O == TrueClass.Instance)
            {
                result = retVals - 1;
                return true;
            }
            else
            {
                //throw (Exception)L.Stack.Values[L.Stack.Top - 1].CLRObject;
                result = -1;
                return false;
            }
		}
	}

	public class CoWrap : LuaFunction 
	{
		public CoWrap(LuaReference globals) : base(globals)
		{
		}

		public override bool Execute(LuaState L, out int result) 
		{
			LuaFunction func=(LuaFunction)L.Stack.Values[L.Stack.Base].O;
			LuaCoroutine co=new LuaCoroutine(func);
			L.Stack.Values[L.Stack.Top++].O=new CoWrapper(globals,co);
            result = 1;
            return true;
		}
	}

}