using System;
using System.Drawing;
using System.Diagnostics;

namespace nft.framework.drawing {
    /// <summary>
    /// Available view scales.
    /// values must be able to convert zero based sequencial indices.
    /// </summary>
    public enum ZoomScale : int { xQuarter = -2, xHalf = -1, x1 =0, x2 = 1 };

    public abstract class Scaler {
        private static Scaler[] arrScaler;
        private static int lbounds = 0;
        static Scaler() {
            Array a = Enum.GetValues(typeof(ZoomScale));
            foreach(ZoomScale zs in a){
                int v = (int)zs;
                if(lbounds>v) lbounds = v;
            }
            arrScaler = new Scaler[a.Length];
            Set(ZoomScale.xQuarter, new ShiftDownScaler(0,2));
            Set(ZoomScale.xHalf, new ShiftDownScaler(1,1));
            Set(ZoomScale.x1, new NullScaler(2));
            Set(ZoomScale.x2, new ShiftUpScaler(3,1));
        }
        /// <summary>
        /// Get suitable Scaler instance for each ZoomScale.
        /// Several instance can be used to provide fastest caluculation.
        /// </summary>
        /// <param name="zs"></param>
        /// <returns></returns>
        public static Scaler Get(ZoomScale zs) {
            return arrScaler[(int)zs - lbounds];
        }

        protected static Scaler Get(ZoomScale zs, int offset) {
            int idx = (int)zs - lbounds;
            if (offset > 0) {
                return arrScaler[Math.Min(arrScaler.Length-1, idx+offset)];
            } else {
                return arrScaler[Math.Max(0, idx + offset)];
            }
        }

        private static void Set(ZoomScale zs, Scaler sc) {
            arrScaler[(int)zs - lbounds] = sc;
        }

        /// <summary>
        /// Get Scaler of twice higher scale than specified ZoomScale.
        /// </summary>
        /// <param name="zs"></param>
        /// <returns></returns>
        public static Scaler GetDoubled(ZoomScale zs) {
            return Get(zs,1);
        }

        public static ZoomScale ZoomOut(ZoomScale zs) {
            return (ZoomScale)Math.Max(lbounds,(int)zs-1);
        }

        public static ZoomScale ZoomIn(ZoomScale zs) {
            return (ZoomScale)Math.Min(arrScaler.Length+lbounds-1, (int)zs + 1);
        }

        /// <summary>
        /// Default scale = no modification.
        /// </summary>
        public static Scaler Default { get { return Get(ZoomScale.x1); } }
        /// <summary>
        /// Scaler of highest zooming scale.
        /// </summary>
        public static Scaler Highest { get { return Get(ZoomScale.x2); } }
        /// <summary>
        /// Scaler of lowest(widest) zooming scale.
        /// </summary>
        public static Scaler Lowest { get { return Get(ZoomScale.xQuarter); } }

        protected Scaler(int index) {
            this.index = index;
        }
        private int index;
        /// <summary>
        /// Zero based index of ZoomScale as enum element.
        /// </summary>
        /// <param name="index"></param>
        public int Index { get { return index; } }

        public abstract double Value{ get; }        
        public abstract int Scale(int v);
        public abstract double Scale(double v);
        public abstract Size Scale(Size sz);
        public abstract void Scale(ref Size sz);
        public abstract Point Scale(Point pt);
        public abstract void Scale(ref Point pt);
        public abstract Point[] Scale(Point[] pts);
        public abstract void Scale(ref Point[] pts);
        public abstract Rectangle Scale(Rectangle rct);
        public abstract void Scale(ref Rectangle rct);
    }

    /// <summary>
    /// Use left bit-shift
    /// </summary>
    sealed class ShiftUpScaler : Scaler {
        private int s;
        internal ShiftUpScaler(int idx, int shift) : base(idx){
            this.s = shift;
        }
        public override double Value {
            get { return (double)(1<<s); }
        }

        public override int Scale(int v) {
            return v << s;
        }

        public override double Scale(double v) {
            return v * (1 << s);
        }

        public override Size Scale(Size sz) {
            sz.Width <<= s;
            sz.Height <<= s;
            return sz;
        }

        public override void Scale(ref Size sz) {
            sz.Width <<= s;
            sz.Height <<= s;
        }

        public override Point Scale(Point pt) {
            pt.X <<= s;
            pt.Y <<= s;
            return pt;
        }

        public override void Scale(ref Point pt) {
            pt.X <<= s;
            pt.Y <<= s;
        }

        public override Point[] Scale(Point[] pts) {
            Point[] ret = new Point[pts.Length];
            for (int i = 0; i < pts.Length; i++) {
                ret[i] = new Point(pts[i].X << s, pts[i].Y << s);
            }
            return ret;
        }

        public override void Scale(ref Point[] pts) {
            for (int i = 0; i < pts.Length; i++) {
                pts[i].X <<= s;
                pts[i].Y <<= s;
            }
        }

        public override Rectangle Scale(Rectangle rct) {
            rct.X <<= s;
            rct.Y <<= s;
            rct.Width <<= s;
            rct.Height <<= s;
            return rct;
        }

        public override void Scale(ref Rectangle rct) {
            rct.X <<= s;
            rct.Y <<= s;
            rct.Width <<= s;
            rct.Height <<= s;
        }
    }

    /// <summary>
    /// use right bit-shift
    /// </summary>
    sealed class ShiftDownScaler : Scaler {
        private int s;
        internal ShiftDownScaler(int idx, int shift)
            : base(idx) {
            this.s = shift;
        }

        public override double Value {
            get { return 1.0/(1<<s); }
        }

        public override int Scale(int v) {
            return v >> s;
        }

        public override double Scale(double v) {
            return v * (1 >> s);
        }

        public override Size Scale(Size sz) {
            sz.Width >>= s;
            sz.Height >>= s;
            return sz;
        }

        public override void Scale(ref Size sz) {
            sz.Width >>= s;
            sz.Height >>= s;
        }

        public override Point Scale(Point pt) {
            pt.X >>= s;
            pt.Y >>= s;
            return pt;
        }

        public override void Scale(ref Point pt) {
            pt.X >>= s;
            pt.Y >>= s;
        }

        public override Point[] Scale(Point[] pts) {
            Point[] ret = new Point[pts.Length];
            for (int i = 0; i < pts.Length; i++) {
                ret[i] = new Point(pts[i].X >> s, pts[i].Y >> s);
            }
            return ret;
        }

        public override void Scale(ref Point[] pts) {
            for (int i = 0; i < pts.Length; i++) {
                pts[i].X >>= s;
                pts[i].Y >>= s;
            }
        }

        public override Rectangle Scale(Rectangle rct) {
            rct.X >>= s;
            rct.Y >>= s;
            rct.Width >>= s;
            rct.Height >>= s;
            return rct;
        }

        public override void Scale(ref Rectangle rct) {
            rct.X >>= s;
            rct.Y >>= s;
            rct.Width >>= s;
            rct.Height >>= s;
        }
    }

    /// <summary>
    /// Do nothing
    /// </summary>
    sealed class NullScaler : Scaler {
        internal NullScaler(int idx) : base(idx){}

        public override double Value {
            get { return 1.0; }
        }

        public override int Scale(int v) {
            return v;
        }

        public override double Scale(double v) {
            return v;
        }

        public override Size Scale(Size sz) {
            return sz;
        }

        public override void Scale(ref Size sz) {
        }

        public override Point Scale(Point pt) {
            return pt;
        }

        public override void Scale(ref Point pt) {
            throw new NotImplementedException();
        }

        public override Point[] Scale(Point[] pts) {
            return pts;
        }

        public override void Scale(ref Point[] pts) {
        }

        public override Rectangle Scale(Rectangle rct) {
            return rct;
        }

        public override void Scale(ref Rectangle rct) {
        }
    }

    /// <summary>
    /// Scale by arbitral double value
    /// xd
    /// </summary>
    sealed class DoubleScaler : Scaler {
        private double val;
        DoubleScaler(int idx, double v)  : base(idx){
            this.val = v;
        }

        public override double Value {
            get { return val; }
        }

        public override Size Scale(Size sz) {
            return new Size((int)(sz.Width * val), (int)(sz.Height * val));
        }

        public override void Scale(ref Size sz) {
            sz.Width = (int)(sz.Width * val);
            sz.Height = (int)(sz.Height * val);
        }

        public override Point Scale(Point pt) {
            return new Point((int)(pt.X * val), (int)(pt.Y * val));
        }

        public override void Scale(ref Point pt) {
            pt.X = (int)(pt.X * val);
            pt.Y = (int)(pt.Y * val);
        }

        public override Point[] Scale(Point[] pts) {
            Point[] ret = new Point[pts.Length];
            for (int i = 0; i < ret.Length; i++) {
                Point pt = new Point((int)(pts[i].X * val), (int)(pts[i].Y * val));
                ret[i] = pt;
            }
            return ret;
        }

        public override void Scale(ref Point[] pts) {
            for (int i = 0; i < pts.Length; i++) {
                Point pt = pts[i];
                pt.X = (int)(pt.X * val);
                pt.Y = (int)(pt.Y * val);
            }
        }

        public override Rectangle Scale(Rectangle rct) {
            return new Rectangle((int)(rct.X * val), (int)(rct.Y * val), (int)(rct.Width * val), (int)(rct.Height * val));
        }

        public override void Scale(ref Rectangle rct) {
            rct.X = (int)(rct.X * val);
            rct.Y = (int)(rct.Y * val);
            rct.Width = (int)(rct.Width * val);
            rct.Height = (int)(rct.Height * val);
        }

        public override int Scale(int v) {
            return (int)(v * val);
        }

        public override double Scale(double v) {
            return v * val;
        }
    }
}
