/* 
 * FreeModbus Libary: A portable Modbus implementation for Modbus ASCII/RTU.
 * Copyright (c) 2006 Alexandr Litjagin (aka AlexRayne) <alexraynepe196@lavabit.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef _MBSTRUCT_h_
#define _MBSTRUCT_h_

/* ----------------------- Modbus includes ----------------------------------*/

#include <portgen.h>
#include <mbconfig.h>

/* ----------------------- Type definitions ---------------------------------*/

typedef enum 
{
    STATE_NOT_INITIALIZED
    , STATE_CLOSED = STATE_NOT_INITIALIZED
    , STATE_DISABLED
    , STATE_LISTEN      //!< tcp use this state when no connection established
    , STATE_ENABLED
    // modbus poller is in terminating state. when finishes last transaction
    // and come to ready state, status changes to STATE_CLOSED
    , STATE_TERMINATE 
} MBState_t;

typedef enum {
      MB_ST_IDLE        //generic idle
    , MB_ST_ACK         //ack send after receive slave
    , MBM_ST_SEND       //master send
    , MBM_ST_WAIT_ACK   //master wait ack
    } MB_protocol_state;

/*! \ingroup modbus
 * \brief Modbus serial transmission modes (RTU/ASCII).
 *
 * Modbus serial supports two transmission modes. Either ASCII or RTU. RTU
 * is faster but has more hardware requirements and requires a network with
 * a low jitter. ASCII is slower and more reliable on slower links (E.g. modems)
 */
    typedef enum
{
    MB_RTU,                     /*!< RTU transmission mode. */
    MB_ASCII,                   /*!< ASCII transmission mode. */
    MB_TCP                      /*!< TCP mode. */
} eMBMode;

/*! \ingroup modbus
 * \brief If register should be written or read.
 *
 * This value is passed to the callback functions which support either
 * reading or writing register values. Writing means that the application
 * registers should be updated and reading means that the modbus protocol
 * stack needs to know the current register values.
 *
 * \see eMBRegHoldingCB( ), eMBRegCoilsCB( ), eMBRegDiscreteCB( ) and 
 *   eMBRegInputCB( ).
 */
typedef enum
{
    MB_REG_READ,                /*!< Read register values and pass to protocol stack. */
    MB_REG_WRITE                /*!< Update register values. */
} eMBRegisterMode;

/*! \ingroup modbus
 * \brief Errorcodes used by all function in the protocol stack.
 */
typedef enum
{
    MB_ENOERR,                  /*!< no error. */
    MB_BUSY                     /*!< xfer in progress */
    , MB_CLOSED                 //!< Poller handled PORT_CLOSE request.
    , MB_ENOREG,                /*!< illegal register address. */
    MB_EINVAL,                  /*!< illegal argument. */
    MB_EPORTERR,                /*!< porting layer error. */
    MB_ENORES,                  /*!< insufficient resources. */
    MB_EIO,                     /*!< I/O error. */
    MB_EILLSTATE,               /*!< protocol stack in illegal state. */
    MB_ETIMEDOUT                /*!< timeout error occurred. */
    , MB_EACK_FAIL              //!< ack returned wrong adress
    , MB_EREPLY_FAIL            //!< reply ack returned error code
    , MB_EACK_TIMEOUT           //!< timed out for reply ack
    , MB_SEND_TIMEOUT           //!< timed out at sending, cant imagine why it can be
    , MB_EXCEPTION              //!< this is offset for arbitrary Modbus exception: errCode = MB_EXCEPTION + exceptionCode
} eMBErrorCode;

/*! \ingroup modbus
 * \brief this structure defines flags that reciver\transmiter setup for eMBPoll usage.
 */
typedef union {
    struct{
        //used by eMBMasterPoll to denote that Frame is receiveing now and therefore ACK wait 
        // timeout can be durated
        UCHAR       IsRecvFrameLoading  : 1;
    };
    UCHAR data;
} MBStatus;


#if ((MB_ASCII_ENABLED + MB_RTU_ENABLED + MB_TCP_ENABLED) >=2)
#   define MB_USE_FPTRCUR
#elif MB_ASCII_ENABLED > 0

#   define pvMBFrameStartCur  eMBASCIIStart
#   define pvMBFrameStopCur  eMBASCIIStop
#   define peMBFrameSendCur eMBASCIISend
#   define peMBFramePostCur eMBASCIISend
#   define peMBFrameReceiveCur eMBASCIIReceive
#   ifndef MB_PORT_HAS_CLOSE
#       define  pvMBFrameCloseCur NULL
#   else
#       define pvMBFrameCloseCur vMBPortClose
#   endif
#   define pxMBFrameCBByteReceived xMBASCIIReceiveFSM
#   define pxMBFrameCBTransmitterEmpty xMBASCIITransmitFSM
#   define pxMBPortCBTimerExpired xMBASCIITimerT1SExpired

#elif MB_RTU_ENABLED > 0

#   define pvMBFrameStartCur eMBRTUStart
#   define pvMBFrameStopCur eMBRTUStop
#   define peMBFrameSendCur eMBRTUSend
#   define peMBFramePostCur eMBRTUPost
#   define peMBFrameReceiveCur eMBRTUReceive
#   ifndef MB_PORT_HAS_CLOSE
#       define  pvMBFrameCloseCur NULL
#   else
#       define pvMBFrameCloseCur vMBPortClose
#   endif
#   define pxMBFrameCBByteReceived xMBRTUReceiveFSM
#   define pxMBFrameCBTransmitterEmpty xMBRTUTransmitFSM
#   define pxMBPortCBTimerExpired xMBRTUTimerT35Expired

#elif MB_TCP_ENABLED > 0
#   define pvMBFrameStartCur eMBTCPStart
#   define pvMBFrameStopCur eMBTCPStop
#   define peMBFrameReceiveCur eMBTCPReceive
#   define peMBFrameSendCur eMBTCPSend
#   define peMBFramePostCur eMBTCPSend
#   ifndef MB_PORT_HAS_CLOSE
#       define  pvMBFrameCloseCur NULL
#   else
#       define pvMBFrameCloseCur vMBTCPPortClose
#   endif

#else
#   error "what happen?"
#endif



#endif
