/* 
 * FreeModbus Libary: A portable Modbus implementation for Modbus ASCII/RTU.
 * Copyright (c) 2006 Christian Walter <wolti@sil.at>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ru UTF8 
 * 
 * File: $Id: mbport.h,v 1.19 2010/06/06 13:54:40 wolti Exp $
 */

#ifndef _MB_PORT_H
#define _MB_PORT_H

#include <portgen.h>
#include <mbstruct.h>

/** this macros provided by portgen.h and aided to provide accurate linking of code
 * */
// this routines are plased in faster memory region
#ifndef MB_CODE_FAST
#define MB_CODE_FAST
#endif

// this routines are used by ISR and plased in ISR`s memory region
#ifndef MB_CODE_ISR
#define MB_CODE_ISR
#endif

#ifdef __cplusplus
PR_BEGIN_EXTERN_C
#endif


/* ----------------------- Type definitions ---------------------------------*/

typedef enum
{
    EV_NONE = 0                 //!< no event, used by internals
    , EV_READY = 1,             /*!< Startup finished. */
    EV_FRAME_RECEIVED,          /*!< Frame received. */
    EV_EXECUTE,                 /*!< Execute function. */
    EV_FRAME_SENT               /*!< Frame sent. */
    //, EV_RECEIVER_TIMEOUT       //receiver timed out during master waits Acknowledge
    , EV_PROTOCOL_TIMEOUT       //!< timout event from master protocol external timer
    , EV_CLOSE                  //!< finish current transaction and close
    , EV_POLL_UNCKNOWN          //!< this events not processed by eMBPoll[Master]
} eMBEventType;

/*! \ingroup modbus
 * \brief Parity used for characters in serial mode.
 *
 * The parity which should be applied to the characters sent over the serial
 * link. Please note that this values are actually passed to the porting
 * layer and therefore not all parity modes might be available.
 */
typedef enum
{
    MB_PAR_NONE,                /*!< No parity. */
    MB_PAR_ODD,                 /*!< Odd parity. */
    MB_PAR_EVEN                 /*!< Even parity. */
} eMBParity;



#include <mbmulty.h>

/** ENTER/EXIT_CRITICAL_SECTION( )  must be application defined and must 
 * protects access to port ISR context - in this critical code ISR must not be
 * executed 
 * */
//#define ENTER_CRITICAL_SECTION( )
//#define EXIT_CRITICAL_SECTION( )

/*! MODBUS RTU use this macro to define an size of internal receive buffer 
 * */
//#define MB_SER_RECV_SIZE 256

#define xMBPortEventInit()      MB_METHOD_ACT1(xMBPortEventInit)
#define xMBPortEventPost(...)   MB_METHOD_ACT(xMBPortEventPost, __VA_ARGS__)
#define xMBPortEventGet(...)    MB_METHOD_ACT(xMBPortEventGet, __VA_ARGS__)
#define xMBPortSerialInit(...)  MB_METHOD_ACT(xMBPortSerialInit, __VA_ARGS__)
#define vMBPortClose()          MB_METHOD_ACT1(vMBPortClose)
#define xMBPortSerialClose()    MB_METHOD_ACT1(xMBPortSerialClose)
#define vMBPortSerialEnable(...)    MB_METHOD_ACT(vMBPortSerialEnable, __VA_ARGS__)
#define xMBPortSerialGetByte(...)   MB_METHOD_ACT(xMBPortSerialGetByte, __VA_ARGS__)
#define xMBPortSerialPutByte(...)   MB_METHOD_ACT(xMBPortSerialPutByte, __VA_ARGS__)

/* ----------------------- Supporting functions -----------------------------*/

BOOL            MB_METHOD_DECL1(xMBPortEventInit);

BOOL            MB_METHOD_DECL(xMBPortEventPost, eMBEventType eEvent );

BOOL            MB_METHOD_DECL(xMBPortEventGet,  /*@out@ */ eMBEventType * eEvent );

/* ----------------------- Serial port functions ----------------------------*/

BOOL            MB_METHOD_DECL(xMBPortSerialInit, UCHAR ucPort, ULONG ulBaudRate,
                                   UCHAR ucDataBits, eMBParity eParity );

/** MB_PORT_HAS_CLOSE should be declared in portgen.h 
 * */
//#define MB_PORT_HAS_CLOSE

void            MB_METHOD_DECL1(vMBPortClose);

void            MB_METHOD_DECL1(xMBPortSerialClose);

/* this macro declares that transmiter is not responded on output beffer empty,
 * therefore port direction control must check that port is not busy by transmit
 * */
//#define MB_CONTROL_BY_LASY_TXBUF

//transmitter relyes on buffer empty events, so to disable TX port need wait until transmiter
//  flushes output register. therefore vMBPortSerialEnable can return FALSE when tx busy
//\return TRUE  - ok
//\return FALSE - can`t change satate due port busy, try later
#ifdef MB_CONTROL_BY_LASY_TXBUF
typedef BOOL MB_SEResult;
#else
//this is obsolete! need only for old code compatible
typedef void MB_SEResult;
#endif

MB_SEResult     MB_METHOD_DECL(vMBPortSerialEnable, BOOL xRxEnable, BOOL xTxEnable );

BOOL            MB_METHOD_DECL(xMBPortSerialGetByte, UCHAR * pucByte );

BOOL            MB_METHOD_DECL(xMBPortSerialPutByte, UCHAR ucByte );

#ifdef UFRAMESIZE
/**\~russian
 *  это макро включает использование блочной отсылки буффера вместо посимвольной
 *  через xMBPortSerialPutByte
 * \~english
 *  this macro enables use block output xMBPortSerialPutBytes vs per-char xMBPortSerialPutByte
 *  declare it in port.h
 * */
#define MB_USE_TXFIFO

/**\~russian
 * \seealso MB_USE_TXFIFO
 * 
 * \param[in] len Длина буффера для отправки
 * \return        количество актуально отправленых символов из буффера
 * 
 * \~english
 * \param[in] len ucBytes buffer len to send
 * \return        actual sended symbols from buffer
 * */
#define xMBPortSerialPutBytes(...)   MB_METHOD_ACT(xMBPortSerialPutBytes, __VA_ARGS__)
UFRAMESIZE      MB_METHOD_DECL(xMBPortSerialPutBytes, UCHAR* ucBytes, UFRAMESIZE len);
#endif

#ifdef MB_HAVE_XGETBYTE
//* same as xMBPortSerialGetByte, also returns uart error status
//* \param[out] puByte - low byte - received char, 
//*                       high byte - error info, 0 - if no errors
#define xMBPortSerialGetByteX(...)   MB_METHOD_ACT(xMBPortSerialGetByteX, __VA_ARGS__)
BOOL            MB_METHOD_DECL(xMBPortSerialGetByteX, unsigned * puByte );
#endif

/* ----------------------- Timers functions ---------------------------------*/
#define xMBPortTimersInit(...)  MB_METHOD_ACT(xMBPortTimersInit, __VA_ARGS__)
#define xMBPortTimersClose()    MB_METHOD_ACT1(xMBPortTimersClose)
#define vMBPortTimersEnable()   MB_METHOD_ACT1(vMBPortTimersEnable)
#define vMBPortTimersDisable()  MB_METHOD_ACT1(vMBPortTimersDisable)
#define vMBPortTimersDelay(...) MB_METHOD_ACT(vMBPortTimersDelay, __VA_ARGS__)

BOOL            MB_METHOD_DECL(xMBPortTimersInit, USHORT usTimeOut50us );

void            MB_METHOD_DECL1(xMBPortTimersClose);

void            MB_METHOD_DECL1(vMBPortTimersEnable);

void            MB_METHOD_DECL1(vMBPortTimersDisable);

void            MB_METHOD_DECL(vMBPortTimersDelay, USHORT usTimeOutMS );

/** \~russian
 * modbus-master позволяет инициировать транзакцию отправки в течение 
 *  транзакции приема если декларирован MB_FLEX_TX
 *  иначе запрос отправки завершается ошибкой если идет прием фрейма
 *  \~english
 *  modbus-master can initiate send-request during receiving frame if MB_FLEX_TX
 *  declared, and starts sending after receive finished.
 *  else - sending request fails if receiving
 * */
//#define MB_FLEX_TX

//master uses this to timeout Acknowledge
#define MBProtocolTimeOut(...)      MB_METHOD_ACT(MBProtocolTimeOut, __VA_ARGS__)
#define MBProtocolTimeOutExpired()  MB_METHOD_ACT1(MBProtocolTimeOutExpired)
#define MBProtocolTimeOutOff()      MB_METHOD_ACT1(MBProtocolTimeOutOff)

void            MB_METHOD_DECL(MBProtocolTimeOut, USHORT usTimeOut50us );
BOOL            MB_METHOD_DECL1(MBProtocolTimeOutExpired);
void            MB_METHOD_DECL1(MBProtocolTimeOutOff);

/* ----------------------- Callback for the protocol stack ------------------*/

#ifndef MB_IS_MULTY

#ifdef MB_USE_FPTRCUR

/*!
 * \brief Callback function for the porting layer when a new byte is
 *   available.
 *
 * Depending upon the mode this callback function is used by the RTU or
 * ASCII transmission layers. In any case a call to xMBPortSerialGetByte()
 * must immediately return a new character.
 *
 * \return <code>TRUE</code> if a event was posted to the queue because
 *   a new byte was received. The port implementation should wake up the
 *   tasks which are currently blocked on the eventqueue.
 */
extern          BOOL( *pxMBFrameCBByteReceived ) ( void );

extern          BOOL( *pxMBFrameCBTransmitterEmpty ) ( void );

extern          BOOL( *pxMBPortCBTimerExpired ) ( void );

#endif  //MB_USE_FPTRCUR

#endif //!MB_IS_MULTY

/* ----------------------- TCP port functions -------------------------------*/
#define xMBTCPPortInit(...)         MB_METHOD_ACT(xMBTCPPortInit, __VA_ARGS__)
#define vMBTCPPortClose()           MB_METHOD_ACT1(vMBTCPPortClose)
#define vMBTCPPortDisable()         MB_METHOD_ACT1(vMBTCPPortDisable)
#define xMBTCPPortGetRequest(...)   MB_METHOD_ACT(xMBTCPPortGetRequest, __VA_ARGS__)
#define xMBTCPPortSendResponse(...) MB_METHOD_ACT(xMBTCPPortSendResponse, __VA_ARGS__)

BOOL            MB_METHOD_DECL(xMBTCPPortInit, USHORT usTCPPort );

void            MB_METHOD_DECL1(vMBTCPPortClose);

void            MB_METHOD_DECL1(vMBTCPPortDisable);

BOOL            MB_METHOD_DECL(xMBTCPPortGetRequest, UCHAR **ppucMBTCPFrame, UFRAMESIZE * usTCPLength );

BOOL            MB_METHOD_DECL(xMBTCPPortSendResponse, const UCHAR *pucMBTCPFrame, UFRAMESIZE usTCPLength );

//*-----------------------  debug wrapers  ------------------------------------
/* this macro should be defined 
#ifdef DEBUG_MB

#ifndef MB_printf
#define MB_printf(...)
#endif

#ifndef MB_puts
#define MB_puts(s)
#endif

#ifndef MB_putc
#define MB_putc(c)
#endif

#else
#define MB_printf(...)
#define MB_puts(s)
#define MB_putc(c)
#endif
*/

#ifdef __cplusplus
PR_END_EXTERN_C
#endif
#endif
