/* 
 * FreeModbus Libary: A portable Modbus implementation for Modbus ASCII/RTU.
 * Copyright (c) 2006 Christian Walter <wolti@sil.at>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * File: $Id: mbframe.h,v 1.9 2006/12/07 22:10:34 wolti Exp $
 */

#ifndef _MB_FRAME_H
#define _MB_FRAME_H

//#include <mb.h>
#include <portgen.h>

#ifdef __cplusplus
PR_BEGIN_EXTERN_C
#endif

/*!
 * Constants which defines the format of a modbus frame. The example is
 * shown for a Modbus RTU/ASCII frame. Note that the Modbus PDU is not
 * dependent on the underlying transport.
 *
 * <code>
 * <------------------------ MODBUS SERIAL LINE PDU (1) ------------------->
 *              <----------- MODBUS PDU (1') ---------------->
 *  +-----------+---------------+----------------------------+-------------+
 *  | Address   | Function Code | Data                       | CRC/LRC     |
 *  +-----------+---------------+----------------------------+-------------+
 *  |           |               |                                   |
 * (2)        (3/2')           (3')                                (4)
 *
 * (1)  ... MB_SER_PDU_SIZE_MAX = 256
 * (2)  ... MB_SER_PDU_ADDR_OFF = 0
 * (3)  ... MB_SER_PDU_PDU_OFF  = 1
 * (4)  ... MB_SER_PDU_SIZE_CRC = 2
 *
 * (1') ... MB_PDU_SIZE_MAX     = 253
 * (2') ... MB_PDU_FUNC_OFF     = 0
 * (3') ... MB_PDU_DATA_OFF     = 1
 * </code>
 */

/* ----------------------- Defines ------------------------------------------*/
#define MB_PDU_SIZE_MAX     253 /*!< Maximum size of a PDU. */
#define MB_PDU_SIZE_MIN     1   /*!< Function Code */
#define MB_PDU_FUNC_OFF     0   /*!< Offset of function code in PDU. */
#define MB_PDU_DATA_OFF     1   /*!< Offset for response data in PDU. */

/* ----------------------- Prototypes  0-------------------------------------*/
typedef void    ( *pvMBFrameStart ) ( MB_PORT );

typedef void    ( *pvMBFrameStop ) ( MB_PORT );

typedef eMBErrorCode( *peMBFrameReceive ) (MB_WITH_PORT UCHAR * pucRcvAddress,
                                            UCHAR ** pucFrame,
                                            UFRAMESIZE * pusLength );

typedef eMBErrorCode( *peMBFrameSend ) (MB_WITH_PORT UCHAR slaveAddress,
                                         const UCHAR * pucFrame,
                                         UFRAMESIZE usLength );

typedef void( *pvMBFrameClose ) ( MB_PORT );

//modbus uses BIG ENDIAN model - most byte sent first
typedef USHORT MBWord;

typedef struct{
    MBWord adress;
    MBWord count;
    } __attribute__((__packed__)) MB_rd_holdregs_req;

typedef struct{
    UCHAR datalen;
//    MBWord data[1];
    } __attribute__((__packed__)) MB_rd_holdregs_ack;

typedef struct{
    MBWord adress;
    MBWord count;
    UCHAR datalen;
//    MBWord data[1];
    } __attribute__((__packed__)) MB_wr_holdregs_req;

typedef struct{
    MBWord adress;
    MBWord count;
//    MBWord data[1];
    } __attribute__((__packed__)) MB_wr_holdregs_ack;

typedef struct {
    UCHAR function;
    union {
        struct {
            UCHAR                   excode;
        } exception_ack;
        struct {
            MB_rd_holdregs_req      req;
            MBWord                  crc;
        } rd_holdregs_req;
        struct {
            MB_rd_holdregs_ack      ack;
            UCHAR                   data[1];
        } rd_holdregs_ack;
        struct {
            MB_wr_holdregs_req      req;
            UCHAR                   data[1];
        } wr_holdregs_req;
        struct {
            MB_wr_holdregs_ack      ack;
            MBWord                  crc;
        } wr_holdregs_ack;
    };
} __attribute__((__packed__)) MB_frame_PDU;

typedef struct {
    UCHAR adress;
    union {
        MB_frame_PDU PDU;
        unsigned char data[MB_PDU_SIZE_MAX];
    };
} __attribute__((__packed__)) mbFrameMaxBuffer;

#ifndef MB_SER_RECV_SIZE
typedef mbFrameMaxBuffer mbFrameUseBuffer;
#define MB_rd_holdregs_data_limit (MB_PDU_SIZE_MAX - 4 - sizeof(MB_rd_holdregs_ack))
#define MB_wr_holdregs_data_limit (MB_PDU_SIZE_MAX - 4 - sizeof(MB_wr_holdregs_req))

#else
typedef struct {
    UCHAR adress;
    union {
        MB_frame_PDU PDU;
        unsigned char data[MB_SER_RECV_SIZE];
    };
} __attribute__((__packed__)) mbFrameUseBuffer;

#define MB_rd_holdregs_data_limit (MB_SER_RECV_SIZE - 4 - sizeof(MB_rd_holdregs_ack))
#define MB_wr_holdregs_data_limit (MB_SER_RECV_SIZE - 4 - sizeof(MB_wr_holdregs_req))
#endif




#include <mbutils.h>

#define MB_FILL_RD_HORLD_REQ(buf, target, regadress, regcount) (\
    buf.adress = target,\
    buf.PDU.function = MB_FUNC_READ_HOLDING_REGISTER,\
    buf.PDU.rd_holdregs_req.req.adress = AsMBNetWord(regadress),\
    buf.PDU.rd_holdregs_req.req.count = AsMBNetWord(regcount),\
    sizeof(buf.PDU.rd_holdregs_req.req) + sizeof(buf.PDU.function)\
    )

UCHAR MB_CALC_WR_HORLD_REQSIZE(MB_frame_PDU * target, UCHAR cursize);


/* ----------------------- MBAP Header --------------------------------------*/
/*
 *
 * <------------------------ MODBUS TCP/IP ADU(1) ------------------------->
 *              <----------- MODBUS PDU (1') ---------------->
 *  +-----------+---------------+------------------------------------------+
 *  | TID | PID | Length | UID  |Code | Data                               |
 *  +-----------+---------------+------------------------------------------+
 *  |     |     |        |      |
 * (2)   (3)   (4)      (5)    (6)
 *
 * (2)  ... MB_TCP_TID          = 0 (Transaction Identifier - 2 Byte)
 * (3)  ... MB_TCP_PID          = 2 (Protocol Identifier - 2 Byte)
 * (4)  ... MB_TCP_LEN          = 4 (Number of bytes - 2 Byte)
 * (5)  ... MB_TCP_UID          = 6 (Unit Identifier - 1 Byte)
 * (6)  ... MB_TCP_FUNC         = 7 (Modbus Function Code)
 *
 * (1)  ... Modbus TCP/IP Application Data Unit
 * (1') ... Modbus Protocol Data Unit
 */

#define MB_TCP_TID          0
#define MB_TCP_PID          2
#define MB_TCP_LEN          4
#define MB_TCP_UID          6
#define MB_TCP_FUNC         7

/* ----------------------- Defines  -----------------------------------------*/
#define MB_TCP_DEFAULT_PORT 502 /* TCP listening port. */
#define MB_TCP_BUF_SIZE     ( 256 + 7 ) /* Must hold a complete Modbus TCP frame. */
#define MB_TCP_PROTOCOL_ID  0   /* 0 = Modbus Protocol */

typedef struct{
    MBWord TransactionID;
    MBWord ProtocolID;
    MBWord Len;
    } __attribute__((__packed__)) MB_TCP_header;

typedef struct{
    MB_TCP_header       htcp;
    UCHAR  adress;//UnitID;
    UCHAR  function;
    } __attribute__((__packed__)) MBAP_header;

typedef struct {
    MB_TCP_header       htcp;
    mbFrameMaxBuffer    ADU;
} __attribute__((__packed__)) mbtcpFrameBuffer;




#ifdef __cplusplus
PR_END_EXTERN_C
#endif
#endif
