/*
 * FreeModbus Libary: lwIP Port
 * Copyright (C) 2006 Christian Walter <wolti@sil.at>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * File: $Id: porttcp.c,v 1.1 2006/08/30 23:18:07 wolti Exp $
 */

/* ----------------------- System includes ----------------------------------*/
#include <runtime/lib.h>
#include <kernel/uos.h>
#include <mem/mem.h>

#include <port.h>

/* ----------------------- lwIP includes ------------------------------------*/
#include <net/tcp.h>

/* ----------------------- Modbus includes ----------------------------------*/
#include "mb.h"
#include "mbport.h"
#include <mbframe.h>
#include <portevent.h>
#include <mbutils.h>
#include <mbmulty.h>

/* ----------------------- Prototypes ---------------------------------------*/
//void            vMBPortEventClose( void );

#ifdef MB_TCP_DEBUG
#define MBPortLog(verb, ...)  DEBUG_MBprintf(verb, __VA_ARGS__)
#else
#define MBPortLog(...)
#endif

#ifndef MB_TCP_CONN_LIMIT
#define MB_TCP_CONN_LIMIT   1
#endif

/* ----------------------- Static variables ---------------------------------*/
typedef unsigned    MBTCP_client_id;
struct mb_tcp_s {
        mb_nif  mb;
        tcp_socket_t*   listen;
        mutex_irq_t     listen_irq;
        tcp_socket_t*   client[MB_TCP_CONN_LIMIT];
        mutex_irq_t     client_irq[MB_TCP_CONN_LIMIT];

        mutex_t         post_signal;    //signal receives on qEventPost  
        MUTEX_GROUP(2/*listner + post*/ +MB_TCP_CONN_LIMIT)  signals_group;

        //bit-mask for ready to read listner sockets
        volatile unsigned        serve_signals;
        //bit-mask for ready to read sockets, that was signaled by Modbus event
        volatile unsigned        ready_signals;
        struct {
            //* текущий клиент освободим его и поищем новый запрос
            MBTCP_client_id id;
            tcp_socket_t*   client;  //client sock for wich processed poll
            //* текущий буфер  - загружается по запросу чтения,
            //* каждый запрос чтения или записи, освобождает текущий буфер
            buf_t*          buf;
            UFRAMESIZE      FrameLen;
            UCHAR           FrameTmp[MB_TCP_BUF_SIZE];
        } target;
};
typedef struct mb_tcp_s mb_tcp_self;

mb_tcp_self  mb_tcp_io;
#define MB_CTX      &mb_tcp_io

#if MB_TCP_CONN_LIMIT > 1
#define MB_CLIENT(n)    MB_NET->client[n]
#else
#define MB_CLIENT(n)    MB_NET->client[0]
#endif

#ifdef MB_IS_MULTY
#define MB_PORT_TCP_DECL MB_PORT_NET_DECLAS(mb_tcp_self*)
#else
#define MB_PORT_TCP_DECL mb_tcp_self* MB_NET = MB_CTX
#endif

typedef enum {
      EV_LISTEN   = EV_POLL_UNCKNOWN
    , EV_CLIENT
    , EV_CLIENT_LAST = EV_CLIENT + MB_TCP_CONN_LIMIT-1
} eMBTCPEventType;

/* ----------------------- Static functions ---------------------------------*/
static bool_t   mb_OnTCPPortAccept(void *pvArg);
static bool_t   mb_OnTCPPortReceive(void *pvArg);
//static void   MB_METHOD_DECL(TCPPortError, eMBTCPEventType client_id, unsigned xErr);

/* ----------------------- Begin implementation -----------------------------*/

eMBEventType  MB_METHOD_DECL(tcpOnNoPollEvent, eMBEventType event);

BOOL
MB_METHOD_DECL(xMBTCPPortInit, USHORT usTCPPort )
{
    MB_SELF->MB_NET = MB_CTX;
    MB_PORT_TCP_DECL;
    mb_tcp_self* self = MB_NET;
    
    BOOL            bOkay = FALSE;
    USHORT          usPort;

    memset(self, 0, sizeof(*self));
    MB_SELF->MB_NET = self;

    eQueuedEvent.OnNoPollEvent = MB_METHOD_NAME(tcpOnNoPollEvent);
    mutex_group_t *mb_signals;
    mb_signals = mutex_group_init (self->signals_group.data, sizeof(self->signals_group));

    if( usTCPPort == 0 )
        usPort = MB_TCP_DEFAULT_PORT;
    else
        usPort = ( USHORT ) usTCPPort;

    ip_addr ip_any = {0};

    self->listen = tcp_listen(&ip, ip_any.ucs, usPort);

    if( self->listen == NULL )
    {
        tcp_close( self->listen );
        bOkay = FALSE;
    }
    else{
        mutex_attach_swi(&self->listen->lock, &self->listen_irq, mb_OnTCPPortAccept, MB_SELF);
        mutex_group_add (mb_signals, &(self->post_signal));
        mutex_group_add (mb_signals, &(self->listen->lock));

        mutex_group_listen (mb_signals);
        eQueuedEvent.signals_group = mb_signals;
        MBPortLog(vMB_note, "MBTCP: Protocol stack ready on port %d (sock $%p)\n"
                        , usPort, self->listen);
    }
    bOkay = TRUE;
    return bOkay;
}

static bool_t   MB_METHOD_DECL1(TCPPortAccept);

tcp_socket_t* MB_METHOD_DECL1(tcpSelectClient);
#define tcpSelectClient() MB_METHOD_ACT1(tcpSelectClient)

eMBEventType  MB_METHOD_DECL(tcpOnNoPollEvent, eMBEventType event)
{
    MB_PORT_TCP_DECL;
    if ((int)event == (int)EV_LISTEN){
        // socket listen have signaled, so process it
        if (MB_METHOD_ACT1(TCPPortAccept))
            event = EV_CLIENT;
        else
            event = EV_NONE;
    }
    while ((int)event == (int)EV_CLIENT){
        if ((MB_NET->target.buf != 0) && (MBHasEvent()))
            //уже обрабатывается буфер, подождем пока он будет освобожден
            //  прежде чем начать обратботку нового запроса
            //  полагаемся на то что ТСПфреймы всегда обрабатываются, и всегда отсылаются ответы
            break;
        tcp_socket_t* s = tcpSelectClient();
        MBPortLog(vMB_debug, "MBTCP: have received on sock $%p\n", s);
        if (s != 0)
            return EV_FRAME_RECEIVED;
        MB_NET->ready_signals = 0;
        break;
    }
    return event;
}


#define tcpReleaseClient(...) MB_METHOD_ACT(tcpReleaseClient, __VA_ARGS__ )
void MB_METHOD_DECL(tcpReleaseClient, eMBTCPEventType client_id );

#define tcpReleaseClientAll() MB_METHOD_ACT1(tcpReleaseClientAll)
void MB_METHOD_DECL1(tcpReleaseClientAll)
{
    /* Shutdown any open client sockets. */
    int cid = EV_CLIENT;
    for (cid = EV_CLIENT; cid <= EV_CLIENT_LAST; cid++)
        tcpReleaseClient(cid);
}

void MB_METHOD_DECL1(vMBTCPPortClose)
{
    MBPortLog(vMB_info, "MBTCP: Close port\n");
    tcpReleaseClientAll();

    /* Shutdown or listening socket. */
    tcpReleaseClient( EV_LISTEN );

    /* Release resources for the event queue. */
    vMBPortEventClose(  );
}

void MB_METHOD_DECL1(vMBTCPPortDisable)
{
    tcpReleaseClientAll();
}




INLINE
bool_t MB_METHOD_DECL(OnTCPPort, tcp_socket_t* s, eMBTCPEventType post_ev)
{
    MB_PORT_TCP_DECL;
    assert(s != 0);
    if(tcp_socket_is_online(s))
    if(tcp_socket_is_avail(s)){
        MBPortLog(vMB_debug, "MBTCP: LISTEN sock $%p\n", s);
        MB_NET->serve_signals = 1;
        return xMBPortEventPost(EV_LISTEN);
    }
    return 0;
}

static bool_t   mb_OnTCPPortAccept( void *Arg)
{
    MB_putc('L');
    MB_PORT_DECL_FROM(Arg);
    MB_PORT_TCP_DECL;
    if (MB_NET->serve_signals > 0)
        //alredy signaled to MB
        return 0;
    tcp_socket_t* s = MB_NET->listen;
    MB_METHOD_ACT(OnTCPPort, s, EV_LISTEN);
    return 0;
}

static bool_t   mb_OnTCPPortReceive( void *Arg)
{
    MB_putc('S');
    MB_PORT_DECL_FROM(Arg);
    MB_PORT_TCP_DECL;
    if (MB_NET->ready_signals > 0)
        //alredy signaled to MB
        return 0;

    MB_NET->ready_signals++;
    xMBPortEventPost(EV_CLIENT);
    return 0;
}

#define TCPPortAllocClientId() MB_METHOD_ACT1(TCPPortAllocClientId)
MBTCP_client_id MB_METHOD_DECL1(TCPPortAllocClientId);

static bool_t   MB_METHOD_DECL1(TCPPortAccept)
{
    MB_PORT_TCP_DECL;
    tcp_socket_t* listen = MB_NET->listen;

     assert(listen != 0);
     if(!tcp_socket_is_online(listen))
         return 0;

     bool_t ok = 0;
     while(tcp_socket_is_avail(listen))
     {
         tcp_socket_t* c = tcp_accept(listen);
         if (c == 0)
             break;
         if (SEANYERROR(c)){
             MBPortLog(vMB_fail, "MBTCP:Accept failure $%x\n", c);
             break;
         }

         MBTCP_client_id id = TCPPortAllocClientId();
         if (id >= MB_TCP_CONN_LIMIT){
             MBPortLog(vMB_fail, "MBTCP:!!Rejected client %@.4@Du:%d\n"
                             , c->remote_ip.ucs, c->remote_port);
             tcp_close(c);
             mem_free(c);
             continue;
         }

         MB_CLIENT(id) = c;
         if (!mutex_group_add(&MB_NET->signals_group.g, &c->lock)){
             MBPortLog(vMB_fail, "MBTCP:!!Rejected client[%d] %@.4@Du:%d\n"
                             , id, c->remote_ip.ucs, c->remote_port);
             MB_CLIENT(id) = 0;
             tcp_close(c);
             mem_free(c);
             continue;
         }
         mutex_attach_swi(&(c->lock), &(MB_NET->client_irq[id]), &(mb_OnTCPPortReceive), MB_SELF);
         mutex_group_relisten(&MB_NET->signals_group.g);
         //потребую чтобы отсылаемые PDU не объединялись вместе, а отсылались 1PDU / 1ТСП-сегмент
         c->flags |= TF_NOCORK;  
         MBPortLog(vMB_info, "MBTCP:Accepted new client(%x) [%d] %@.4@Du:%d\n"
                             ,c , id, c->remote_ip.ucs, c->remote_port);
         ok = 1;
     }
     MB_NET->serve_signals = 0;
     return ok;
}


MBTCP_client_id MB_METHOD_DECL1(TCPPortAllocClientId){
    MB_PORT_TCP_DECL;
    /* Shutdown any open client sockets. */
    MBTCP_client_id cid = 0;
    for (cid = 0; cid <= MB_TCP_CONN_LIMIT; cid++){
        tcp_socket_t* s = MB_CLIENT(cid);
        if (s == 0)
            return cid;
        if (!tcp_socket_is_online(s)){
            tcpReleaseClient(EV_CLIENT+cid);
            return cid;
        }
    }
    return MB_TCP_CONN_LIMIT;
}



#define tcpReleaseTarget() MB_METHOD_ACT1(tcpReleaseTarget)
void MB_METHOD_DECL1(tcpReleaseTarget);

void MB_METHOD_DECL(tcpReleaseClient, eMBTCPEventType client_id )
{
    MB_PORT_TCP_DECL;
    tcp_socket_t* s;
    int     id = -1;

    if (client_id == EV_LISTEN)
        s = MB_NET->listen;
    else {
        assert( (client_id>=EV_CLIENT)&&(client_id <= EV_CLIENT_LAST));
        id = client_id-EV_CLIENT;
        s= MB_CLIENT(id);
    }

    MBPortLog(vMB_debug, "MBTCP: close client [%d] to %@.4@Du:%d\n"
                        , id
                        , s->remote_ip.ucs, s->remote_port);
    if( tcp_close( s ) != 1 )
    {
        tcp_abort( s );
    }
    mutex_dettach_irq(&s->lock);
    mutex_group_remove(&MB_NET->signals_group.g, &s->lock);

    MBPortLog(vMB_info, "MBTCP: closed client [%d]\n", id);


    if (client_id == EV_LISTEN)
        MB_NET->listen = 0;
    else {
        MB_CLIENT(id) = 0;
        if ( id == MB_NET->target.id )
            tcpReleaseTarget();
    }

    mem_free(s);
}

#define tcpReleaseTargetClient() MB_METHOD_ACT1(tcpReleaseTargetClient)
void MB_METHOD_DECL1(tcpReleaseTargetClient)
{
    MB_PORT_TCP_DECL;
    tcpReleaseClient( (EV_CLIENT + MB_NET->target.id) );
}

INLINE unsigned tcpClientNextId(unsigned x){
    unsigned res = x+1;
    if (res < MB_TCP_CONN_LIMIT)
        return res;
    return 0;
}

#define tcpAquireClient(...) MB_METHOD_ACT(tcpAquireClient, __VA_ARGS__)
tcp_socket_t* MB_METHOD_DECL(tcpAquireClient, unsigned idx);

tcp_socket_t* MB_METHOD_DECL1(tcpSelectClient)
{
    //* перебираю клиентов начиная с текущего, пока не обнаружу имеющий запрос
    MB_PORT_TCP_DECL;
    if (MB_NET->target.buf != 0)
        if (MB_NET->target.buf->tot_len > 0)
            return MB_NET->target.client;

    tcp_socket_t* s = 0;
#if MB_TCP_CONN_LIMIT > 1
    unsigned id = MB_NET->target.id;

    s = tcpAquireClient(id);
    if (s != 0)
        return s;

    id = tcpClientNextId(id);
    for ( ; id != MB_NET->target.id ; id = tcpClientNextId(id) ) {
        s = tcpAquireClient(id);
        if (s != 0) {
            MB_NET->target.id = id;
            return s;
        }
    }
    return 0;
#else
    s = tcpAquireClient(0);
    return 0;
#endif
}

tcp_socket_t* MB_METHOD_DECL(tcpAquireClient, unsigned idx)
{
    MB_PORT_TCP_DECL;
    tcp_socket_t* s = MB_CLIENT(idx);
    if (s != 0){
        if (tcp_socket_is_online(s)){
            if (tcp_socket_is_avail(s)){
                MB_NET->target.client = s;
                return s;
            }
        }
        else {
            tcpReleaseClient(EV_CLIENT+idx);
            return 0;
        }
    }
    return s;
}

#define tcpSelectTarget() MB_METHOD_ACT1(tcpSelectTarget)
buf_t* MB_METHOD_DECL1(tcpSelectTarget)
{
    //* перебираю клиентов пока не получу нормальный буфер с запросом.
    //* выявляю закрывающиеся или побитые клиенты и закрываю их.
    MB_PORT_TCP_DECL;
    tcp_socket_t* s = 0;
    buf_t* buf = MB_NET->target.buf;
    while ((s = tcpSelectClient(), s) != 0) {
        if (buf != 0)
            mem_free(buf);
        buf = tcp_take_buf(s);
        if (SEANYERROR(buf)){
            //* If pbuf is NULL then remote end has closed connection
            //if (!tcp_socket_is_online(s))
            tcpReleaseTargetClient();
            buf = 0;
        }
        if (buf == 0){
            continue;
        }
        if (buf->tot_len == 0){
            //* If pbuf is NULL then remote end has closed connection
            if (!tcp_socket_is_online(s))
                tcpReleaseTargetClient();
            continue;
        }

        MBPortLog(vMB_trace, "MBTCP:client($%p) %@.4@Du:%d selected\n"
                    , s, s->remote_ip.ucs, s->remote_port
                    );

        MB_NET->target.client = s;
        MB_NET->target.buf = buf;
        MB_NET->target.FrameLen = buf->tot_len;
        return buf;
    }//while ((s =
    return 0;
}

void MB_METHOD_DECL1(tcpReleaseTarget)
{
    MB_PORT_TCP_DECL;
    buf_t* buf = MB_NET->target.buf;
    if (buf != 0) {
        mem_free(buf);
        MB_NET->target.buf = 0;
    }
    MBPortLog(vMB_trace, "MBTCP:release target %d\n", MB_NET->target.id );
    //освобождая клиента перемещусь на следующий, чтобы форсировать смену клиента
    //  при опросе нескольких активных слотов
    if (MB_NET->target.client != 0){
        MB_NET->target.client = 0;
        MB_NET->target.id = tcpClientNextId(MB_NET->target.id);
    }
}

BOOL MB_METHOD_DECL(xMBTCPPortGetRequest, UCHAR ** ppucMBTCPFrame, UFRAMESIZE * usTCPLength )
{
    MB_PORT_TCP_DECL;
    buf_t* buf = MB_NET->target.buf;
    if (buf != 0){
        //* опущу обработынй фрейм в текущем пакете, возьму для обработки следующий
        //*     (если в одном ТСПпакете пришло несколько Modbus фреймов)
        buf_add_header(buf, MB_NET->target.FrameLen);
        if (buf->tot_len == 0) {
            //* текущий буфер полностью просмотрен, освободим его и поищем новый запрос
            tcpReleaseTarget();// mem_free(MB_NET->target.buf);
            buf = 0;
        }
    }
    if (buf == 0)
        buf = tcpSelectTarget();

    while (buf != 0) {
        tcp_socket_t* s = MB_NET->target.client;
        MB_TCP_header*  htcp = (MB_TCP_header*)buf->payload;

        UFRAMESIZE len = buf->tot_len;
        UFRAMESIZE body_len = AsHostMBWord(&htcp->Len);
        if (len >= body_len+sizeof(MB_TCP_header)){
            MB_NET->target.FrameLen = len;
            buf_copy_continous(MB_NET->target.FrameTmp, buf, len);
            *ppucMBTCPFrame = MB_NET->target.FrameTmp;
            *usTCPLength    = len;
            MBPortLog(vMB_trace, "MBTCP:client %@.4@Du:%d read frame (len = %d): %*D\n"
                        , s->remote_ip.ucs, s->remote_port
                        , len
                        , len, MB_NET->target.FrameTmp
                        );
            //  полагаемся на то что ТСПфреймы всегда обрабатываются, и всегда отсылаются ответы
            //  поэтому завершение обработки и запуск нового фрейма делается в отсылке фрейма
            return TRUE;
        }
        else{
            //* look that this buffer is corrupted, just close connection
            MBPortLog(vMB_fail, "MBTCP:client %@.4@Du:%d accepted broken frame (len = %d)\n"
                        , s->remote_ip.ucs, s->remote_port
                        , len
                        );
            tcpReleaseTargetClient();
        }
        tcpReleaseTarget();
        buf = tcpSelectTarget();
    }
    return FALSE;
}

BOOL MB_METHOD_DECL(xMBTCPPortSendResponse, const UCHAR * pucMBTCPFrame, UFRAMESIZE usTCPLength )
{
    MB_PORT_TCP_DECL;
    BOOL            bFrameSent = FALSE;
    tcp_socket_t* s = MB_NET->target.client;

    if( s == 0)
        return FALSE;


    if( tcp_write( s, pucMBTCPFrame, usTCPLength) == usTCPLength )
    {
        MBPortLog(vMB_trace, "MBTCP:client %@.4@Du:%d SENT %d bytes\n"
                    , s->remote_ip.ucs, s->remote_port
                    , usTCPLength
                    );
        bFrameSent = TRUE;
    }
    else
    {
        MBPortLog(vMB_fail, "MBTCP:client %@.4@Du:%d SENT failed\n"
                    , s->remote_ip.ucs, s->remote_port
                    );
        /* Drop the connection in case of an write error. */
        tcpReleaseTargetClient();
    }
    //  полагаемся на то что ТСПфреймы всегда обрабатываются, и всегда отсылаются ответы
    //  поэтому завершение обработки и запуск нового фрейма делаю здесь
    MB_NET->ready_signals = 0;

    buf_t* buf = MB_NET->target.buf;
    if (buf != 0){
        if (buf->tot_len <= MB_NET->target.FrameLen){
            //* текущий буфер завершен, освободим его и поищем новый запрос
            tcpReleaseTarget();// mem_free(MB_NET->target.buf);
            buf = 0;
        }
    }
    if (buf == 0){
        s = tcpSelectClient();
        if (s != 0) {
            MB_NET->ready_signals++;
            xMBPortEventPost(EV_FRAME_RECEIVED);
        }
    }
    else {
        MB_NET->ready_signals++;
        xMBPortEventPost(EV_FRAME_RECEIVED);
    }

    return bFrameSent;
}

/* Called in case of an unrecoverable error. In any case we drop the client
 * connection. */
/*
static
void MB_METHOD_DECL(TCPPortError, eMBTCPEventType client_id, unsigned xErr )
{
        MBPortLog("MBTCP: Error with connection %d! Droping it.\n", client_id );
        tcpReleaseClient( client_id );
}
*/
