/**********************************************************************
 *
 * 1.07.2011 - Добавлены следующие возможности:
 *             Инверсия полярности сигнала управления направлением передачи
 *                 по RS-485 с помощью MB_PORT_RS485_DIR_INVERSE
 *                 (используется в E-124)
 *             Использование дробного делителя для задания частоты UART'а,
 *                 если определен MB_PORT_USE_FRACTIONAL_DIVIDER
 *             Явное запрещение приема данных, а не только прерываний
 *                 при вызове vMBPortSerialEnable с xRxEnable = FALSE
 **********************************************************************/

#include "LPC17xx.h"
#include "core_cm3.h"
#include "iolpc17XX.h"

#include "port.h"

#include "mb.h"
#include "mbport.h"

#ifndef MB_PORT_UART_NUM
    #define MB_PORT_UART_NUM 1
#endif

#define MB_UART_ISR            MB_UART_ISR_NUM(MB_PORT_UART_NUM)
#define MB_UART_ISR_NUM(tnum)  _MB_UART_ISR_NUM(tnum)
#define _MB_UART_ISR_NUM(tnum) UART##tnum##_IRQHandler

#define MB_UART               MB_UART_NUM(MB_PORT_UART_NUM)
#define MB_UART_NUM(tnum)     _MB_UART_NUM(tnum)
#define _MB_UART_NUM(tnum)    LPC_UART##tnum

#define MB_UART_IRQ              MB_UART_IRQ_NUM(MB_PORT_UART_NUM)
#define MB_UART_IRQ_NUM(tnum)   _MB_UART_IRQ_NUM(tnum)
#define _MB_UART_IRQ_NUM(tnum)   UART##tnum##_IRQn


/*================================================================================================*/
static void f_get_baudrate_settings(unsigned long ulBaudRate, unsigned *dl, unsigned *fract_div,
    unsigned *fract_mul );
static void prvvUARTRxISR(void);
static void prvvUARTTxReadyISR(void);
/*================================================================================================*/

/*================================================================================================*/
bool g_uart1_inten = false;

/*================================================================================================*/

/*================================================================================================*/


#ifdef MB_PORT_USE_FRACTIONAL_DIVIDER

typedef struct
{
    float fr;
    uint8_t div;
    uint8_t mul;
} t_fr_rec;

static const t_fr_rec f_fr_table[] =
{
    {1.000F,  0,  1},
    {1.067F,  1, 15},
    {1.071F,  1, 14},
    {1.077F,  1, 13},
    {1.083F,  1, 12},
    {1.091F,  1, 11},
    {1.100F,  1, 10},
    {1.111F,  1,  9},
    {1.125F,  1,  8},
    {1.133F,  2, 15},
    {1.143F,  1,  7},
    {1.154F,  2, 13},
    {1.167F,  1,  6},
    {1.182F,  2, 11},
    {1.200F,  1,  5},
    {1.214F,  3, 14},
    {1.222F,  2,  9},
    {1.231F,  3, 13},
    {1.250F,  1,  4},
    {1.267F,  4, 15},
    {1.273F,  3, 11},
    {1.286F,  2,  7},
    {1.300F,  3, 10},
    {1.308F,  4, 13},
    {1.333F,  1,  3},
    {1.357F,  5, 14},
    {1.364F,  4, 11},
    {1.375F,  3,  8},
    {1.385F,  5, 13},
    {1.400F,  2,  5},
    {1.417F,  5, 12},
    {1.429F,  3,  7},
    {1.444F,  4,  9},
    {1.455F,  5, 11},
    {1.462F,  6, 13},
    {1.467F,  7, 15},
    {1.500F,  1,  2},
    {1.533F,  8, 15},
    {1.538F,  7, 13},
    {1.545F,  6, 11},
    {1.556F,  5,  9},
    {1.571F,  4,  7},
    {1.583F,  7, 12},
    {1.600F,  3,  5},
    {1.615F,  8, 13},
    {1.625F,  5,  8},
    {1.636F,  7, 11},
    {1.643F,  9, 14},
    {1.667F,  2,  3},
    {1.692F,  9, 13},
    {1.700F,  7, 10},
    {1.714F,  5,  7},
    {1.727F,  8, 11},
    {1.733F, 11, 15},
    {1.750F,  3,  4},
    {1.769F, 10, 13},
    {1.778F,  7,  9},
    {1.786F, 11, 14},
    {1.800F,  4,  5},
    {1.818F,  9, 11},
    {1.833F,  5,  6},
    {1.846F, 11, 13},
    {1.857F,  6,  7},
    {1.867F, 13, 15},
    {1.875F,  7,  8},
    {1.889F,  8,  9},
    {1.900F,  9, 10},
    {1.909F, 10, 11},
    {1.917F, 11, 12},
    {1.923F, 12, 13},
    {1.929F, 13, 14},
    {1.933F, 14, 15}
};

#define FR_TABLE_SIZE sizeof(f_fr_table)/sizeof(f_fr_table[0])

#endif

/*------------------------------------------------------------------------------------------------*/
static void f_get_baudrate_settings
    (
    unsigned long ulBaudRate,
    unsigned *dl,
    unsigned *fract_div,
    unsigned *fract_mul
    )
    {
#ifndef MB_PORT_USE_FRACTIONAL_DIVIDER
    *fract_div = 0;
    *fract_mul = 1;
    *dl = UART_CLK / (16 * ulBaudRate);
#else
    if ((UART_CLK % (16 * ulBaudRate)==0))
        {
        *fract_div = 0;
        *fract_mul = 1;
        *dl = UART_CLK / (16 * ulBaudRate);
        }
    else
        {
        float m = ((float)UART_CLK)/(16*ulBaudRate);
        float dbr_min = ulBaudRate;
        int fnd_i = 0;
        unsigned fnd_dl = 0;
        for (int i=0; i < (FR_TABLE_SIZE); i++)
            {
                unsigned dlest = m/f_fr_table[i].fr + 0.5;
                float dbr = ((float)UART_CLK)/(16*dlest*f_fr_table[i].fr) - ulBaudRate;
                if (dbr < 0)
                    dbr=-dbr;
                if (dbr < dbr_min)
                {
                    fnd_i = i;
                    dbr_min = dbr;
                    fnd_dl = dlest;
                }
            }

        *fract_div = f_fr_table[fnd_i].div;
        *fract_mul = f_fr_table[fnd_i].mul;
        *dl = fnd_dl;
        }
#endif
    }
/*------------------------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------------------------*/
static void prvvUARTRxISR
    (
    void
    )
    {
    pxMBFrameCBByteReceived();
    }
/*------------------------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------------------------*/
static void prvvUARTTxReadyISR
    (
    void
    )
    {
    pxMBFrameCBTransmitterEmpty();
    }
/*------------------------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------------------------*/
__attribute__((__interrupt__)) void MB_UART_ISR
    (
    void
    )
    {
    unsigned long iir;

    while (!((iir = MB_UART->IIR) & LPC_UART_IIR_IntStatus_Msk))
        {
        switch ((iir & LPC_UART_IIR_IntId_Msk) >> LPC_UART_IIR_IntId_Pos)
            {
            case 0x03:
                (volatile void)MB_UART->LSR;
                break;
            case 0x02:
            case 0x06:
                MB_INDICATE_RX();
                prvvUARTRxISR();
                break;
            case 0x01:
                prvvUARTTxReadyISR();
                break;
            default:
                break;
            }
        }
    }
/*------------------------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------------------------*/
void vMBPortClose
    (
    void
    )
    {
    vMBPortSerialEnable( FALSE, FALSE );
    NVIC_DisableIRQ( MB_UART_IRQ );
    g_uart1_inten = false;
    }
/*------------------------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------------------------*/
void vMBPortSerialEnable
    (
    BOOL xRxEnable,
    BOOL xTxEnable
    )
    {
    if (xRxEnable == TRUE)
        {
        MB_UART->IER |= LPC_UART_IER_RBRInterruptEnable_Msk;
#if MB_PORT_UART_NUM == 1
        MB_UART->RS485CTRL &= ~LPC_UART1_RS485CTRL_RXDIS_Msk;
#endif
        }
    else
        {
        MB_UART->IER &= ~LPC_UART_IER_RBRInterruptEnable_Msk;
#if MB_PORT_UART_NUM == 1
        MB_UART->RS485CTRL |= LPC_UART1_RS485CTRL_RXDIS_Msk;
#endif
        }

    if (xTxEnable == TRUE)
        {
        MB_UART->IER |= LPC_UART_IER_THREInterruptEnable_Msk;
        while (!(MB_UART->LSR & LPC_UART_LSR_THRE_Msk))
            {
            continue;
            }
        prvvUARTTxReadyISR();
        }
    else
        {
        MB_UART->IER &= ~LPC_UART_IER_THREInterruptEnable_Msk;
        }

    }
/*------------------------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------------------------*/
BOOL xMBPortSerialGetByte
    (
    CHAR *pucByte
    )
    {
    *pucByte = MB_UART->RBR;
    return TRUE;
    }
/*------------------------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------------------------*/
BOOL xMBPortSerialInit
    (
    UCHAR ucPort,
    ULONG ulBaudRate,
    UCHAR ucDataBits,
    eMBParity eParity
    )
    {
    BOOL init_ok = TRUE;

    unsigned long cfg = 0;

    switch (ucDataBits)
        {
        case 5:
            break;
        case 6:
            cfg |= 0x01 << LPC_UART_LCR_WordLengthSelect_Pos;
            break;
        case 7:
            cfg |= 0x02 << LPC_UART_LCR_WordLengthSelect_Pos;
            break;
        case 8:
            cfg |= 0x03 << LPC_UART_LCR_WordLengthSelect_Pos;
            break;
        default:
            init_ok = FALSE;
        }

    switch(eParity)
        {
        case MB_PAR_NONE:
            break;
        case MB_PAR_ODD:
            cfg |= LPC_UART_LCR_ParityEnable_Msk;
            break;
        case MB_PAR_EVEN:
            cfg |= LPC_UART_LCR_ParityEnable_Msk | (0x01 << LPC_UART_LCR_ParitySelect_Pos);
            break;
        default:
            init_ok = FALSE;
        }

    if (init_ok)
        {
        MB_UART->LCR = cfg;
        MB_UART->IER = 0;
#if MB_PORT_UART_NUM == 1
        MB_UART->RS485CTRL = LPC_UART1_RS485CTRL_DCTRL_Msk;
#ifndef MB_PORT_RS485_DIR_RTS
        MB_UART->RS485CTRL |= LPC_UART1_RS485CTRL_SEL_Msk;
#endif
#ifdef MB_PORT_RS485_DIR_INVERSE
        MB_UART->RS485CTRL |= LPC_UART1_RS485CTRL_OINV_Msk;
#endif
#endif

        unsigned dl;
        unsigned fract_div;
        unsigned fract_mul;
        f_get_baudrate_settings( ulBaudRate, &dl, &fract_div, &fract_mul );
        if (fract_mul != 0)
            {
            MB_UART->LCR |= LPC_UART_LCR_DLAB_Msk;
            MB_UART->DLL = dl & 0xFF;
            MB_UART->DLM = (dl >> 8) & 0xFF;
            MB_UART->FDR =((fract_div << LPC_UART_FDR_DIVADDVAL_Pos)
                    & LPC_UART_FDR_DIVADDVAL_Msk) |
                    ((fract_mul << LPC_UART_FDR_MULVAL_Pos) & LPC_UART_FDR_MULVAL_Msk);
            MB_UART->LCR &= ~LPC_UART_LCR_DLAB_Msk;
            MB_UART->FCR = LPC_UART_FCR_FIFOEnable_Msk | LPC_UART_FCR_RXFIFOReset_Msk |
                LPC_UART_FCR_TXFIFOReset_Msk;
            MB_UART->FCR = LPC_UART_FCR_FIFOEnable_Msk;

            NVIC_EnableIRQ( MB_UART_IRQ );
            g_uart1_inten = true;


            (volatile void)MB_UART->IIR;
            }
        else
            {
            init_ok = FALSE;
            }
        }

    g_mbport_mode = e_MBPortMode_Serial;

    return init_ok;
    }
/*------------------------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------------------------*/
BOOL xMBPortSerialPutByte
    (
    CHAR ucByte
    )
    {
    MB_UART->THR = ucByte;
    MB_INDICATE_TX();
    return TRUE;
    }
/*------------------------------------------------------------------------------------------------*/
/*================================================================================================*/
