package jp.sourceforge.freegantt.swing;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.DefaultCellEditor;
import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTable;
import javax.swing.KeyStroke;
import javax.swing.border.Border;
import javax.swing.event.TableModelEvent;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.undo.CompoundEdit;

import jp.sourceforge.freegantt.data.Column;
import jp.sourceforge.freegantt.data.Member;
import jp.sourceforge.freegantt.data.Project;
import jp.sourceforge.freegantt.data.Task;
import jp.sourceforge.freegantt.data.model.ProjectViewChangedListener;
import jp.sourceforge.freegantt.data.model.TaskListTableModel;
import jp.sourceforge.freegantt.locale.Resource;
import jp.sourceforge.freegantt.swing.TaskNameCellRenderer.FoldBorder;

public class TaskListTable extends JTable implements MouseListener {
	private static final long serialVersionUID = -2059641956588584594L;

	Application app;
	Project project;
	TaskListTableModel tableModel;
	TaskNameCellRenderer taskNameCellRenderer;
	NumberCellRenderer numberCellRenderer;
	boolean initialized = false;
	
	Color borderColor = new Color(0xC0, 0xC0, 0xC0);
	
	public TaskListTableModel getTaskListTableModel() {
		return tableModel;
	}

	public TaskListTable(Application app) {
		super();
		this.app = app;
		this.project = app.getProject();
		this.taskNameCellRenderer = new TaskNameCellRenderer(project);
		this.numberCellRenderer = new NumberCellRenderer();
		
		HeaderSwitchMouseHandler headerSwitchMouseHandler = new HeaderSwitchMouseHandler();
		getTableHeader().addMouseListener(new HeaderPopupMouseListener());
		getTableHeader().addMouseListener(headerSwitchMouseHandler);
		getTableHeader().addMouseMotionListener(headerSwitchMouseHandler);
		project.getProjectViewModel().addProjectViewChangedListener(new ProjectViewChangedHandler());
		
		setGridColor(borderColor);
		setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);
		addMouseListener(this);
		registerKeyboardAction(new RemoveAction(), KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0), JComponent.WHEN_FOCUSED);
		registerKeyboardAction(new InsertAction(), KeyStroke.getKeyStroke(KeyEvent.VK_INSERT, 0), JComponent.WHEN_FOCUSED);
		registerKeyboardAction(app.getMenu().getCutAction(), KeyStroke.getKeyStroke(KeyEvent.VK_X, InputEvent.CTRL_DOWN_MASK), JComponent.WHEN_FOCUSED);
		registerKeyboardAction(app.getMenu().getCopyAction(), KeyStroke.getKeyStroke(KeyEvent.VK_C, InputEvent.CTRL_DOWN_MASK), JComponent.WHEN_FOCUSED);
		registerKeyboardAction(app.getMenu().getPasteAction(), KeyStroke.getKeyStroke(KeyEvent.VK_V, InputEvent.CTRL_DOWN_MASK), JComponent.WHEN_FOCUSED);
		registerKeyboardAction(app.getMenu().getUndoAction(), KeyStroke.getKeyStroke(KeyEvent.VK_Z, InputEvent.CTRL_DOWN_MASK), JComponent.WHEN_FOCUSED);
		registerKeyboardAction(app.getMenu().getRedoAction(), KeyStroke.getKeyStroke(KeyEvent.VK_Y, InputEvent.CTRL_DOWN_MASK), JComponent.WHEN_FOCUSED);
		
		tableModel = project.getTaskTableModel();
		setModel(tableModel);
		TableColumnModel columnModel = getColumnModel();
		
		String[] columnKeys = tableModel.getColumnKeys();
		for (int i=0; i<columnKeys.length; i++) {
			TableColumn column = columnModel.getColumn(i);
			column.setIdentifier(columnKeys[i]);
			column.setMinWidth(0);
		}

		updateColumnArrangement();
		
		columnModel.getColumn(2).setCellRenderer(new TaskMemberCellRenderer());
		columnModel.getColumn(2).setCellEditor(new DefaultCellEditor(new TaskMemberCellEditor(project)));
		columnModel.getColumn(0).setCellRenderer(getTableHeader().getDefaultRenderer());
		columnModel.getColumn(0).setMaxWidth(30);
		columnModel.getColumn(0).setMinWidth(30);
		columnModel.getColumn(1).setCellRenderer(taskNameCellRenderer);
		columnModel.getColumn(3).setCellRenderer(numberCellRenderer);
		columnModel.getColumn(4).setCellRenderer(numberCellRenderer);
		
		
		initialized = true;
	}
	
	private void updateColumnArrangement() {
		List<Column> columns = project.getView().getColumns();
		int index = 1;
		for (Column column: columns) {
			getColumnModel().moveColumn(getColumnModel().getColumnIndex(column.getKey()), index);
			TableColumn tableColumn = getColumn(column.getKey());
			tableColumn.setPreferredWidth(column.getWidth());
			tableColumn.setWidth(column.getWidth());
			index++;
		}
	}

	@Override
	public void tableChanged(TableModelEvent e) {
		super.tableChanged(e);
		
		if (app == null) return;
		
		// チャートペインの高さをタスクリストテーブルの高さと同じに設定する
		TaskLineDataPane taskLineDataPane = app.getTaskLineDataPane();
		if (taskLineDataPane != null) {
			taskLineDataPane.setSize(new Dimension(taskLineDataPane.getWidth(), getTaskListTableModel().getRowCount() * 16));
			taskLineDataPane.setPreferredSize(new Dimension(taskLineDataPane.getWidth(), getTaskListTableModel().getRowCount() * 16));
			System.out.println("DataPane setPreferredSize: " + taskLineDataPane.getWidth() + "," +  getTaskListTableModel().getRowCount() * 16);
		}

		repaint();
		
		if (app.getTaskListRootPane() != null) {
			app.getTaskListRootPane().getViewport().validate();
		}
		
		if (app.getTaskLineDataPane() != null) {
			app.getTaskLineDataPane().repaint();
		}
		
		if (app.getTaskListTable() != null) {
			app.getTaskListTable().repaint();
		}
	}

	@Override
	public void mouseClicked(MouseEvent e) {
	}

	@Override
	public void mousePressed(MouseEvent e) {
		if (e.isPopupTrigger()) {
			triggerPopup(e);
			return;
		}
		
		if (handleFoldOperation(e)) {
			return;
		}
	}

	@Override
	public void mouseReleased(MouseEvent e) {
		if (e.isPopupTrigger()) {
			triggerPopup(e);
			return;
		}
	}
	
	public void removeSelectedRows() {
		CompoundEdit compound = new CompoundEdit();
		project.getUndoManager().addEdit(compound);
		try {
			int deleted = 0;
			int[] selectedRows = getSelectedRows();
			for (int s: selectedRows) {
				int row = s - deleted;
				project.getController().removeIndex(project.getIndexFromRow(row));
				deleted ++;
			}
			project.update();
		} finally {
			compound.end();
		}
	}
	
	public void insertSelectedRows() {
		CompoundEdit compound = new CompoundEdit();
		project.getUndoManager().addEdit(compound);
		try {
			int inserted = 0;
			int chained = 0;
			int preSelected = -2;
			int[] selectedRows = getSelectedRows();
			for (int s: selectedRows) {
				if (preSelected + 1 == s) chained ++; else chained = 0;
				int row = s + inserted - chained;
				Task newTask = new Task();
				Task lowerTask = project.getTaskAtRow(row + 1);
				if (lowerTask != null) {
					newTask.setLevel(lowerTask.getLevel());
				}
				project.getController().insertTaskAtIndex(newTask, project.getIndexFromRow(row));
				inserted ++;
				preSelected = s;
			}
			project.update();
		} finally {
			compound.end();
		}
	}

	private void triggerPopup(MouseEvent e) {
		final int row = TaskListTable.this.rowAtPoint(e.getPoint());
		if (row < 0) return;
		
		boolean found = false;
		int[] selectedRows = getSelectedRows();
		for (int s: selectedRows) {
			if (s == row) found = true;
		}
		if (!found) {
			setRowSelectionInterval(row, row);
			repaint();
		}
		
		boolean isAvailableLevelUp = true;
		boolean isAvailableLevelDown = true;
		
		JPopupMenu menu = new JPopupMenu();
		JMenuItem item = new JMenuItem(Resource.get("listLevelUp"));
		item.setEnabled(isAvailableLevelUp);
		item.addActionListener(new LevelUpAction());
		menu.add(item);
		item = new JMenuItem(Resource.get("listLevelDown"));
		item.setEnabled(isAvailableLevelDown);
		item.addActionListener(new LevelDownAction());
		menu.add(item);
		menu.addSeparator();
		item = new JMenuItem(Resource.get("listRemoveTask"));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				removeSelectedRows();
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("listInsertTask"));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				insertSelectedRows();
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("menuEditCut"));
		item.addActionListener(app.getMenu().getCutAction());
		menu.add(item);
		item = new JMenuItem(Resource.get("menuEditCopy"));
		item.addActionListener(app.getMenu().getCopyAction());
		menu.add(item);
		item = new JMenuItem(Resource.get("menuEditPaste"));
		item.addActionListener(app.getMenu().getPasteAction());
		menu.add(item);
		menu.show(this, e.getX(), e.getY());
	}

	@Override
	public void mouseEntered(MouseEvent e) {
	}

	@Override
	public void mouseExited(MouseEvent e) {
	}

	private boolean handleFoldOperation(MouseEvent e) {
		TableColumnModel columnModel = getColumnModel();
		int row = rowAtPoint(e.getPoint());
		int column = columnAtPoint(e.getPoint());
		int columnIndex = columnModel.getColumnIndexAtX(e.getX());
		Object columnKey = columnModel.getColumn(columnIndex).getIdentifier();
		if (!columnKey.equals("name")) return false;
		
		Rectangle cellRect = getCellRect(row, column, false);
		Point cellPoint = new Point(e.getX() - cellRect.x, e.getY() - cellRect.y);

		taskNameCellRenderer.getTableCellRendererComponent(this, "", true, true, row, column);
		Border border = taskNameCellRenderer.getBorder();
		if (!(border instanceof FoldBorder)) return false;
		
		Rectangle foldRect = ((FoldBorder)border).getFoldRect();
		if (foldRect == null) return false;
		if (!foldRect.contains(cellPoint)) return false;
		
		project.toggleTaskFold(row);
		project.getTaskTableModel().fireTableChanged();
		return true;
	}

	@Override
	public void doLayout() {
		JTableHeader header = getTableHeader();
		TableColumn resizingColumn = header.getResizingColumn();
		if (resizingColumn != null) {
			resizingColumn.setPreferredWidth(resizingColumn.getWidth());
		}
		
		TableColumnModel model = getColumnModel();
		for (int i=0; i<model.getColumnCount(); i++) {
			TableColumn column = model.getColumn(i);
			column.setWidth(column.getPreferredWidth());
		}
	}
	
	private List<Column> getCurrentColumns() {
		List<Column> columns = new ArrayList<Column>();
		for (int i=1; i<getColumnModel().getColumnCount(); i++) {
			TableColumn column = getColumnModel().getColumn(i);
			String key = (String)column.getIdentifier();
			columns.add(new Column(key, column.getWidth()));
		}
		return columns;
	}
	
	public void copySelectedToClipboard() {
		int[] selectedRows = getSelectedRows();
		StringBuilder builder = new StringBuilder();
		for (int s: selectedRows) {
			for (int i=0; i<getColumnCount(); i++) {
				Object value = getValueAt(s, i);
				if (value instanceof Member) value = ((Member)value).getName();
				builder.append(value == null ? "" : value.toString());
				builder.append(i + 1 == getColumnCount() ? "\n" : "\t");
			}
		}
		Clipboard clip = Toolkit.getDefaultToolkit().getSystemClipboard();
		StringSelection ss = new StringSelection(builder.toString());
		clip.setContents(ss, ss);
	}
	
	public void insertFromClipboard() throws UnsupportedFlavorException, IOException {
		Clipboard clip = Toolkit.getDefaultToolkit().getSystemClipboard();
		String data = (String)clip.getData(DataFlavor.stringFlavor);
		
		CompoundEdit compound = new CompoundEdit();
		project.getUndoManager().addEdit(compound);
		try {
			int[] selectedRows = getSelectedRows();
			int baseRow = (selectedRows.length > 0 ? selectedRows[0] : 0);
			String[] lines = data.split("\n");
			for (int l=0; l<lines.length; l++) {
				String line = lines[l];
				String[] fragments = line.split("\t");
				project.getController().insertTaskAtIndex(new Task(), project.getIndexFromRow(baseRow + l));
				for (int f=0; f<fragments.length; f++) {
					setValueAt(fragments[f], baseRow + l, f);
				}
			}
			project.update();
		} finally {
			compound.end();
		}
	}

//	public void pasteFromClipboard() throws UnsupportedFlavorException, IOException {
//		Clipboard clip = Toolkit.getDefaultToolkit().getSystemClipboard();
//		String data = (String)clip.getData(DataFlavor.stringFlavor);
//		
//		int[] selectedRows = getSelectedRows();
//		int baseRow = (selectedRows.length > 0 ? selectedRows[0] : 0);
//		
//		String[] lines = data.split("\n");
//		for (int l=0; l<lines.length; l++) {
//			String line = lines[l];
//			String[] fragments = line.split("\t");
//			for (int f=0; f<fragments.length; f++) {
//				setValueAt(fragments[f], baseRow + l, f);
//			}
//		}
//	}

	class HeaderPopupMouseListener implements MouseListener {
		@Override
		public void mouseClicked(MouseEvent e) {
		}

		@Override
		public void mousePressed(MouseEvent e) {
			if (e.isPopupTrigger()) {
				triggerPopup(e);
			}
		}

		@Override
		public void mouseReleased(MouseEvent e) {
			if (e.isPopupTrigger()) {
				triggerPopup(e);
			}
		}
		
		public void triggerPopup(MouseEvent e) {
			JPopupMenu menu = new JPopupMenu();
			String columnNames[] = getTaskListTableModel().getColumnNames();
			for (int index=0; index<columnNames.length; index++) {
				if (index == 0) continue;
				JMenuItem item = new JMenuItem(columnNames[index]);
				item.addActionListener(new ColumnNameActionListener(index));
				menu.add(item);
			}
			menu.show(e.getComponent(), e.getX(), e.getY());
		}

		@Override
		public void mouseEntered(MouseEvent e) {
		}

		@Override
		public void mouseExited(MouseEvent e) {
		}
	}
	
	class ColumnNameActionListener implements ActionListener {
		int index;
		
		public ColumnNameActionListener(int index) {
			this.index = index;
		}
		
		@Override
		public void actionPerformed(ActionEvent e) {
			List<Column> oldColumns = getCurrentColumns();
			List<Column> newColumns = new ArrayList<Column>();
			for (Column column: oldColumns) {
				newColumns.add(column.clone());
			}

			String key = tableModel.getColumnKey(index);
			for (Column column: newColumns) {
				if (column.getKey().equals(key)) {
					int width = (column.getWidth() == 0 ? project.getView().getDefaultColumn(key).getWidth() : 0);
					column.setWidth(width);
				}
			}

			boolean equals = (oldColumns.size() == newColumns.size());
			if (equals) {
				for (int i=0; i<oldColumns.size(); i++) {
					equals = equals && (oldColumns.get(i).equals(newColumns.get(i)));
				}
			}
			if (equals) return;
			
			CompoundEdit compound = new CompoundEdit();
			project.getUndoManager().addEdit(compound);
			try {
				project.getController().setViewTaskTableColumns(project, oldColumns, newColumns);
			} finally {
				compound.end();
			}
		}
	}

	class RemoveAction implements ActionListener {
		@Override
		public void actionPerformed(ActionEvent e) {
			removeSelectedRows();
		}
	}
	
	class InsertAction implements ActionListener {
		@Override
		public void actionPerformed(ActionEvent e) {
			insertSelectedRows();
		}
	}
	
	class HeaderSwitchMouseHandler implements MouseListener, MouseMotionListener {

		boolean dragged = false;
		List<Column> oldColumns;
		List<Column> newColumns;
		
		@Override
		public void mouseClicked(MouseEvent e) {
		}

		@Override
		public void mousePressed(MouseEvent e) {
			if (e.getButton() != MouseEvent.BUTTON1) return;
			oldColumns = getCurrentColumns();
			dragged = false;
		}

		@Override
		public void mouseReleased(MouseEvent e) {
			if (e.getButton() != MouseEvent.BUTTON1) return;
			if (!dragged) return;
			newColumns = getCurrentColumns();
			boolean equals = (oldColumns.size() == newColumns.size());
			if (equals) {
				for (int i=0; i<oldColumns.size(); i++) {
					equals = equals && (oldColumns.get(i).equals(newColumns.get(i)));
				}
			}
			if (equals) return;
			
			CompoundEdit compound = new CompoundEdit();
			project.getUndoManager().addEdit(compound);
			try {
				project.getController().setViewTaskTableColumns(project, oldColumns, newColumns);
			} finally {
				compound.end();
				dragged = false;
			}
		}

		@Override
		public void mouseEntered(MouseEvent e) {
		}

		@Override
		public void mouseExited(MouseEvent e) {
		}

		@Override
		public void mouseDragged(MouseEvent e) {
			dragged = true;
		}

		@Override
		public void mouseMoved(MouseEvent e) {
		}
	}
	
	class LevelUpAction extends AbstractAction {
		private static final long serialVersionUID = 785583037631700955L;

		@Override
		public void actionPerformed(ActionEvent e) {
			CompoundEdit compound = new CompoundEdit();
			project.getUndoManager().addEdit(compound);
			
			try {
				int[] rows = getSelectedRows();
				for (int row: rows) {
					project.getController().levelUpTask(project.getTaskAtRow(row));
				}
				project.update();
			} finally {
				compound.end();
				tableModel.fireTableChanged();
			}
		}
	}
	
	class LevelDownAction extends AbstractAction {
		private static final long serialVersionUID = 785583037631700955L;

		@Override
		public void actionPerformed(ActionEvent e) {
			CompoundEdit compound = new CompoundEdit();
			project.getUndoManager().addEdit(compound);
			
			try {
				int[] rows = getSelectedRows();
				for (int row: rows) {
					project.getController().levelDownTask(project.getTaskAtRow(row));
				}
				project.update();
			} finally {
				compound.end();
				tableModel.fireTableChanged();
			}
		}
	}
	
	class ProjectViewChangedHandler implements ProjectViewChangedListener {

		@Override
		public void projectViewChanged() {
			updateColumnArrangement();
		}
		
	}
}
