package jp.sourceforge.freegantt.swing;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.io.IOException;
import java.util.Locale;

import javax.swing.AbstractAction;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.KeyStroke;
import javax.swing.UIManager;
import javax.swing.event.MenuEvent;
import javax.swing.event.MenuListener;
import javax.swing.undo.CompoundEdit;

import jp.sourceforge.freegantt.locale.Resource;
import jp.sourceforge.freegantt.print.PreviewDialog;

public class ApplicationMenu extends JMenuBar {
	private static final long serialVersionUID = -8682706642684252609L;

	Application app;
	UndoAction undoAction 			= new UndoAction();
	RedoAction redoAction 			= new RedoAction();
	CutAction cutAction 			= new CutAction();
	CopyAction copyAction 			= new CopyAction();
	PasteAction pasteAction 		= new PasteAction();
	ScaleUpAction scaleUpAction 	= new ScaleUpAction();
	ScaleDownAction scaleDownAction = new ScaleDownAction();
	
	JMenuItem editUndo;
	JMenuItem editRedo;
	JMenuItem editCut;
	JMenuItem editCopy;
	JMenuItem editPaste;
	
	

	public ScaleUpAction getScaleUpAction() {
		return scaleUpAction;
	}
	public ScaleDownAction getScaleDownAction() {
		return scaleDownAction;
	}
	public UndoAction getUndoAction() {
		return undoAction;
	}
	public RedoAction getRedoAction() {
		return redoAction;
	}
	public CutAction getCutAction() {
		return cutAction;
	}
	public CopyAction getCopyAction() {
		return copyAction;
	}
	public PasteAction getPasteAction() {
		return pasteAction;
	}
	public ApplicationMenu(final Application app) {
		super();
		this.app = app;
		
		Font font = getFont();
		Font newFont = new Font(font.getName(), 0, font.getSize());
		UIManager.put("Menu.font", newFont);
		UIManager.put("MenuItem.font", newFont);
		UIManager.put("MenuItem.acceleratorFont", newFont);
		UIManager.put("MenuItem.acceleratorForeground", Color.gray);
		
		JMenu menu = new JMenu(Resource.get("menuFile"));
		JMenuItem item = new JMenuItem(Resource.get("menuFileNew"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, InputEvent.CTRL_MASK));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.newFile();
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("menuFileOpen"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, InputEvent.CTRL_MASK));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.loadData();
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("menuFileSave"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, InputEvent.CTRL_MASK));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.saveData();
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("menuFileSaveAs"));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.saveAsData();
			}
		});
		menu.add(item);
		menu.addSeparator();
		item = new JMenuItem(Resource.get("menuFilePreview"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P, InputEvent.CTRL_MASK));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				PreviewDialog dialog = new PreviewDialog(app);
				dialog.setVisible(true);
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("menuFilePage"));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.configurePage();
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("menuFilePrint"));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.print();
			}
		});
		menu.add(item);
		add(menu);

		menu = new JMenu(Resource.get("menuEdit"));
		menu.addMenuListener(new EditPopupListener());
		item = new JMenuItem(Resource.get("menuEditUndo"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Z, InputEvent.CTRL_MASK));
		item.addActionListener(undoAction);
		menu.add(item);
		editUndo = item;
		item = new JMenuItem(Resource.get("menuEditRedo"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Y, InputEvent.CTRL_MASK));
		item.addActionListener(redoAction);
		menu.add(item);
		editRedo = item;
		menu.addSeparator();
		item = new JMenuItem(Resource.get("menuEditCut"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X, InputEvent.CTRL_MASK));
		item.addActionListener(cutAction);
		menu.add(item);
		editCut =  item;
		item = new JMenuItem(Resource.get("menuEditCopy"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C, InputEvent.CTRL_MASK));
		item.addActionListener(copyAction);
		menu.add(item);
		editCopy = item;
		item = new JMenuItem(Resource.get("menuEditPaste"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V, InputEvent.CTRL_MASK));
		item.addActionListener(pasteAction);
		menu.add(item);
		editPaste = item;
		add(menu);

		menu = new JMenu(Resource.get("menuView"));
		item = new JMenuItem(Resource.get("menuViewCalendarMode"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_W, InputEvent.CTRL_MASK | InputEvent.SHIFT_MASK));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.changeCalendarMode();
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("menuViewProgress"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_G, InputEvent.CTRL_MASK | InputEvent.SHIFT_MASK));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.getTaskLineDataPane().setProgressLineMode(!app.getTaskLineDataPane().isProgressLineMode());
				app.getTaskLineDataPane().repaint();
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("menuViewPage"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P, InputEvent.CTRL_MASK | InputEvent.SHIFT_MASK));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.getTaskLineDataPane().setPrintRangeMode(!app.getTaskLineDataPane().isPrintRangeMode());
				app.getTaskLineDataPane().repaint();
			}
		});
		menu.add(item);
		menu.addSeparator();
		item = new JMenuItem(Resource.get("menuViewScaleUp"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_PLUS, InputEvent.CTRL_MASK));
		item.addActionListener(scaleUpAction);
		menu.add(item);
		item = new JMenuItem(Resource.get("menuViewScaleDown"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_MINUS, InputEvent.CTRL_MASK));
		item.addActionListener(scaleDownAction);
		menu.add(item);
		add(menu);
		
		menu = new JMenu(Resource.get("menuWindow"));
		item = new JMenuItem(Resource.get("menuWindowInfo"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_I, InputEvent.CTRL_MASK | InputEvent.SHIFT_MASK));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.openConfig(Application.CONFIG_PROJECT);
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("menuWindowResource"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R, InputEvent.CTRL_MASK | InputEvent.SHIFT_MASK));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.openConfig(Application.CONFIG_MEMBER);
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("menuWindowHoliday"));
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_H, InputEvent.CTRL_MASK | InputEvent.SHIFT_MASK));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.openConfig(Application.CONFIG_HOLIDAY);
			}
		});
		menu.add(item);
		add(menu);
		
		menu = new JMenu(Resource.get("menuLang"));
		item = new JMenuItem(Resource.get("menuLangEn"));
		item.setEnabled(!app.getLocale().getLanguage().equals(Locale.ENGLISH.getLanguage()));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.updateLocale(Locale.ENGLISH);
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("menuLangJp"));
		item.setEnabled(!app.getLocale().getLanguage().equals(Locale.JAPANESE.getLanguage()));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.updateLocale(Locale.JAPANESE);
			}
		});
		menu.add(item);
		add(menu);
		
		menu = new JMenu(Resource.get("menuHelp"));
		item = new JMenuItem(Resource.get("menuHelpDocument"));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.openHelpDocument();
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("menuHelpVersion"));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				app.openVersionDialog();
			}
		});
		menu.add(item);
		add(menu);
	}
	
	class EditPopupListener implements MenuListener {

		@Override
		public void menuSelected(MenuEvent e) {
			editUndo.setEnabled(app.getProject().getUndoManager().canUndo());
			editRedo.setEnabled(app.getProject().getUndoManager().canRedo());
			TaskListTable table = app.getTaskListTable();
			if (table == null) return;
			editCut.setEnabled(table.getSelectedRows().length > 0);
			editCopy.setEnabled(table.getSelectedRows().length > 0);
		}

		@Override
		public void menuDeselected(MenuEvent e) {
			editUndo.setEnabled(true);
			editRedo.setEnabled(true);
			editCut.setEnabled(true);
			editCopy.setEnabled(true);
		}

		@Override
		public void menuCanceled(MenuEvent e) {
			editUndo.setEnabled(true);
			editRedo.setEnabled(true);
			editCut.setEnabled(true);
			editCopy.setEnabled(true);
		}
		
	}

	class UndoAction extends AbstractAction {
		private static final long serialVersionUID = -2191070476189083744L;

		@Override
		public void actionPerformed(ActionEvent e) {
			if (app.getProject().getUndoManager().canUndo()) {
				app.getProject().getUndoManager().undo();
				app.getProject().getTaskTableModel().fireTableChanged();
			}
		}
	}
	
	class RedoAction extends AbstractAction {
		private static final long serialVersionUID = -5980994835631840346L;
		
		@Override
		public void actionPerformed(ActionEvent e) {
			if (app.getProject().getUndoManager().canRedo()) {
				app.getProject().getUndoManager().redo();
				app.getProject().getTaskTableModel().fireTableChanged();
			}
		}
	}
	
	class CutAction extends AbstractAction {
		private static final long serialVersionUID = 1046831944357988605L;

		@Override
		public void actionPerformed(ActionEvent e) {
			TaskListTable table = app.getTaskListTable();
			if (table == null) return;
			
			table.copySelectedToClipboard();
			
			CompoundEdit compound = new CompoundEdit();
			app.getProject().getUndoManager().addEdit(compound);
			try {
				table.removeSelectedRows();
			} finally {
				compound.end();
			}
		}
	}
	
	class CopyAction extends AbstractAction {
		private static final long serialVersionUID = 318323821561167362L;

		@Override
		public void actionPerformed(ActionEvent e) {
			TaskListTable table = app.getTaskListTable();
			if (table == null) return;
			
			table.copySelectedToClipboard();
		}
	}
	
	class PasteAction extends AbstractAction {
		private static final long serialVersionUID = 3415153071727519882L;

		@Override
		public void actionPerformed(ActionEvent e) {
			TaskListTable table = app.getTaskListTable();
			if (table == null) return;
			
			CompoundEdit compound = new CompoundEdit();
			app.getProject().getUndoManager().addEdit(compound);
			try {
				table.insertFromClipboard();
			} catch (UnsupportedFlavorException e1) {
				e1.printStackTrace();
			} catch (IOException e1) {
				e1.printStackTrace();
			} finally {
				compound.end();
			}
			
		}
	}
	
	class ScaleDownAction extends AbstractAction {
		private static final long serialVersionUID = -6101051094133885905L;

		@Override
		public void actionPerformed(ActionEvent e) {
			Dimension cellSize = (Dimension)app.getProject().getCellSize().clone();
			cellSize.width -= 1;
			app.getProject().getController().setCellSize(cellSize);
		}
	}
	
	class ScaleUpAction extends AbstractAction {
		private static final long serialVersionUID = -716691612709376253L;

		@Override
		public void actionPerformed(ActionEvent e) {
			Dimension cellSize = (Dimension)app.getProject().getCellSize().clone();
			cellSize.width += 1;
			app.getProject().getController().setCellSize(cellSize);
		}
	}

}
