package jp.sourceforge.freegantt.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Desktop;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.awt.print.PageFormat;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Locale;

import javax.swing.ImageIcon;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.JViewport;
import javax.swing.SwingUtilities;
import javax.swing.filechooser.FileNameExtensionFilter;

import jp.sourceforge.freegantt.data.Project;
import jp.sourceforge.freegantt.data.model.ProjectModifiedListener;
import jp.sourceforge.freegantt.data.serializer.FileFormatException;
import jp.sourceforge.freegantt.data.serializer.FtgSerializer;
import jp.sourceforge.freegantt.locale.Resource;
import jp.sourceforge.freegantt.persistence.Initialize;
import jp.sourceforge.freegantt.print.GanttChartPrintable;

import org.xml.sax.SAXException;

public class Application extends JFrame  {

	private static final long serialVersionUID = 1035207996697568561L;

	public final static String TITLE = "FreeGantt";
	public final static String VERSION = "0.0.7";
	public final static String SKELETON_FILE = "skel.fgt";
	public final static String ICON_FILE = "icon.png";
	
	public final static int CONFIG_PROJECT = 0;
	public final static int CONFIG_MEMBER = 1;
	public final static int CONFIG_HOLIDAY = 2;
	public final static int CONFIG_MAX = 3;

	
	Project project;
	ApplicationMenu menuBar;
	Image iconImage = new ImageIcon(getClass().getResource(ICON_FILE)).getImage();	
	
	// components
	JPanel iconMenuPane;
	ConfigSplitPane configSplitPane;
	GanttChartPane chartPane;
	ConfigRootPane configPane;
	ProjectRootPane projectPane;
	MemberRootPane memberPane;
	HolidayRootPane holidayPane;
	
	
	public ApplicationMenu getMenu() {
		return menuBar;
	}
	public Project getProject() {
		return project;
	}
	public void setProject(Project project) {
		this.project = project;
	}

	public static void main(String[] args) {
		final Application app = new Application();
		Resource.makeInstance(app.getLocale());
		Initialize.getInstance().load();
		
		app.setProject(new Project());
		if (args.length > 0) {
			app.loadData(args[0]);
		} else {
			app.loadSkelton();
		}
		
		SwingUtilities.invokeLater(new Runnable() {
            public void run () {
        		app.initialize();
        		
        		app.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
        		app.setSize(
        				Initialize.getInstance().getInt("applicationWidth", 1024), 
        				Initialize.getInstance().getInt("applicationHeight", 768));
        		app.setBackground(Color.white);
        		app.setTitle(TITLE);
        		app.addWindowListener(app.createWindowListener());
        		app.setIconImage(app.getApplicationIconImage());
            }
        });
	}
	
	/**
	 * 新規ファイル
	 */
	public void newFile() {
		if (confirmDiscard()) return;

		loadSkelton();
		initialize();
	}
	
	private void initialize() {
		setVisible(false);
		getProject().getProjectModifiedModel().addProjectModifiedListener(new ProjectModifiedHandler());
		project.updateChartRange();
		
		JOptionPane.setDefaultLocale(Resource.getLocale()); 
		
		menuBar = new ApplicationMenu(this);
		setJMenuBar(menuBar);
		
		getContentPane().removeAll();
		
		iconMenuPane = new ApplicationItemMenu(this);
		getContentPane().add(iconMenuPane, BorderLayout.NORTH);
		
		configSplitPane = new ConfigSplitPane(JSplitPane.HORIZONTAL_SPLIT);
		configSplitPane.setBounds(0, 0, 1, 1);
		configSplitPane.setPreferredSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		configSplitPane.setMaximumSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		configSplitPane.setDividerSize(4);
		configSplitPane.setContinuousLayout(true);
		configSplitPane.setResizeWeight(1.0d);
		getContentPane().add(configSplitPane, BorderLayout.CENTER);
		
		chartPane = new GanttChartPane(this);
		chartPane.setBounds(0, 0, 1, 1);
		chartPane.setPreferredSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		chartPane.setMaximumSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		chartPane.setLayout(null);
		configSplitPane.add(chartPane);
		
		configPane = new ConfigRootPane();
		configPane.setBounds(0, 0, 1, 1);
		configPane.setPreferredSize(new Dimension(0, Integer.MAX_VALUE));
		configPane.setMaximumSize(new Dimension(300, Integer.MAX_VALUE));
		configPane.setMinimumSize(new Dimension(1,1));
		configPane.setTabLayoutPolicy(JTabbedPane.SCROLL_TAB_LAYOUT);
		configSplitPane.add(configPane);
	
		projectPane = new ProjectRootPane(this);
		projectPane.setBounds(0, 0, 300, 200);
		projectPane.setPreferredSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		projectPane.setMaximumSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		configPane.addTab(Resource.get("tabInfo"), projectPane);

		memberPane = new MemberRootPane(this);
		memberPane.setBounds(0, 0, 300, 200);
		memberPane.setPreferredSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		memberPane.setMaximumSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		configPane.addTab(Resource.get("tabResource"), memberPane);
		
		holidayPane = new HolidayRootPane(this);
		holidayPane.setBounds(0, 0, 300, 200);
		holidayPane.setPreferredSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		holidayPane.setMaximumSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		configPane.addTab(Resource.get("tabHoliday"), holidayPane);
		
		getTaskLineRootPane().initializeViewportOffset();
		setVisible(true);
	}
	
	protected void saveData(String filename) {
		FileOutputStream out = null;
		try {
			out = new FileOutputStream(filename);
			(new FtgSerializer()).write(project, out);
			out.close();
			this.project.setFilename(filename);
			this.project.setFileLastModified(new File(filename).lastModified());
			this.project.setModified(false);
			updateTitle();
		} catch (IOException e) {
			e.printStackTrace();
			JOptionPane.showMessageDialog(null, 
					Resource.get("fileIOErrorMessage"), 
					Resource.get("fileIOErrorTitle"), 
					JOptionPane.ERROR_MESSAGE);
		} finally {
			try {
				if (out != null)
					out.close();
			} catch (IOException e1) {
			}
		}

	}
	
	private void updateTitle() {
		StringBuilder builder = new StringBuilder();
		builder.append(TITLE);
		builder.append(project.isModified() ? " * " : " ");
		if (project.getFilename() != null) {
			builder.append(project.getFilename());
		}
		setTitle(builder.toString());
	}
	
	public void saveData() {
		if (project.getFilename() == null) {
			saveAsData();
		} else {
			File file = new File(project.getFilename());
			if (file.lastModified() != project.getFileLastModified()) {
				int ret = JOptionPane.showConfirmDialog(this, 
						Resource.get("saveLastModifiedConfirmMessage"), 
						Resource.get("saveLastModifiedConfirmTitle"),
						JOptionPane.OK_CANCEL_OPTION);
				if (ret == JOptionPane.OK_OPTION) {
					saveData(project.getFilename());
				}
			} else {
				saveData(project.getFilename());
			}
		}
	}

	public void saveAsData() {
		JFileChooser chooser = new JFileChooser();
		FileNameExtensionFilter filter = new FileNameExtensionFilter(
				Resource.get("dialogFileType"), "fgt");
		chooser.setFileFilter(filter);
		int ret = chooser.showSaveDialog(this);
		if (ret == JFileChooser.APPROVE_OPTION) {
			String filename = chooser.getSelectedFile().getAbsolutePath();
			if (!filename.endsWith(".fgt")) {
				filename = filename.concat(".fgt");
			}
			saveData(filename);
		}
	}
	
	protected void loadSkelton() {
		if ((new File(SKELETON_FILE)).exists()) {
			loadData(SKELETON_FILE);
			project.setFilename(null);
		} else {
			project = new Project();
		}
	}
	
	protected void loadData(String filename) {
		FileInputStream in = null;
		try {
			File file = new File(filename);
			in = new FileInputStream(file);
			Project project = new Project();
			
			(new FtgSerializer()).read(project, in);
			in.close();
			this.project = project;
			this.project.setFilename(filename);
			this.project.setFileLastModified(file.lastModified());
			this.project.update();
			updateTitle();
			initialize();
		} catch (IOException e) {
			e.printStackTrace();
			JOptionPane.showMessageDialog(null, "FreeGanttプロジェクトファイルとして読み込めませんでした。", "フォーマットエラー", JOptionPane.ERROR_MESSAGE);
		} catch (SAXException e) {
			e.printStackTrace();
			JOptionPane.showMessageDialog(null, "FreeGanttプロジェクトファイルとして読み込めませんでした。", "フォーマットエラー", JOptionPane.ERROR_MESSAGE);
		} catch (FileFormatException e) {
			e.printStackTrace();
			JOptionPane.showMessageDialog(null, "FreeGanttプロジェクトファイルとして読み込めませんでした。", "フォーマットエラー", JOptionPane.ERROR_MESSAGE);
		} finally {
			try {
				if (in != null)
					in.close();
			} catch (IOException e1) {
			}
		}
	}
	
	/**
	 * 編集内容が更新されている場合にダイアログを表示し、
	 * キャンセルされたらtrueを返す
	 * @return
	 */
	protected boolean confirmDiscard() {
		if (!project.isModified()) return false;

		int ret = JOptionPane.showConfirmDialog(this, 
				Resource.get("loadModifiedConfirmMessage"), 
				Resource.get("loadModifiedConfirmTitle"),
				JOptionPane.OK_CANCEL_OPTION);
		return ret == JOptionPane.CANCEL_OPTION;
	}
	
	public void loadData() {
		if (confirmDiscard()) return;
		
		JFileChooser chooser = new JFileChooser();
		FileNameExtensionFilter filter = new FileNameExtensionFilter(
				Resource.get("dialogFileType"), "fgt");
		chooser.setFileFilter(filter);
		int ret = chooser.showOpenDialog(this);
		if (ret == JFileChooser.APPROVE_OPTION) {
			String filename = chooser.getSelectedFile().getAbsolutePath();
			loadData(filename);
		}
	}
	
	public Image getApplicationIconImage() {
		return iconImage;
	}

	public TaskLineDataPane getTaskLineDataPane() {
		if (chartPane == null || chartPane.getTaskLineRootPane() == null) return null;
		return chartPane.getTaskLineRootPane().getTaskLineDataPane();
	}

	public JViewport getDataViewport() {
		if (chartPane == null || chartPane.getTaskLineRootPane() == null) return null;
		return chartPane.getTaskLineRootPane().getDataViewport();
	}

	public TaskListTable getTaskListTable() {
		if (chartPane == null || chartPane.getTaskListRootPane() == null) return null;
		return chartPane.getTaskListRootPane().getTaskListTable();
	}

	public TaskListRootPane getTaskListRootPane() {
		if (chartPane == null) return null;
		return chartPane.getTaskListRootPane();
	}
	
	public GanttChartPane getGanttChartPane() {
		return chartPane;
	}

	public TaskLineRootPane getTaskLineRootPane() {
		return chartPane.getTaskLineRootPane();
	}
	
	public TaskLineHeaderPane getTaskLineHeaderPane() {
		if (chartPane == null || chartPane.getTaskListRootPane() == null) return null;
		return chartPane.getTaskLineRootPane().getTaskLineHeaderPane();
	}
	
	public void openConfig(int type) {
		configPane.setSelectedIndex(type);
		int location = configSplitPane.getWidth() - 200;
		configSplitPane.setDividerLocation(location);
	}

	public void updateLocale(Locale locale) {
		setLocale(locale);
		Resource.makeInstance(getLocale());
		project.updateTableModel();
		initialize();
	}

	public void configurePage() {
		PrinterJob printerJob = PrinterJob.getPrinterJob();
		PageFormat pageFormat = printerJob.defaultPage();
		pageFormat = printerJob.pageDialog(getProject().getPrint().getPageFormat());
		project.getController().setPageFormat(project, pageFormat);
	}

	public void print() {
		PrinterJob job = PrinterJob.getPrinterJob();
		GanttChartPrintable printable = new GanttChartPrintable(this);
		job.setPrintable(printable, getProject().getPrint().getPageFormat());
		job.setPageable(printable);
		
		if (job.printDialog()) {
			try {
				job.print();
			} catch (PrinterException e1) {
				e1.printStackTrace();
			}
		}
	}

	public void changeCalendarMode() {
		int mode = project.isCalendarModeDate() ? Project.CALENDAR_MODE_WEEK : Project.CALENDAR_MODE_DATE;
		project.getController().setCalendarMode(mode);
	}
	
	public WindowListener createWindowListener() {
		return new AppWindowListener();
	}
	
	public void savePreference() {
		Initialize.getInstance().putValue("applicationWidth", getWidth());
		Initialize.getInstance().putValue("applicationHeight", getHeight());
		Initialize.getInstance().save();
	}

	public void openVersionDialog() {
		Dialog dialog = new VersionDialog(this);
		dialog.setVisible(true);
	}

	public void openHelpDocument() {
		Desktop desktop = Desktop.getDesktop();
        try {
			desktop.browse(new URI(Resource.get("helpDocumentURL")));
		} catch (IOException ee) {
		} catch (URISyntaxException ee) {
		}
	}

	public void openSiteRoot() {
		Desktop desktop = Desktop.getDesktop();
        try {
			desktop.browse(new URI(Resource.get("siteRootURL")));
		} catch (IOException ee) {
		} catch (URISyntaxException ee) {
		}
	}
	
	class AppWindowListener implements WindowListener {

		@Override
		public void windowOpened(WindowEvent e) {
		}

		@Override
		public void windowClosing(WindowEvent e) {
			if (project.isModified() && confirmDiscard()) {
				return;
			}
			savePreference();
			dispose();
		}

		@Override
		public void windowClosed(WindowEvent e) {
		}

		@Override
		public void windowIconified(WindowEvent e) {
		}

		@Override
		public void windowDeiconified(WindowEvent e) {
		}

		@Override
		public void windowActivated(WindowEvent e) {
		}

		@Override
		public void windowDeactivated(WindowEvent e) {
		}
		
	}
	
	public class ProjectModifiedHandler implements ProjectModifiedListener {
		@Override
		public void projectModified() {
			updateTitle();
		}
	}
}
