package jp.sourceforge.freegantt.print;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.awt.print.PageFormat;
import java.awt.print.PrinterException;

import javax.swing.JDialog;
import javax.swing.JPanel;

import jp.sourceforge.freegantt.locale.Resource;
import jp.sourceforge.freegantt.swing.Application;

public class PreviewDialog extends JDialog {
	private static final long serialVersionUID = 6695686076348590807L;
	
	Application app;
	PageFormat pageFormat;
	int page = 0;
	int pageCount = 0;
	BufferedImage image;
	PreviewIconMenu itemMenuPanel;
	ImagePanel imagePanel;
	
	public PreviewDialog(Application app) {
		super(app, true);
		this.app = app;
		
		imagePanel = new ImagePanel();
		imagePanel.setBounds(0, 0, 100, 100);
		imagePanel.setPreferredSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));

		getContentPane().add(imagePanel, BorderLayout.CENTER);
		setTitle(Resource.get("previewTitle"));
		setSize(640, 480);
		setIconImage(app.getApplicationIconImage());
		
		itemMenuPanel = new PreviewIconMenu(app, this);
		getContentPane().add(itemMenuPanel, BorderLayout.NORTH);
		
		setJMenuBar(new PreviewMenu(app, this));
		
		pageCount = app.getTaskLineDataPane().getPrintRange().getPrintRanges().size();
		
		updatePageFormat(app.getProject().getPrint().getPageFormat());
		updatePage();
		renderPage(page);
	}
	
	public int getPage() {
		return page;
	}
	public void setPage(int page) {
		this.page = page;
	}
	public int getPageCount() {
		return pageCount;
	}

	public void updatePageFormat(PageFormat pageFormat) {
		this.pageFormat = pageFormat;
		image = new BufferedImage((int)pageFormat.getWidth(), (int)pageFormat.getHeight(), BufferedImage.TYPE_INT_BGR);
		
		Graphics g = null;
		try {
			g = image.getGraphics();
			g.setColor(Color.white);
			g.fillRect(0, 0, image.getWidth(this), image.getHeight(this));
		} finally {
			if (g != null) g.dispose();
		}
		imagePanel.setImage(image);
		repaint();
	}
	
	public void renderPage(int pageIndex) {
		GanttChartPrintable printable = new GanttChartPrintable(app);
		Graphics g = null;
		try {
			g = image.getGraphics();
			g.setColor(Color.white);
			g.fillRect(0, 0, image.getWidth(this), image.getHeight(this));
			printable.print(g, pageFormat, pageIndex);
		} catch (PrinterException e) {
			e.printStackTrace();
		} finally {
			if (g != null) g.dispose();
		}
		imagePanel.repaint();
	}

	public void updatePage() {
		itemMenuPanel.getPageLabel().setText("" + Math.min(page + 1, pageCount) + " / " + pageCount);
		itemMenuPanel.getPrevPageButton().setEnabled(page > 0);
		itemMenuPanel.getPostPageButton().setEnabled(page < pageCount - 1);
	}

	class ImagePanel extends JPanel {
		private static final long serialVersionUID = 7267024727103791877L;
		
		Image image;

		public Image getImage() {
			return image;
		}

		public void setImage(Image image) {
			this.image = image;
		}

		@Override
		protected void paintComponent(Graphics g) {
			super.paintComponent(g);
			
			if (image == null) return;
			double scaleX = (double)getWidth() / (double)image.getWidth(this);
			double scaleY = (double)getHeight() / (double)image.getHeight(this);
			double scale = Math.min(scaleX, scaleY);
			int scaledWidth = (int)((double)image.getWidth(this) * scale);
			int scaledHeight = (int)((double)image.getHeight(this) * scale);
			
			Graphics2D g2d = (Graphics2D)g;
			g2d.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
			AffineTransform transform = new AffineTransform();
			transform.translate( (getWidth() - scaledWidth)/2, (getHeight() - scaledHeight)/2);
			transform.scale(scale, scale);
			g2d.drawImage(image, transform, this);
		}
	}
	
}
