package jp.sourceforge.freegantt.data;

import java.awt.Dimension;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import javax.swing.undo.UndoManager;

import jp.sourceforge.freegantt.data.controller.ProjectController;
import jp.sourceforge.freegantt.data.model.HolidayTableModel;
import jp.sourceforge.freegantt.data.model.MemberTableModel;
import jp.sourceforge.freegantt.data.model.ProjectInfoModel;
import jp.sourceforge.freegantt.data.model.ProjectModifiedModel;
import jp.sourceforge.freegantt.data.model.ProjectViewModel;
import jp.sourceforge.freegantt.data.model.TaskListTableModel;
import jp.sourceforge.freegantt.data.model.TaskMemberComboBoxModel;
import jp.sourceforge.freegantt.data.undo.StatefullUndoManager;
import jp.sourceforge.freegantt.util.CalendarUtil;

public class Project {
	public static final int CALENDAR_MODE_DATE = 0;
	public static final int CALENDAR_MODE_WEEK = 1;
	
	/** ファイル名 */
	String filename;
	/** プロジェクト名 */
	String name = "";
	/** プロジェクト概要 */
	String summary = "";
	/** 担当者 */
	List<Member> members = new ArrayList<Member>();
	/** タスク */
	List<Task> tasks = new ArrayList<Task>();
	/** 固定休日 */
	List<Integer> fixedHolidays = new ArrayList<Integer>();
	/** 個別休日 */
	List<Calendar> additionalHolidays = new ArrayList<Calendar>();
	/** セル描画サイズ */
	Dimension cellSize = new Dimension(14, 16);
	/** カレンダーモード */
	int calendarMode = CALENDAR_MODE_DATE;
	/** 画面表示 */
	View view = new View();
	/** 印刷設定 */
	Print print = new Print();
	
	// 以下は自動計算される
	/**　データ範囲開始日 */
	Calendar chartFromDate;
	/** データ範囲終了日 */
	Calendar chartToDate;
	/** ファイル最終更新日時 */
	long fileLastModified = Long.MAX_VALUE;
	/** ファイル更新フラグ */
	boolean modified = false;
	
	protected ProjectController controller;
	protected UndoManager undoManager;
	
	protected TaskListTableModel taskTableModel;
	protected MemberTableModel memberTableModel;
	protected HolidayTableModel holidayTableModel;
	protected TaskMemberComboBoxModel taskMemberComboBoxModel;
	protected ProjectInfoModel projectInfoModel;
	protected ProjectViewModel projectViewModel;
	protected ProjectModifiedModel projectModifiedModel;
	
	public void setCellSize(Dimension cellSize) {
		this.cellSize = cellSize;
	}
	public ProjectModifiedModel getProjectModifiedModel() {
		return projectModifiedModel;
	}
	public boolean isModified() {
		return modified;
	}
	public void setModified(boolean modified) {
		this.modified = modified;
	}
	public ProjectViewModel getProjectViewModel() {
		return projectViewModel;
	}
	public ProjectInfoModel getProjectInfoModel() {
		return projectInfoModel;
	}
	public ProjectController getController() {
		return controller;
	}
	public UndoManager getUndoManager() {
		return undoManager;
	}
	public long getFileLastModified() {
		return fileLastModified;
	}
	public void setFileLastModified(long fileLastModified) {
		this.fileLastModified = fileLastModified;
	}
	public void setCalendarMode(int mode) {
		this.calendarMode = mode;
	}
	public int getCalendarMode() {
		return calendarMode;
	}
	public boolean isCalendarModeDate() {
		return calendarMode == CALENDAR_MODE_DATE;
	}
	public Print getPrint() {
		return print;
	}
	public void setPrint(Print print) {
		this.print = print;
	}
	public View getView() {
		return view;
	}
	public void setView(View view) {
		this.view = view;
	}
	public Dimension getCellSize() {
		return cellSize;
	}
	public String getFilename() {
		return filename;
	}
	public void setFilename(String filename) {
		this.filename = filename;
	}
	public String getName() {
		return name;
	}
	public void setName(String name) {
		this.name = name;
	}
	public String getSummary() {
		return summary;
	}
	public void setSummary(String summary) {
		this.summary = summary;
	}
	public List<Member> getMembers() {
		return members;
	}
	public void setMembers(List<Member> members) {
		this.members = members;
	}
	public List<Task> getTasks() {
		return tasks;
	}
	public void setTasks(List<Task> tasks) {
		this.tasks = tasks;
	}
	public List<Integer> getFixedHolidays() {
		return fixedHolidays;
	}
	public void setFixedHolidays(List<Integer> fixedHolidays) {
		this.fixedHolidays = fixedHolidays;
	}
	public List<Calendar> getAdditionalHolidays() {
		return additionalHolidays;
	}
	public void setAdditionalHolidays(List<Calendar> additionalHolidays) {
		this.additionalHolidays = additionalHolidays;
	}
	public TaskListTableModel getTaskTableModel() {
		return taskTableModel;
	}
	public MemberTableModel getMemberTableModel() {
		return memberTableModel;
	}
	public HolidayTableModel getHolidayTableModel() {
		return holidayTableModel;
	}
	public TaskMemberComboBoxModel getTaskMemberComboBoxModel() {
		return taskMemberComboBoxModel;
	}
	public Calendar getChartRangeFromDate() {
		return chartFromDate;
	}
	public Calendar getChartRangeToDate() {
		return chartToDate;
	}
	
	public Project() {
		updateTableModel();
	}
	
	public void updateTableModel() {
		this.controller = new ProjectController(this);
		this.undoManager = new StatefullUndoManager();
		
		this.taskTableModel 			= new TaskListTableModel(this);
		this.memberTableModel 			= new MemberTableModel(this);
		this.holidayTableModel 			= new HolidayTableModel(this);
		this.taskMemberComboBoxModel 	= new TaskMemberComboBoxModel(this);
		this.projectInfoModel 			= new ProjectInfoModel(this);
		this.projectViewModel 			= new ProjectViewModel(this);
		this.projectModifiedModel		= new ProjectModifiedModel(this);
	}
	
	public boolean isHoliday(Calendar calendar) {
		int week = calendar.get(Calendar.DAY_OF_WEEK);
		if (fixedHolidays.contains(week)) return true;
		for (Calendar holiday: additionalHolidays) {
			if (CalendarUtil.dateEquals(calendar, holiday)) return true;
		}
		return false;
	}
	
	public boolean isAdditionalHoliday(Calendar calendar) {
		return additionalHolidays.contains(calendar);
	}
	
	public List<Task> getRestrictionSrcTasks(Task task) {
		List<Task> result = new ArrayList<Task>();
		if (task == null) return result;
		
		for (Task src: getTasks()) {
			if (src.getRestrictions().contains(task)) {
				result.add(src);
			}
		}
		return result;
	}
	
	/**
	 * 子タスクを集めて返す
	 * @param targetTask
	 */
	public List<Task> getChildTasks(Task targetTask) {
		List<Task> childTasks = new ArrayList<Task>();
		int index = getIndexByTask(targetTask);
		index ++;
	
		while (index < tasks.size()) {
			Task task = getTaskAtIndex(index);
			if (task.getLevel() <= targetTask.getLevel()) return childTasks;
			childTasks.add(task);
			index++;
		}
		return childTasks;
	}
	
	/**
	 * 親タスクを返す
	 * @param targetTask 見つからなければnull
	 * @return
	 */
	private Task getParentTask(Task targetTask) {
		if (targetTask.getLevel() == 0) return null;
		
		int index = getIndexByTask(targetTask);
		index--;
		while (index >= 0) {
			Task task = getTaskAtIndex(index);
			if (task.getLevel() < targetTask.getLevel()) return task;
			index--;
		}
		
		return null;
	}
	
	public List<Integer> collectRestrictionIndexes(Task targetTask) {
		List<Integer> list = new ArrayList<Integer>();
		if (targetTask == null) return list;
		for (Task task: targetTask.getRestrictions()) {
			int index = getIndexByTask(task);
			if (index >= 0) list.add(index);
		}
		return list;
	}
	
	public List<Task> collectTasksAtIndexes(List<Integer> indexes) {
		List<Task> tasks = new ArrayList<Task>();
		for (int index: indexes) {
			Task task = getTaskAtIndex(index);
			if (task != null) tasks.add(task);
		}
		return tasks;
	}

	public void update() {
		
		// グループフラグを更新する
		for (Task task: tasks) {
			getController().setTaskGroup(!getChildTasks(task).isEmpty(), getIndexByTask(task));
		}

		// タスクをlevelの降順にソートする
		List<Task> sortedTasks = new ArrayList<Task>(tasks);
		Collections.sort(sortedTasks, new Comparator<Task>() {
			@Override
			public int compare(Task lhs, Task rhs) {
				if (lhs == null || rhs == null) return -1;
				if (lhs.getLevel() == rhs.getLevel()) return 0;
				return (lhs.getLevel() > rhs.getLevel()) ? -1 : 1;
			}
		});
		
		// 階層の深いものから実質工期を求める
		for (Task task: sortedTasks) {
			if (task.isGroup()) {
				updateParentPeriod(task);
			} else if (task.isMilestone()) {
				updateMilestone(task);
			} else {
				updateChildrenPeriod(task, false);
			}
		}
		
		// クリティカルパスを計算する
		updateCliticalPath();
		
		// フォールドによる表示状態を計算する
		updateAllVisibility();
	}
	
	public void updateMilestone(Task task) {
		Task updateTask = task.clone();
		
		updateTask.setRealPeriod(0);
		
		List<Task> srcs = getRestrictionSrcTasks(task);
		if (srcs.size() == 0) {
			updateTask.setCompletion(100);
		} else {
			int completeTaskCount = 0;
			for (Task src: srcs) {
				if (src.getCompletion() == 100) completeTaskCount++;
			}
			updateTask.setCompletion(completeTaskCount * 100 / srcs.size());
		}
		
		if (!updateTask.equals(task)) {
			getController().setTaskAtIndex(updateTask, getIndexByTask(task));
		}
	}

	public void updateChildrenPeriod(Task task, boolean isFloating) {
		if (task.isGroup()) return;
		if (task.getStartDate() == null) return;
		if (task.getPeriod() == null) task.setPeriod(0);
		Task updateTask = (isFloating ? task : task.clone());
		
		int leftPeriod = updateTask.getPeriod();
		int realPeriod = 0;
		Calendar now = (Calendar)updateTask.getStartDate().clone();
		
		// 無限ループを防ぐためにリミッタを使用する
		for (int i=0; i<365 && leftPeriod > 0.0; i++) {
			if (!isHoliday(now)) {
				leftPeriod -= 1.0;
			}
			realPeriod += 1.0;
			now.add(Calendar.DATE, 1);
		}
		
		updateTask.setRealPeriod(realPeriod);
		
		if (!isFloating && !updateTask.equals(task)) {
			getController().setTaskAtIndex(updateTask, getIndexByTask(task));
		}
	}

	
	/**
	 * 親タスクの開始日と工期は子タスクの全日程を含む長さとする。
	 * 開始日と工期が無い子タスクは無視され、
	 * 全ての子タスクが無視された場合は親タスクに工期が設定されないものとする。
	 * @param task
	 */
	public void updateParentPeriod(Task task) {
		if (!task.isGroup()) return;
		Task updateTask = task.clone();
		updateTask.setStartDate(null);
		updateTask.setPeriod(null);
		updateTask.setRealPeriod(null);
		
		{ // 工期の計算
			long from = Long.MAX_VALUE;
			long to = Long.MIN_VALUE;
			
			List<Task> children = getChildTasks(task);
			for (Task child: children) {
				if (child.getStartDate() == null) continue;
				long childFrom = child.getStartDate().getTimeInMillis();
				if (childFrom < from) from = childFrom;
				if (childFrom > to) to = childFrom;
				
				if (child.getRealPeriod() == null) continue;
				long childTo = childFrom + child.getRealPeriod().longValue() * 86400000;
				if (childTo < from) from = childTo;
				if (childTo > to) to = childTo;
			}
			
			if (to < from) return;
			
			Calendar startDate = Calendar.getInstance();
			startDate.setTimeInMillis(from);
			updateTask.setStartDate(startDate);
			updateTask.setPeriod(CalendarUtil.subDate(CalendarUtil.toDateCalendar(to), CalendarUtil.toDateCalendar(from)));
			updateTask.setRealPeriod(updateTask.getPeriod());
		}
		
		{ // 達成率の計算
			int wholePeriod = 0;
			double completePeriod = 0;
			List<Task> children = getChildTasks(task);
			for (Task child: children) {
				if (child.isGroup()) continue;
				if (child.getPeriod() == null || child.getPeriod() <= 0) continue;
				wholePeriod += child.getPeriod();
				completePeriod += (double)(child.getPeriod() * child.getCompletion()) / 100.0;
			}
			updateTask.setCompletion((int)(completePeriod * 100.0 / wholePeriod));
		}
		
		if (!updateTask.equals(task)) {
			System.out.println("Group Task updated: " + updateTask.getName());
			// この仲ではupdateが呼ばれない
			getController().setTaskAtIndex(updateTask, getIndexByTask(task));
		}
	}
	
	/**
	 * 全ての可視タスクを返す
	 * @return
	 */
	public List<Task> getVisibleTasks() {
		List<Task> visibleTasks = new ArrayList<Task>();
		for (Task task: tasks) {
			if (task.isVisible()) visibleTasks.add(task);
		}
		return visibleTasks;
	}
	
	/**
	 * インデックス番号に対応したタスクを返す
	 * @param index
	 * @return
	 */
	public Task getTaskAtIndex(int index) {
		if (index < 0 || tasks.size() <= index) return null;
		return tasks.get(index);
	}
	
	/**
	 * 行番号に対応したタスクを返す
	 * @param row
	 * @return
	 */
	public Task getTaskAtRow(int row) {
		if (row < 0) return null;
		int index = getIndexFromRow(row);
		if (index < 0 || index >= tasks.size()) return null;
		return getTaskAtIndex(index);
	}
	
	/**
	 * 行番号からインデックス番号を求める
	 * @param row
	 * @return
	 */
	public int getIndexFromRow(int row) {
		if (row < 0) return -1;
		for (int index=0, rowCount=-1; rowCount<=row; index++) {
			Task task = getTaskAtIndex(index);
			if (task == null || task.isVisible()) rowCount++;
			if (rowCount == row) return index;
		}
		return -1;
	}
	
	/**
	 * タスクから行番号を求めて返す
	 * @param targetTask
	 * @return
	 */
	public int getRowByTask(Task targetTask) {
		if (targetTask == null) return -1;
		for (int index=0, rowCount=-1; index<tasks.size(); index++) {
			Task task = getTaskAtIndex(index);
			if (task == null || task.isVisible()) rowCount++;
			if (task == targetTask) return rowCount;
		}
		return -1;
	}
	
	/**
	 * タスクからインデックス番号を求めて返す
	 * @param targetTask
	 * @return
	 */
	public int getIndexByTask(Task targetTask) {
		for (int index=0; index<tasks.size(); index++) {
			Task task = getTaskAtIndex(index);
			if (task == targetTask) return index;
		}
		return -1;
	}

	public Calendar getFirstDate() {
		Calendar firstDate = Calendar.getInstance();
		boolean found = false;
		firstDate.setTimeInMillis(Long.MAX_VALUE);
		for (Task task: tasks) {
			if (task.getStartDate() != null) {
				if (firstDate.getTimeInMillis() > task.getStartDate().getTimeInMillis()) {
					found = true;
					firstDate = (Calendar)(task.getStartDate().clone());
				}
			}
		}
		if (!found) return null;
		return firstDate;
	}
	
	public int getWholePeriod() {
		Calendar firstDate = getFirstDate();
		Calendar lastDate = Calendar.getInstance();
		boolean found = false;
		lastDate.setTimeInMillis(Long.MIN_VALUE);
		if (firstDate == null) return 0;
		for (Task task: tasks) {
			Calendar endDate = task.getEndDate();
			if (endDate != null) {
				if (lastDate.getTimeInMillis() < endDate.getTimeInMillis()) {
					found = true;
					lastDate = endDate;
				}
			}
		}
		if (!found) return 0;
		return CalendarUtil.subDate(lastDate, firstDate);
	}
	
	public int getRowCount() {
		int rowCount = 0;
		for (int index=0; index<tasks.size(); index++) {
			Task task = getTaskAtIndex(index);
			if (task == null || task.isVisible()) rowCount ++;
		}
		return rowCount;
	}
	
	/**
	 * リソースを削除し、関連したタスクの担当者をなしにする
	 * @param index
	 */
	public void removeMember(int index) {
		if (index >= members.size()) return;

		Member member = members.get(index);
		for (Task task: tasks) {
			if (task.getMember() == member) {
				task.setMember(null);
			}
		}
		members.remove(index);
		getMemberTableModel().fireTableChanged();
		getTaskTableModel().fireTableChanged();
	}
	
	/**
	 * 現在のタスクデータから描画するべき範囲の初期値を決める
	 * タスクが空であれば本日を基準とする。
	 */
	public void updateChartRange() {
		chartFromDate = getFirstDate();
		if (chartFromDate == null) chartFromDate = CalendarUtil.toDateCalendar(Calendar.getInstance());
		chartToDate = getFirstDate();
		int period = getWholePeriod();
		if (chartToDate != null) chartToDate.add(Calendar.DATE, period);
		if (chartToDate == null) chartToDate = CalendarUtil.toDateCalendar(Calendar.getInstance());
	}
	
	/**
	 * クリティカルパスを再計算し、フラグを立てる
	 */
	public void updateCliticalPath() {
		List<Task> criticalTasks = new ArrayList<Task>();
		List<Task> endTasks = getEndTasks();
		criticalTasks.addAll(endTasks);
		for (Task endTask: endTasks) {
			criticalTasks.addAll( traceCliticalTask(endTask) );
		}
		
		for (Task task: tasks) {
			getController().setTaskCriticalPath(criticalTasks.contains(task), getIndexByTask(task));
			task.setCriticalPath(criticalTasks.contains(task));
		}
	}
	
	/**
	 * 最後に終わるタスクを返す
	 * @return
	 */
	protected List<Task> getEndTasks() {
		List<Task> endTasks = new ArrayList<Task>();
		
		long lastEndTime = 0;
		for (Task task: tasks) {
			Calendar endDate = task.getEndDate();
			if (endDate == null) continue;
			
			long endTime = endDate.getTimeInMillis();
			if (lastEndTime < endTime) lastEndTime = endTime;
		}
		for (Task task: tasks) {
			Calendar endDate = task.getEndDate();
			if (endDate == null) continue;
			
			if (endDate.getTimeInMillis() == lastEndTime) endTasks.add(task);
		}
		
		return endTasks;
	}
	
	/**
	 * リンクされたタスクをたどってクリティカルパスフラグを立てる
	 * @param endTask
	 */
	protected List<Task> traceCliticalTask(Task endTask) {
		List<Task> criticalTasks = new ArrayList<Task>();
		criticalTasks.add(endTask);

		List<Task> restrictionTasks = new ArrayList<Task>();
		for (Task task: tasks) {
			if (task == endTask) continue;
			Calendar endDate = task.getEndDate();
			if (endDate == null) continue;
			if (!task.getRestrictions().contains(endTask)) continue;
			
			System.out.println(CalendarUtil.subDateUsingHoliday(this, endTask.startDate, endDate));
			if (CalendarUtil.subDateUsingHoliday(this, endTask.startDate, endDate) <= 0) {
				restrictionTasks.add(task);
			}
		}
		
		for (Task task: restrictionTasks) {
			criticalTasks.addAll( traceCliticalTask(task) );
		}
		
		return criticalTasks;
	}
	
	/**
	 * 指定した行のタスクのフォールド状態を切り替える
	 * @param row
	 */
	public void toggleTaskFold(int row) {
		Task task = getTaskAtRow(row);
		if (task == null) return;
		
		if (task.isOpened()) {
			task.setOpened(false);
			updateChildTasksVisibility(task);
		} else {
			task.setOpened(true);
			updateChildTasksVisibility(task);
		}
	}
	
	/**
	 * 全てのタスクの表示状態を更新する
	 */
	public void updateAllVisibility() {
		for (Task task: tasks) {
			updateTaskVisibility(task);
		}
	}
	
	/**
	 * 全ての子タスクの表示状態を更新する
	 * @param targetTask
	 */
	private void updateChildTasksVisibility(Task targetTask) {
		List<Task> childTasks = getChildTasks(targetTask);
		for (Task child: childTasks) {
			updateTaskVisibility(child);
		}
	}
	
	/**
	 * 任意のタスクの表示状態を更新する
	 * @param targetTask
	 */
	private void updateTaskVisibility(Task targetTask) {
		Task parent = getParentTask(targetTask);
		while (parent != null) {
			if (!parent.isOpened()) {
				getController().setTaskVisible(false, getIndexByTask(targetTask));
				return;
			}
			parent = getParentTask(parent);
		}
		getController().setTaskVisible(true, getIndexByTask(targetTask));
	}
	
	public Member findFirstMemberByName(String memberName) {
		if (memberName == null) return null;
		for  (Member member: members) {
			if (memberName.equals(member.getName())) return member;
		}
		return null;
	}
}
