package jp.sourceforge.freegantt.swing;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.DefaultCellEditor;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTable;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.TableColumnModelEvent;
import javax.swing.event.TableColumnModelListener;
import javax.swing.event.TableModelEvent;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import jp.sourceforge.freegantt.data.Column;
import jp.sourceforge.freegantt.data.Project;
import jp.sourceforge.freegantt.data.Task;
import jp.sourceforge.freegantt.locale.Resource;
import jp.sourceforge.freegantt.swing.HeaderChangedListener.HeaderChangedEvent;

public class TaskListTable extends JTable implements MouseListener {
	private static final long serialVersionUID = -2059641956588584594L;

	Application app;
	Project project;
	TaskListTableModel tableModel;
	TaskNameCellRenderer taskNameCellRenderer;
	List<HeaderChangedListener> headerChangedListeners = new ArrayList<HeaderChangedListener>();
	boolean initialized = false;
	
	Color borderColor = new Color(0xC0, 0xC0, 0xC0);
	
	public TaskListTableModel getTaskListTableModel() {
		return tableModel;
	}
	
	public void addHeaderChangedListener(HeaderChangedListener l) {
		headerChangedListeners.add(l);
	}
	
	public void removeHeaderChangedListener(HeaderChangedListener l) {
		headerChangedListeners.remove(l);
	}
	
	public void fireHeaderChangedEvent() {
		for (HeaderChangedListener listener: headerChangedListeners) {
			listener.headerChanged(new HeaderChangedEvent(this));
		}
	}

	public TaskListTable(Application app) {
		super();
		this.app = app;
		this.project = app.getProject();
		
		setGridColor(borderColor);
		setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);
		addMouseListener(this);
		getTableHeader().addMouseListener(new HeaderMouseListener());
		
		tableModel = project.getTaskTableModel();
		setModel(tableModel);
		TableColumnModel columnModel = getColumnModel();
		columnModel.addColumnModelListener(new ColumnModelChangedListener());
		
		String[] columnKeys = tableModel.getColumnKeys();
		for (int i=0; i<columnKeys.length; i++) {
			TableColumn column = columnModel.getColumn(i);
			column.setIdentifier(columnKeys[i]);
			column.setMinWidth(0);
			column.addPropertyChangeListener(new ColumnSizeChangeListener());
		}

		updateColumnArrangement();
		
		columnModel.getColumn(2).setCellRenderer(new TaskMemberCellRenderer());
		columnModel.getColumn(2).setCellEditor(new DefaultCellEditor(new TaskMemberCellEditor(project)));
		taskNameCellRenderer = new TaskNameCellRenderer(project);
		columnModel.getColumn(0).setCellRenderer(getTableHeader().getDefaultRenderer());
		columnModel.getColumn(0).setMaxWidth(30);
		columnModel.getColumn(0).setMinWidth(30);
		columnModel.getColumn(1).setCellRenderer(taskNameCellRenderer);
		
		initialized = true;
	}
	
	private void updateColumnArrangement() {
		List<Column> columns = project.getView().getColumns();
		int index = 1;
		for (Column column: columns) {
			getColumnModel().moveColumn(getColumnModel().getColumnIndex(column.getKey()), index);
			getColumn(column.getKey()).setPreferredWidth(column.getWidth());
			index++;
		}
	}

	@Override
	public void tableChanged(TableModelEvent e) {
		super.tableChanged(e);
		
		if (app == null) return;
		
		// チャートペインの高さをタスクリストテーブルの高さと同じに設定する
		TaskLineDataPane taskLineDataPane = app.getTaskLineDataPane();
		if (taskLineDataPane != null) {
			taskLineDataPane.setSize(new Dimension(taskLineDataPane.getWidth(), getTaskListTableModel().getRowCount() * 16));
			taskLineDataPane.setPreferredSize(new Dimension(taskLineDataPane.getWidth(), getTaskListTableModel().getRowCount() * 16));
			System.out.println("DataPane setPreferredSize: " + taskLineDataPane.getWidth() + "," +  getTaskListTableModel().getRowCount() * 16);
		}

		
		if (app.getTaskListRootPane() != null) {
			app.getTaskListRootPane().getViewport().validate();
		}
		
		if (app.getTaskLineDataPane() != null) {
			app.getTaskLineDataPane().repaint();
		}
		
		if (app.getTaskListPane() != null) {
			app.getTaskListPane().repaint();
		}
	}

	@Override
	public void mouseClicked(MouseEvent e) {
	}

	@Override
	public void mousePressed(MouseEvent e) {
		if (e.isPopupTrigger()) triggerPopup(e);
	}

	@Override
	public void mouseReleased(MouseEvent e) {
		if (e.isPopupTrigger()) triggerPopup(e);
	}

	private void triggerPopup(MouseEvent e) {
		final int row = TaskListTable.this.rowAtPoint(e.getPoint());
		if (row < 0) return;
		
		boolean found = false;
		int[] selectedRows = getSelectedRows();
		for (int s: selectedRows) {
			if (s == row) found = true;
		}
		if (!found) {
			setRowSelectionInterval(row, row);
			repaint();
		}
		
		final Task task = project.findTask(row);
		boolean isAvailableLevelUp = (task != null);
		boolean isAvailableLevelDown = (task != null && task.getLevel() > 0);
		
		JPopupMenu menu = new JPopupMenu();
		JMenuItem item = new JMenuItem(Resource.get("listRemoveTask"));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				int deleted = 0;
				int[] selectedRows = getSelectedRows();
				for (int s: selectedRows) {
					project.removeRow(s - deleted);
					deleted ++;
				}
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("listInsertTask"));
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				int inserted = 0;
				int chained = 0;
				int preSelected = -1;
				int[] selectedRows = getSelectedRows();
				for (int s: selectedRows) {
					if (preSelected + 1 == s) chained ++; else chained = 0;
					project.insertRow(s + inserted - chained);
					inserted ++;
					preSelected = s;
				}
			}
		});
		menu.add(item);
		menu.addSeparator();
		item = new JMenuItem(Resource.get("listLevelUp"));
		item.setEnabled(isAvailableLevelUp);
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				task.setLevel(task.getLevel() + 1);
				project.update();
				tableModel.fireTableChanged();
			}
		});
		menu.add(item);
		item = new JMenuItem(Resource.get("listLevelDown"));
		item.setEnabled(isAvailableLevelDown);
		item.addActionListener(new ActionListener(){
			@Override
			public void actionPerformed(ActionEvent e) {
				task.setLevel(task.getLevel() - 1);
				project.update();
				tableModel.fireTableChanged();
			}
		});
		menu.add(item);
		menu.show(this, e.getX(), e.getY());
	}

	@Override
	public void mouseEntered(MouseEvent e) {
	}

	@Override
	public void mouseExited(MouseEvent e) {
	}

	@Override
	public void doLayout() {
		JTableHeader header = getTableHeader();
		TableColumn resizingColumn = header.getResizingColumn();
		if (resizingColumn != null) {
			resizingColumn.setPreferredWidth(resizingColumn.getWidth());
		}
		
		TableColumnModel model = getColumnModel();
		for (int i=0; i<model.getColumnCount(); i++) {
			TableColumn column = model.getColumn(i);
			column.setWidth(column.getPreferredWidth());
		}
	}
	
	class HeaderMouseListener implements MouseListener {
		@Override
		public void mouseClicked(MouseEvent e) {
		}

		@Override
		public void mousePressed(MouseEvent e) {
			if (e.isPopupTrigger()) {
				triggerPopup(e);
			}
		}

		@Override
		public void mouseReleased(MouseEvent e) {
			if (e.isPopupTrigger()) {
				triggerPopup(e);
			}
		}
		
		public void triggerPopup(MouseEvent e) {
			JPopupMenu menu = new JPopupMenu();
			String columnNames[] = getTaskListTableModel().getColumnNames();
			for (int index=0; index<columnNames.length; index++) {
				if (index == 0) continue;
				JMenuItem item = new JMenuItem(columnNames[index]);
				item.addActionListener(new ColumnNameActionListener(index));
				menu.add(item);
			}
			menu.show(e.getComponent(), e.getX(), e.getY());
		}

		@Override
		public void mouseEntered(MouseEvent e) {
		}

		@Override
		public void mouseExited(MouseEvent e) {
		}
	}
	
	class ColumnNameActionListener implements ActionListener {
		int index;
		
		public ColumnNameActionListener(int index) {
			this.index = index;
		}
		
		@Override
		public void actionPerformed(ActionEvent e) {
			String key = tableModel.getColumnKey(index);
			Column column = project.getView().getColumn(key);
			if (column == null) return;	
			column.setWidth(column.getWidth() == 0 ? project.getView().getDefaultColumn(key).getWidth() : 0);
		
			updateColumnArrangement();
			fireHeaderChangedEvent();
		}
	}
	
	class ColumnModelChangedListener implements TableColumnModelListener {

		@Override
		public void columnAdded(TableColumnModelEvent e) {
		}

		@Override
		public void columnRemoved(TableColumnModelEvent e) {
		}

		@Override
		public void columnMoved(TableColumnModelEvent e) {
			if (!initialized) return;
			updateProjectView();
		}

		@Override
		public void columnMarginChanged(ChangeEvent e) {
		}

		@Override
		public void columnSelectionChanged(ListSelectionEvent e) {
		}
		
		protected void updateProjectView() {
			List<Column> columns = new ArrayList<Column>();
			for (int i=1; i<getColumnModel().getColumnCount(); i++) {
				TableColumn column = getColumnModel().getColumn(i);
				String key = (String)column.getIdentifier();
				columns.add(new Column(key, column.getWidth()));
			}
			project.getView().setColumns(columns);
		}
	}
	
	class ColumnSizeChangeListener implements PropertyChangeListener {

		@Override
		public void propertyChange(PropertyChangeEvent e) {
			if (!e.getPropertyName().equals("width")) return;
			if (!initialized) return;
			
			TableColumn column = (TableColumn)e.getSource();
			project.getView().getColumn((String)column.getIdentifier()).setWidth((Integer)e.getNewValue());
		
			fireHeaderChangedEvent();
		}
		
	}

}
