package jp.sourceforge.freegantt.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.print.PageFormat;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Locale;

import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.JViewport;
import javax.swing.filechooser.FileNameExtensionFilter;

import jp.sourceforge.freegantt.data.Project;
import jp.sourceforge.freegantt.data.serializer.FileFormatException;
import jp.sourceforge.freegantt.data.serializer.FtgSerializer;
import jp.sourceforge.freegantt.locale.Resource;
import jp.sourceforge.freegantt.print.GanttChartPrintable;

import org.xml.sax.SAXException;

public class Application extends JFrame  {

	private static final long serialVersionUID = 1035207996697568561L;

	public final static String TITLE = "FreeGantt 0.0.3";
	public final static String SKELETON_FILE = "skel.fgt";
	
	public final static int CONFIG_PROJECT = 0;
	public final static int CONFIG_MEMBER = 1;
	public final static int CONFIG_HOLIDAY = 2;
	public final static int CONFIG_MAX = 3;

	
	Project project;
	int previousWindowWidth = 0;
	
	// components
	GlobalSplitPane globalPane;
	GanttChartPane chartPane;
	ConfigRootPane configPane;
	ProjectRootPane projectPane;
	MemberRootPane memberPane;
	HolidayRootPane holidayPane;
	
	public Project getProject() {
		return project;
	}

	public void setProject(Project project) {
		this.project = project;
	}

	public static void main(String[] args) {
		Application app = new Application();
		Resource.makeInstance(app.getLocale());
		
		app.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		app.setSize(1024, 768);
		app.setJMenuBar(new ApplicationMenu(app));
		app.setTitle(TITLE);
		app.setBackground(Color.white);

		app.setProject(new Project());
		if (args.length > 0) {
			app.loadData(args[0]);
		} else {
			app.loadSkelton();
		}

		app.initialize();
	}
	
	/**
	 * 新規ファイル
	 */
	public void newFile() {
		loadSkelton();
		initialize();
	}
	
	private void initialize() {
		setVisible(false);

		project.updateChartRange();
		
		getContentPane().removeAll();
		
		globalPane = new GlobalSplitPane(JSplitPane.HORIZONTAL_SPLIT);
		globalPane.setBounds(0, 0, 1, 1);
		globalPane.setPreferredSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		globalPane.setMaximumSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		globalPane.setDividerSize(4);
		globalPane.setContinuousLayout(true);
		globalPane.setResizeWeight(1.0d);
		getContentPane().add(globalPane, BorderLayout.CENTER);
		
		chartPane = new GanttChartPane(this);
		chartPane.setBounds(0, 0, 1, 1);
		chartPane.setPreferredSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		chartPane.setMaximumSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		chartPane.setLayout(null);
		globalPane.add(chartPane);
		
		configPane = new ConfigRootPane();
		configPane.setBounds(0, 0, 1, 1);
		configPane.setPreferredSize(new Dimension(0, Integer.MAX_VALUE));
		configPane.setMaximumSize(new Dimension(300, Integer.MAX_VALUE));
		configPane.setMinimumSize(new Dimension(1,1));
		configPane.setTabLayoutPolicy(JTabbedPane.SCROLL_TAB_LAYOUT);
		globalPane.add(configPane);
	
		projectPane = new ProjectRootPane(this);
		projectPane.setBounds(0, 0, 300, 200);
		projectPane.setPreferredSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		projectPane.setMaximumSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		configPane.addTab(Resource.get("tabInfo"), projectPane);

		memberPane = new MemberRootPane(this);
		memberPane.setBounds(0, 0, 300, 200);
		memberPane.setPreferredSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		memberPane.setMaximumSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		configPane.addTab(Resource.get("tabResource"), memberPane);
		
		holidayPane = new HolidayRootPane(this);
		holidayPane.setBounds(0, 0, 300, 200);
		holidayPane.setPreferredSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		holidayPane.setMaximumSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
		configPane.addTab(Resource.get("tabHoliday"), holidayPane);
		
		getTaskLineRootPane().initializeViewportOffset();
		setVisible(true);
	}
	
	public void saveData(String filename) {
		FileOutputStream out = null;
		try {
			out = new FileOutputStream(filename);
			(new FtgSerializer()).write(project, out);
			out.close();
			setTitle(TITLE + " - " + filename);
			this.project.setFilename(filename);
		} catch (IOException e) {
			e.printStackTrace();
			JOptionPane.showMessageDialog(null, "ファイルの操作ができませんでした。", "ファイルエラー", JOptionPane.ERROR_MESSAGE);
		} finally {
			try {
				if (out != null)
					out.close();
			} catch (IOException e1) {
			}
		}

	}
	
	public void saveData() {
		if (project.getFilename() == null) {
			saveAsData();
		} else {
			saveData(project.getFilename());
		}
	}

	public void saveAsData() {
		JFileChooser chooser = new JFileChooser();
		FileNameExtensionFilter filter = new FileNameExtensionFilter(
				"freeganttファイル", "fgt");
		chooser.setFileFilter(filter);
		int ret = chooser.showSaveDialog(this);
		if (ret == JFileChooser.APPROVE_OPTION) {
			String filename = chooser.getSelectedFile().getAbsolutePath();
			if (!filename.endsWith(".fgt")) {
				filename = filename.concat(".fgt");
			}
			saveData(filename);
		}
	}
	
	public void loadSkelton() {
		if ((new File(SKELETON_FILE)).exists()) {
			loadData(SKELETON_FILE);
			project.setFilename(null);
		} else {
			project = new Project();
		}
	}
	
	public void loadData(String filename) {
		FileInputStream in = null;
		try {
			in = new FileInputStream(filename);
			Project project = new Project();
			(new FtgSerializer()).read(project, in);
			in.close();
			setTitle(TITLE + " - " + filename);
			this.project = project;
			this.project.setFilename(filename);
			this.project.update();
			initialize();
		} catch (IOException e) {
			e.printStackTrace();
			JOptionPane.showMessageDialog(null, "FreeGanttプロジェクトファイルとして読み込めませんでした。", "フォーマットエラー", JOptionPane.ERROR_MESSAGE);
		} catch (SAXException e) {
			e.printStackTrace();
			JOptionPane.showMessageDialog(null, "FreeGanttプロジェクトファイルとして読み込めませんでした。", "フォーマットエラー", JOptionPane.ERROR_MESSAGE);
		} catch (FileFormatException e) {
			e.printStackTrace();
			JOptionPane.showMessageDialog(null, "FreeGanttプロジェクトファイルとして読み込めませんでした。", "フォーマットエラー", JOptionPane.ERROR_MESSAGE);
		} finally {
			try {
				if (in != null)
					in.close();
			} catch (IOException e1) {
			}
		}
	}
	
	public void loadData() {
		JFileChooser chooser = new JFileChooser();
		FileNameExtensionFilter filter = new FileNameExtensionFilter(
				"freeganttファイル", "fgt");
		chooser.setFileFilter(filter);
		int ret = chooser.showOpenDialog(this);
		if (ret == JFileChooser.APPROVE_OPTION) {
			String filename = chooser.getSelectedFile().getAbsolutePath();
			loadData(filename);
		}
	}

	public TaskLineDataPane getTaskLineDataPane() {
		if (chartPane == null || chartPane.getTaskLineRootPane() == null) return null;
		return chartPane.getTaskLineRootPane().getTaskLineDataPane();
	}

	public JViewport getDataViewport() {
		if (chartPane == null || chartPane.getTaskLineRootPane() == null) return null;
		return chartPane.getTaskLineRootPane().getDataViewport();
	}

	public TaskListTable getTaskListPane() {
		if (chartPane == null || chartPane.getTaskListRootPane() == null) return null;
		return chartPane.getTaskListRootPane().getTaskListTable();
	}

	public TaskListRootPane getTaskListRootPane() {
		if (chartPane == null) return null;
		return chartPane.getTaskListRootPane();
	}
	
	public GanttChartPane getGanttChartPane() {
		return chartPane;
	}

	public TaskLineRootPane getTaskLineRootPane() {
		return chartPane.getTaskLineRootPane();
	}
	
	public TaskLineHeaderPane getTaskLineHeaderPane() {
		if (chartPane == null || chartPane.getTaskListRootPane() == null) return null;
		return chartPane.getTaskLineRootPane().getTaskLineHeaderPane();
	}
	
	public void openConfig(int type) {
		configPane.setSelectedIndex(type);
		int location = globalPane.getWidth() - 200;
		globalPane.setDividerLocation(location);
	}

	public void updateLocale(Locale locale) {
		setLocale(locale);
		Resource.makeInstance(getLocale());
		setJMenuBar(new ApplicationMenu(this));
		project.updateTableModel();
		initialize();
	}

	public void configurePage() {
		PrinterJob printerJob = PrinterJob.getPrinterJob();
		PageFormat pageFormat = printerJob.defaultPage();
		pageFormat = printerJob.pageDialog(getProject().getPrint().getPageFormat());
		getProject().getPrint().setPageFormat(pageFormat);
	}

	public void print() {
		PrinterJob job = PrinterJob.getPrinterJob();
		GanttChartPrintable printable = new GanttChartPrintable(this);
		job.setPrintable(printable, getProject().getPrint().getPageFormat());
		job.setPageable(printable);
		
		if (job.printDialog()) {
			try {
				job.print();
			} catch (PrinterException e1) {
				e1.printStackTrace();
			}
		}
	}
}
