/* dmjpeg.c
 *
 * This software is modified from djpeg.c
 *    by Kenichi Kurimoto
 */
/*
 * djpeg.c
 *
 * Copyright (C) 1991-1997, Thomas G. Lane.
 * This file is part of the Independent JPEG Group's software.
 * For conditions of distribution and use, see the accompanying README file.
 *
 * This file contains a command-line user interface for the JPEG decompressor.
 * It should work on any system with Unix- or MS-DOS-style command lines.
 *
 * Two different command line styles are permitted, depending on the
 * compile-time switch TWO_FILE_COMMANDLINE:
 *	djpeg [options]  inputfile outputfile
 *	djpeg [options]  [inputfile]
 * In the second style, output is always to standard output, which you'd
 * normally redirect to a file or pipe to some other program.  Input is
 * either from a named file or from standard input (typically redirected).
 * The second style is convenient on Unix but is unhelpful on systems that
 * don't support pipes.  Also, you MUST use the first style if your system
 * doesn't do binary I/O to stdin/stdout.
 * To simplify script writing, the "-outfile" switch is provided.  The syntax
 *	djpeg [options]  -outfile outputfile  inputfile
 * works regardless of which command line style is used.
 */

#include "cdjpeg.h"		/* Common decls for cjpeg/djpeg applications */
#include "jversion.h"		/* for version message */

#include <ctype.h>		/* to declare isprint() */

#ifdef USE_CCOMMAND		/* command-line reader for Macintosh */
#ifdef __MWERKS__
#include <SIOUX.h>              /* Metrowerks needs this */
#include <console.h>		/* ... and this */
#endif
#ifdef THINK_C
#include <console.h>		/* Think declares it here */
#endif
#endif

#define DEBUG

/* Create the add-on message string table. */

#define JMESSAGE(code,string)	string ,

static const char * const cdjpeg_message_table[] = {
#include "cderror.h"
  NULL
};

/* Add for LEON MJPEG project */
#include <fcntl.h>
#include <linux/fb.h>
#include <linux/fs.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include "kmjpeg.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

static char * ip_address;
static int f_num=0;
static int markflag=0;


/*
 * This list defines the known output image formats
 * (not all of which need be supported by a given version).
 * You can change the default output format by defining DEFAULT_FMT;
 * indeed, you had better do so if you undefine PPM_SUPPORTED.
 */

typedef enum {
	FMT_BMP,		/* BMP format (Windows flavor) */
	FMT_GIF,		/* GIF format */
	FMT_OS2,		/* BMP format (OS/2 flavor) */
	FMT_PPM,		/* PPM/PGM (PBMPLUS formats) */
	FMT_RLE,		/* RLE format */
	FMT_TARGA,		/* Targa format */
	FMT_TIFF		/* TIFF format */
} IMAGE_FORMATS;

#ifndef DEFAULT_FMT		/* so can override from CFLAGS in Makefile */
#define DEFAULT_FMT	FMT_PPM
#endif

static IMAGE_FORMATS requested_fmt;


/*
 * Argument-parsing code.
 * The switch parser is designed to be useful with DOS-style command line
 * syntax, ie, intermixed switches and file names, where only the switches
 * to the left of a given file name affect processing of that file.
 * The main program in this file doesn't actually use this capability...
 */


static const char * progname;	/* program name for error messages */
static char * outfilename;	/* for -outfile switch */

//#define MAKESTMT(stuff)         do { stuff } while (0)

#define INPUT_VARS(cinfo)  \
        struct jpeg_source_mgr * datasrc = (cinfo)->src;  \
        const JOCTET * next_input_byte = datasrc->next_input_byte;  \
        size_t bytes_in_buffer = datasrc->bytes_in_buffer

/* Reload the local copies --- used only in MAKE_BYTE_AVAIL */
#define INPUT_RELOAD(cinfo)  \
        ( next_input_byte = datasrc->next_input_byte,  \
          bytes_in_buffer = datasrc->bytes_in_buffer )

/* Internal macro for INPUT_BYTE and INPUT_2BYTES: make a byte available.
 * Note we do *not* do INPUT_SYNC before calling fill_input_buffer,
 * but we must reload the local copies after a successful fill.
 */
#define MAKE_BYTE_AVAIL(cinfo,action)  \
        if (bytes_in_buffer == 0) {  \
          if (! (*datasrc->fill_input_buffer) (cinfo))  \
            { action; }  \
          INPUT_RELOAD(cinfo);  \
         }
#define INPUT_4BYTES(cinfo, V, action) \
  MAKESTMT(MAKE_BYTE_AVAIL(cinfo,action);       \
           bytes_in_buffer--;                                           \
           V = ((unsigned int) GETJOCTET(*next_input_byte++))<<24;      \
           MAKE_BYTE_AVAIL(cinfo,action);                               \
           bytes_in_buffer--;                                           \
           V += ((unsigned int)GETJOCTET(*next_input_byte++))<<16;      \
           MAKE_BYTE_AVAIL(cinfo,action);                               \
           bytes_in_buffer--;                                           \
           V += ((unsigned int)GETJOCTET(*next_input_byte++))<<8;       \
           MAKE_BYTE_AVAIL(cinfo,action);                               \
           bytes_in_buffer--;                                           \
           V += ((unsigned int)GETJOCTET(*next_input_byte++)) ; )


/*
 * Add for LEON-mjpeg project
 */

int setup_network( j_decompress_ptr cinfo)
{
  struct sockaddr_in server;
  char buf[256];
  unsigned int **addrptr;
  int i;

  cinfo->sock = socket(AF_INET,SOCK_STREAM,0);
  if(cinfo->sock < 0){
    perror("socket");
    return 1;
  }
  server.sin_family = AF_INET;
  server.sin_port = htons(8090);
  server.sin_addr.s_addr = inet_addr(ip_address);
  if(server.sin_addr.s_addr == 0xffffffff){
    struct hostent *host;
    host = gethostbyname(ip_address);
    if(host == NULL){
      printf("host not found : %s\n", ip_address);
      return 1;
    }
    addrptr = (unsigned int **)host->h_addr_list;

    while(*addrptr != NULL){
      server.sin_addr.s_addr = *(*addrptr);
      if(connect(cinfo->sock, (struct sockaddr *)&server, sizeof(server))==0){
        break;
      }
      addrptr++;
    }
    if(*addrptr == NULL){
      perror("connect");
      return 1;
    }
  }else{
    if(connect(cinfo->sock, (struct sockaddr *)&server, sizeof(server))!=0){
      perror("connect");
      return 1;
    }
  }
  /*request string (HTTP) */
  memset(buf, 0, sizeof(buf));
  snprintf(buf, sizeof(buf), "GET /video.mjpeg HTTP/1.0\r\n\r\n");

  int n = write(cinfo->sock, buf, (int)strlen(buf));
  if(n < 0){
    perror("write");
    return 1;
  }
  n = read(cinfo->sock, buf, 66);
  if(n<0){
    perror("read");
    return 1;
  }
}



void setup_hardware(j_decompress_ptr cinfo)
{
  //  int xres,yres,vbpp,line_len,yfac;
  long int screensize;
  unsigned int  xmcumax,ymcumax,incaddmcuy,size, incaddy, incadd;
  int i, j, samp_f, incaddmcux;
  jpeg_component_info *compptr;
  int h_samp[3];
  int v_samp[3];
  int qtbl_no[3];
  JQUANT_TBL *qtbl;
  struct ioctl_cmdreg regset;
  struct ioctl_cmdq ioctl_dataq;
  struct ioctl_offdata offdata;
  struct ioctl_cmdd datad;
  unsigned int toregister, cadd;

  /** V1 framebuffer & motionJPEG support **/
  if(!(cinfo->fd_framebuffer = open( "/dev/fb0", O_RDWR))){
    fprintf(stderr,"Framebuffer open error!\n");
    exit(1);
  }
#ifdef DEBUG
  fprintf(stderr,"Framebuffer was opened successfully\n");
#endif

  if(ioctl(cinfo->fd_framebuffer, FBIOGET_FSCREENINFO, &(cinfo->fb_finfo))){
    fprintf(stderr,"Framebuffer fixed information get error\n");
    exit(1);
  }
  if(ioctl(cinfo->fd_framebuffer, FBIOGET_VSCREENINFO, &(cinfo->fb_vinfo))){
    fprintf(stderr, "Framebuffer variable information get error\n");
    exit(1);
  }
  //#ifdef DEBUG
  fprintf(stderr, "Framebuffer xsize:%d, ysize:%d, vbpp:%d\n", 
          cinfo->fb_vinfo.xres, cinfo->fb_vinfo.yres, cinfo->fb_vinfo.bits_per_pixel);
  //#endif
  
  if(cinfo->fb_vinfo.bits_per_pixel != 16){
    fprintf(stderr, "This system support only 16 bits per pixel Framebuffer\n");
    exit(1);  
  }
  screensize = cinfo->fb_vinfo.xres * cinfo->fb_vinfo.yres * cinfo->fb_vinfo.bits_per_pixel / 8;
  cinfo->fbptr = (char *)mmap(0,screensize,PROT_READ | PROT_WRITE, MAP_SHARED,cinfo->fd_framebuffer,0);
  if((int)(cinfo->fbptr) == -1){
    fprintf(stderr, "Can't get memory map of framebuffer device\n");
    exit(1);  
  }
  cinfo->fb_yinc = cinfo->fb_vinfo.bits_per_pixel / 8;

  if(cinfo->image_width % 16 != 0 || cinfo->image_width % 16 != 0){
    fprintf(stderr, "This system support only multiple of 16 X,Y size motion JPEG.\n");
    exit(1);  
  }

  /* access IP core */
  printf("setting up hardware\n");
  cinfo->dev_fd = open("/dev/kmjpeg", O_RDWR);
  if(cinfo->dev_fd == -1){
    printf("kmjpeg device file open error\n");
    exit(1);
  }
  regset.fb = cinfo->fb_start_add;
  xmcumax = cinfo->image_width / 16 -1;
  ymcumax = cinfo->image_height /16 -1;
  incaddmcuy = (cinfo->fb_vinfo.xres - (xmcumax +1)*16 )* 2 + 4;

#ifdef DEBUG
  fprintf(stderr, "xmcumax = %d, ymcumax = %d, incaddmcuy = %x\n",xmcumax,ymcumax,incaddmcuy);
#endif

  compptr = cinfo->comp_info;
  for(j=0;j<3;j++,compptr++){
    h_samp[j] = compptr->h_samp_factor;
    v_samp[j] = compptr->v_samp_factor;
    qtbl_no[j] = compptr->quant_tbl_no;
    printf("h_samp = %d, v_samp = %d, quant table no =%d\n", h_samp[j],v_samp[j],qtbl_no[j]);
  }

  if(!((h_samp[0] == 2) && (v_samp[0]==2))){
    printf("this mjpeg file is not 4:1:1 4:2:2\n");
    exit(1);
  }
  if((h_samp[1]==1) && (v_samp[1]==1) && (h_samp[2]==1) && (v_samp[2]==1)){
    samp_f = 0;
    printf("this mjpeg file is 4:1:1\n");
  }else if((h_samp[1]==1) && (v_samp[1]==2) && (h_samp[2]==1) && (v_samp[2]==2)){
    samp_f = 1;
    printf("this mjpeg file is 4:2:2\n");
  }else{
    printf("this mjpeg file is not 4:1:1, 4:2:2\n");
    exit(1);
  }


  size = (unsigned int) ((samp_f <<22) | (ymcumax <<17) | (xmcumax <<11) | incaddmcuy) ;
  regset.size_info = size;

  incaddy = cinfo->fb_vinfo.xres * 2 - 28; 
  incaddmcux = -1 *(15 * cinfo->fb_vinfo.xres * 2 - 4);

  incadd = (unsigned int)((incaddy <<16) |(incaddmcux & 0x0000ffff));
  regset.incadd = incadd; 

#ifdef DEBUG
  fprintf(stderr, "ioctl fb_start_add = %x, size_info = %x, incadd = %x\n",regset.fb, regset.size_info, regset.incadd );
  for(i=0;i<10000;i++);
#endif

  ioctl(cinfo->dev_fd, IOCTL_REGSET, &regset);

  /* call ioctl for ioremap */
  ioctl(cinfo->dev_fd, IOCTL_MAP);

#ifdef DEBUG
  fprintf(stderr, "Start writing quant table into hardware SRAM\n");
#endif
  compptr = cinfo->comp_info;
  for(j=0;j<3;j++,compptr++){
    qtbl = cinfo->quant_tbl_ptrs[qtbl_no[j]];
    for(i=0;i<64;i++){
      //         printf(".%x.",(int)qtbl->quantval[i]);
          ioctl_dataq.quant[j*64+i] = qtbl->quantval[i];
    }
  }
  ioctl(cinfo->dev_fd, IOCTL_QTBL, &ioctl_dataq);


#ifdef DEBUG
  fprintf(stderr, "Start writing DCCache value into hardware SRAM\n");
#endif

  for(j=0;j<2;j++){
    //    printf("DCcache id = %d\n", j);
    for(i=0;i<512;i++){
      //          printf("address=%x, nbits=%d, sym=%x\n", i, cinfo->harddc8nbits[j][i], cinfo->harddc8sym[j][i]);
      toregister = ((cinfo->harddc8nbits[j][i] & 0x0000000f)<<4) | (cinfo->harddc8sym[j][i] & 0x0000000f);
      cadd = (j << 9) | i;  
      offdata.offset[i%128] = cadd;
      offdata.data[i%128] = toregister;
      //      printf("offset=%x,  data=%x\n",offdata.offset[i%128], offdata.data[i%128]);
      if(i%128 == 127){
	ioctl(cinfo->dev_fd, IOCTL_DCCACHE, &offdata);
      }
    }
  }

  memset(&offdata, 0, sizeof(offdata));

  for(j=0;j<2;j++){
    //  printf("ACcache id = %d\n", j);
    for(i=0;i<512;i++){
      //      printf("address=%x, nbits=%d, sym=%x\n", i, cinfo->hardac8nbits[j][i], cinfo->hardac8sym[j][i]);
      toregister = ((cinfo->hardac8nbits[j][i] & 0x0000000f)<<8) | (cinfo->hardac8sym[j][i] & 0x000000ff);
      cadd = (j<<9)| i;
      offdata.offset[i%128] = cadd;
      offdata.data[i%128] = toregister;
      //     printf("offset=%x,  data=%x\n", offdata.offset[i%128], offdata.data[i%128]);
      if(i%128 == 127){
	ioctl(cinfo->dev_fd, IOCTL_ACCACHE, &offdata);
      }
    }
  }

  printf("maxcodemem\n");
  for(i=0;i<4;i++){
    for(j=0;j<32;j++){
      if(j<18){
        datad.data[j+i*32]=cinfo->maxcodemem[i][j];
      }else{
        datad.data[j+i*32]= 0;
      }
    }
  }
  ioctl(cinfo->dev_fd, IOCTL_MAXC, &datad);

  printf("valoffset\n");
  for(i=0;i<4;i++){
    for(j=0;j<32;j++){
      if(j<18){
        datad.data[j+i*32]=cinfo->offsetmem[i][j];
      }else{
        datad.data[j+i*32]=0;
      }
    }
  }
  ioctl(cinfo->dev_fd, IOCTL_OFFS, &datad);
     
  
  printf("huffvalmem\n");
  for(i=0;i<4;i++){
    printf("i=%d\n",i);
    for(j=0;j<256;j++){
      offdata.offset[j%128] = j+i*256;
      offdata.data[j%128] = cinfo->huffvalmem[i][j];
      if(j%128 == 127){
	ioctl(cinfo->dev_fd, IOCTL_HUFFV, &offdata);
      }
    }
  }

  /*call ioctl for iounmap */
  ioctl(cinfo->dev_fd, IOCTL_UNMAP);
  printf(" end of setup hardware\n");
}

push_mjpegdata(j_decompress_ptr cinfo){
  unsigned int mjdata;
  struct ioctl_cmdwrite scandata; 
  int i, ret,j;  
  unsigned int putdata;

  fprintf(stderr,"function : push_mjpegdata is starting\n");
  memset(&scandata, 0, sizeof(scandata));
  INPUT_VARS(cinfo);
  while(1){
    //    if(f_num == 200)break;
    for(i=0;i<64;i++){
      INPUT_4BYTES(cinfo, mjdata, exit(0));
      scandata.fourbdata[i] = mjdata;
      //         fprintf(stderr,"%08X\n",mjdata);
    }
    ioctl(cinfo->dev_fd, IOCTL_WRITE, &scandata);
    //  if(mjdata == 0xd9ffd9ff){
    //    jpeg_destroy_decompress(cinfo);
    //     exit(0);
    //    }
    //              for(i=0;i<100000;i++);
  }
 }




LOCAL(void)
usage (void)
/* complain about bad command line */
{
  fprintf(stderr, "usage: %s [switches] ", progname);
#ifdef TWO_FILE_COMMANDLINE
  fprintf(stderr, "inputfile outputfile\n");
#else
  fprintf(stderr, "[inputfile]\n");
#endif

  fprintf(stderr, "Switches (names may be abbreviated):\n");
  fprintf(stderr, "  -fb_add FramebufferAddress  (Hexadecimal number of Framebuffer start address. ex:41a00000) \n");
  

  exit(EXIT_FAILURE);
}


LOCAL(int)
parse_switches (j_decompress_ptr cinfo, int argc, char **argv,
		int last_file_arg_seen, boolean for_real)
/* Parse optional switches.
 * Returns argv[] index of first file-name argument (== argc if none).
 * Any file names with indexes <= last_file_arg_seen are ignored;
 * they have presumably been processed in a previous iteration.
 * (Pass 0 for last_file_arg_seen on the first or only iteration.)
 * for_real is FALSE on the first (dummy) pass; we may skip any expensive
 * processing.
 */
{
  int argn;
  char * arg;

  ip_address = NULL;
  /* Set up default JPEG parameters. */
  requested_fmt = DEFAULT_FMT;	/* set default output file format */
  outfilename = NULL;
  cinfo->err->trace_level = 0;

  /* Scan command line options, adjust parameters */

  for (argn = 1; argn < argc; argn++) {
    arg = argv[argn];
    if (*arg != '-') {
      /* Not a switch, must be a file name argument */
      if (argn <= last_file_arg_seen) {
	outfilename = NULL;	/* -outfile applies to just one input file */
	continue;		/* ignore this name if previously processed */
      }
      break;			/* else done parsing switches */
    }
    arg++;			/* advance past switch marker character */

    if (keymatch(arg, "bmp", 1)) {
      /* BMP output format. */
      requested_fmt = FMT_BMP;

    } else if (keymatch(arg, "colors", 1) || keymatch(arg, "colours", 1) ||
	       keymatch(arg, "quantize", 1) || keymatch(arg, "quantise", 1)) {
      /* Do color quantization. */
      int val;

      if (++argn >= argc)	/* advance to next argument */
	usage();
      if (sscanf(argv[argn], "%d", &val) != 1)
	usage();
      cinfo->desired_number_of_colors = val;
      cinfo->quantize_colors = TRUE;

    } else if (keymatch(arg, "dct", 2)) {
      /* Select IDCT algorithm. */
      if (++argn >= argc)	/* advance to next argument */
	usage();
      if (keymatch(argv[argn], "int", 1)) {
	cinfo->dct_method = JDCT_ISLOW;
      } else if (keymatch(argv[argn], "fast", 2)) {
	cinfo->dct_method = JDCT_IFAST;
      } else if (keymatch(argv[argn], "float", 2)) {
	cinfo->dct_method = JDCT_FLOAT;
      } else
	usage();

    } else if (keymatch(arg, "dither", 2)) {
      /* Select dithering algorithm. */
      if (++argn >= argc)	/* advance to next argument */
	usage();
      if (keymatch(argv[argn], "fs", 2)) {
	cinfo->dither_mode = JDITHER_FS;
      } else if (keymatch(argv[argn], "none", 2)) {
	cinfo->dither_mode = JDITHER_NONE;
      } else if (keymatch(argv[argn], "ordered", 2)) {
	cinfo->dither_mode = JDITHER_ORDERED;
      } else
	usage();

    } else if (keymatch(arg, "debug", 1) || keymatch(arg, "verbose", 1)) {
      /* Enable debug printouts. */
      /* On first -d, print version identification */
      static boolean printed_version = FALSE;

      if (! printed_version) {
	fprintf(stderr, "Independent JPEG Group's DJPEG, version %s\n%s\n",
		JVERSION, JCOPYRIGHT);
	printed_version = TRUE;
      }
      cinfo->err->trace_level++;

    } else if (keymatch(arg, "fast", 1)) {
      /* Select recommended processing options for quick-and-dirty output. */
      cinfo->two_pass_quantize = FALSE;
      cinfo->dither_mode = JDITHER_ORDERED;
      if (! cinfo->quantize_colors) /* don't override an earlier -colors */
	cinfo->desired_number_of_colors = 216;
      cinfo->dct_method = JDCT_FASTEST;
      cinfo->do_fancy_upsampling = FALSE;

    } else if (keymatch(arg, "gif", 1)) {
      /* GIF output format. */
      requested_fmt = FMT_GIF;

    } else if (keymatch(arg, "grayscale", 2) || keymatch(arg, "greyscale",2)) {
      /* Force monochrome output. */
      cinfo->out_color_space = JCS_GRAYSCALE;

    } else if (keymatch(arg, "map", 3)) {
      /* Quantize to a color map taken from an input file. */
      if (++argn >= argc)	/* advance to next argument */
	usage();
      if (for_real) {		/* too expensive to do twice! */
#ifdef QUANT_2PASS_SUPPORTED	/* otherwise can't quantize to supplied map */
	FILE * mapfile;

	if ((mapfile = fopen(argv[argn], READ_BINARY)) == NULL) {
	  fprintf(stderr, "%s: can't open %s\n", progname, argv[argn]);
	  exit(EXIT_FAILURE);
	}
	read_color_map(cinfo, mapfile);
	fclose(mapfile);
	cinfo->quantize_colors = TRUE;
#else
	ERREXIT(cinfo, JERR_NOT_COMPILED);
#endif
      }

    } else if (keymatch(arg, "maxmemory", 3)) {
      /* Maximum memory in Kb (or Mb with 'm'). */
      long lval;
      char ch = 'x';

      if (++argn >= argc)	/* advance to next argument */
	usage();
      if (sscanf(argv[argn], "%ld%c", &lval, &ch) < 1)
	usage();
      if (ch == 'm' || ch == 'M')
	lval *= 1000L;
      cinfo->mem->max_memory_to_use = lval * 1000L;

    } else if (keymatch(arg, "nosmooth", 3)) {
      /* Suppress fancy upsampling */
      cinfo->do_fancy_upsampling = FALSE;

    } else if (keymatch(arg, "onepass", 3)) {
      /* Use fast one-pass quantization. */
      cinfo->two_pass_quantize = FALSE;

    } else if (keymatch(arg, "os2", 3)) {
      /* BMP output format (OS/2 flavor). */
      requested_fmt = FMT_OS2;

    } else if (keymatch(arg, "outfile", 4)) {
      /* Set output file name. */
      if (++argn >= argc)	/* advance to next argument */
	usage();
      outfilename = argv[argn];	/* save it away for later use */

    } else if (keymatch(arg, "pnm", 1) || keymatch(arg, "ppm", 1)) {
      /* PPM/PGM output format. */
      requested_fmt = FMT_PPM;

    } else if (keymatch(arg, "rle", 1)) {
      /* RLE output format. */
      requested_fmt = FMT_RLE;

    } else if (keymatch(arg, "scale", 1)) {
      /* Scale the output image by a fraction M/N. */
      if (++argn >= argc)	/* advance to next argument */
	usage();
      if (sscanf(argv[argn], "%d/%d",
		 &cinfo->scale_num, &cinfo->scale_denom) != 2)
	usage();

    } else if (keymatch(arg, "targa", 1)) {
      /* Targa output format. */
      requested_fmt = FMT_TARGA;

    } else if (keymatch(arg, "fb_add", 1)) {
      /* Set Framebuffer address */
      unsigned int xval;

      if (++argn >= argc)	/* advance to next argument */
	usage();
      if (sscanf(argv[argn], "%x", &xval) != 1)
	usage();
      cinfo->fb_start_add = xval; 
    } else if(keymatch(arg, "ip", 1)) {
      if (++argn >= argc){       /* advance to next argument */
        usage();
      }else{
      ip_address = argv[argn];
      }
                               /* save it away for later use */
    }else {
      usage();			/* bogus switch */
    }
  }

  return argn;			/* return index of next arg (file name) */
}


/*
 * Marker processor for COM and interesting APPn markers.
 * This replaces the library's built-in processor, which just skips the marker.
 * We want to print out the marker as text, to the extent possible.
 * Note this code relies on a non-suspending data source.
 */

LOCAL(unsigned int)
jpeg_getc (j_decompress_ptr cinfo)
/* Read next byte */
{
  struct jpeg_source_mgr * datasrc = cinfo->src;

  if (datasrc->bytes_in_buffer == 0) {
    if (! (*datasrc->fill_input_buffer) (cinfo))
      ERREXIT(cinfo, JERR_CANT_SUSPEND);
  }
  datasrc->bytes_in_buffer--;
  return GETJOCTET(*datasrc->next_input_byte++);
}


METHODDEF(boolean)
print_text_marker (j_decompress_ptr cinfo)
{
  boolean traceit = (cinfo->err->trace_level >= 1);
  INT32 length;
  unsigned int ch;
  unsigned int lastch = 0;

  length = jpeg_getc(cinfo) << 8;
  length += jpeg_getc(cinfo);
  length -= 2;			/* discount the length word itself */

  if (traceit) {
    if (cinfo->unread_marker == JPEG_COM)
      fprintf(stderr, "Comment, length %ld:\n", (long) length);
    else			/* assume it is an APPn otherwise */
      fprintf(stderr, "APP%d, length %ld:\n",
	      cinfo->unread_marker - JPEG_APP0, (long) length);
  }

  while (--length >= 0) {
    ch = jpeg_getc(cinfo);
    if (traceit) {
      /* Emit the character in a readable form.
       * Nonprintables are converted to \nnn form,
       * while \ is converted to \\.
       * Newlines in CR, CR/LF, or LF form will be printed as one newline.
       */
      if (ch == '\r') {
	fprintf(stderr, "\n");
      } else if (ch == '\n') {
	if (lastch != '\r')
	  fprintf(stderr, "\n");
      } else if (ch == '\\') {
	fprintf(stderr, "\\\\");
      } else if (isprint(ch)) {
	putc(ch, stderr);
      } else {
	fprintf(stderr, "\\%03o", ch);
      }
      lastch = ch;
    }
  }

  if (traceit)
    fprintf(stderr, "\n");

  return TRUE;
}


/*
 * The main program.
 */

int
main (int argc, char **argv)
{
  struct jpeg_decompress_struct cinfo;
  struct jpeg_error_mgr jerr;
#ifdef PROGRESS_REPORT
  struct cdjpeg_progress_mgr progress;
#endif
  int file_index;
  djpeg_dest_ptr dest_mgr = NULL;
  FILE * input_file;
  FILE * output_file;
  JDIMENSION num_scanlines;

  struct timeval j_50f,j_100f;
  double exe_fps;


  /* On Mac, fetch a command line. */
#ifdef USE_CCOMMAND
  argc = ccommand(&argv);
#endif

  progname = argv[0];
  if (progname == NULL || progname[0] == 0)
    progname = "djpeg";		/* in case C library doesn't provide it */

  /* Initialize the JPEG decompression object with default error handling. */
  cinfo.err = jpeg_std_error(&jerr);
  jpeg_create_decompress(&cinfo);
  /* Add some application-specific error messages (from cderror.h) */
  jerr.addon_message_table = cdjpeg_message_table;
  jerr.first_addon_message = JMSG_FIRSTADDONCODE;
  jerr.last_addon_message = JMSG_LASTADDONCODE;

  /* Insert custom marker processor for COM and APP12.
   * APP12 is used by some digital camera makers for textual info,
   * so we provide the ability to display it as text.
   * If you like, additional APPn marker types can be selected for display,
   * but don't try to override APP0 or APP14 this way (see libjpeg.doc).
   */
  jpeg_set_marker_processor(&cinfo, JPEG_COM, print_text_marker);
  jpeg_set_marker_processor(&cinfo, JPEG_APP0+12, print_text_marker);

  /* Now safe to enable signal catcher. */
#ifdef NEED_SIGNAL_CATCHER
  enable_signal_catcher((j_common_ptr) &cinfo);
#endif

  /* Scan command line to find file names. */
  /* It is convenient to use just one switch-parsing routine, but the switch
   * values read here are ignored; we will rescan the switches after opening
   * the input file.
   * (Exception: tracing level set here controls verbosity for COM markers
   * found during jpeg_read_header...)
   */

  file_index = parse_switches(&cinfo, argc, argv, 0, FALSE);

#ifdef TWO_FILE_COMMANDLINE
  /* Must have either -outfile switch or explicit output file name */
  if (outfilename == NULL) {
    if (file_index != argc-2) {
      fprintf(stderr, "%s: must name one input and one output file\n",
	      progname);
      usage();
    }
    outfilename = argv[file_index+1];
  } else {
    if (file_index != argc-1) {
      fprintf(stderr, "%s: must name one input and one output file\n",
	      progname);
      usage();
    }
  }
#else
  /* Unix style: expect zero or one file name */
  if (file_index < argc-1) {
    fprintf(stderr, "%s: only one input file\n", progname);
    usage();
  }
#endif /* TWO_FILE_COMMANDLINE */

  /* Open the input file. */
  if (file_index < argc) {
    if ((input_file = fopen(argv[file_index], READ_BINARY)) == NULL) {
      fprintf(stderr, "%s: can't open %s\n", progname, argv[file_index]);
      exit(EXIT_FAILURE);
    }
  } else {
    /* default input file is stdin */
    input_file = read_stdin();
  }

  /* Open the output file. */
  if (outfilename != NULL) {
    if ((output_file = fopen(outfilename, WRITE_BINARY)) == NULL) {
      fprintf(stderr, "%s: can't open %s\n", progname, outfilename);
      exit(EXIT_FAILURE);
    }
  } else {
    /* default output file is stdout */
    output_file = write_stdout();
  }

#ifdef PROGRESS_REPORT
  start_progress_monitor((j_common_ptr) &cinfo, &progress);
#endif

  /* Specify data source for decompression */
  jpeg_stdio_src(&cinfo, input_file);

 /*V1 LEON-mjpeg project */
  cinfo.is_net = 0;
  if(ip_address != NULL){
    cinfo.is_net = 1;
    setup_network(&cinfo);
  }
  for(f_num=0;f_num<110;f_num++){

  if(f_num == 50){
    gettimeofday(&j_50f,NULL);
  }
  if(f_num == 100){
    gettimeofday(&j_100f,NULL);
  }

  /* Read file header, set default decompression parameters */
  (void) jpeg_read_header(&cinfo, TRUE);
 
  /* Adjust default decompression parameters by re-parsing the options */
   file_index = parse_switches(&cinfo, argc, argv, 0, TRUE);

  /* Initialize the output module now to let it override any crucial
   * option settings (for instance, GIF wants to force color quantization).
   */
  switch (requested_fmt) {
#ifdef BMP_SUPPORTED
  case FMT_BMP:
    dest_mgr = jinit_write_bmp(&cinfo, FALSE);
    break;
  case FMT_OS2:
    dest_mgr = jinit_write_bmp(&cinfo, TRUE);
    break;
#endif
#ifdef GIF_SUPPORTED
  case FMT_GIF:
    dest_mgr = jinit_write_gif(&cinfo);
    break;
#endif
#ifdef PPM_SUPPORTED
  case FMT_PPM:
    dest_mgr = jinit_write_ppm(&cinfo);
    break;
#endif
#ifdef RLE_SUPPORTED
  case FMT_RLE:
    dest_mgr = jinit_write_rle(&cinfo);
    break;
#endif
#ifdef TARGA_SUPPORTED
  case FMT_TARGA:
    dest_mgr = jinit_write_targa(&cinfo);
    break;
#endif
  default:
    ERREXIT(&cinfo, JERR_UNSUPPORTED_FORMAT);
    break;
  }
  dest_mgr->output_file = output_file;

  /* Start decompressor */
  (void) jpeg_start_decompress(&cinfo);

  if(f_num == 0){
    setup_hardware(&cinfo);
  }

  /* Write output file header */
  //  (*dest_mgr->start_output) (&cinfo, dest_mgr);


  push_mjpegdata(&cinfo);

  /* Process data */
  //  while (cinfo.output_scanline < cinfo.output_height) {
  //   num_scanlines = jpeg_read_scanlines(&cinfo, dest_mgr->buffer,
  //					dest_mgr->buffer_height);
    //   (*dest_mgr->put_pixel_rows) (&cinfo, dest_mgr, num_scanlines);
  // }

#ifdef PROGRESS_REPORT
  /* Hack: count final pass as done in case finish_output does an extra pass.
   * The library won't have updated completed_passes.
   */
  progress.pub.completed_passes = progress.pub.total_passes;
#endif

  /* Finish decompression and release memory.
   * I must do it in this order because output module has allocated memory
   * of lifespan JPOOL_IMAGE; it needs to finish before releasing memory.
   */
  (*dest_mgr->finish_output) (&cinfo, dest_mgr);
  (void) jpeg_finish_decompress(&cinfo);
  }
  jpeg_destroy_decompress(&cinfo);

  /* Close files, if we opened them */
  if (input_file != stdin)
    fclose(input_file);
  if (output_file != stdout)
    fclose(output_file);

#ifdef PROGRESS_REPORT
  end_progress_monitor((j_common_ptr) &cinfo);
#endif

  fprintf(stderr, "Frame  #50 = %d.%d\n", j_50f.tv_sec,j_50f.tv_usec);
  fprintf(stderr, "Frame #100 = %d.%d\n", j_100f.tv_sec,j_100f.tv_usec);

  exe_fps = 50/(((double)j_100f.tv_sec + (double)j_100f.tv_usec / 1e6)
                 -((double)j_50f.tv_sec + (double)j_50f.tv_usec / 1e6));


  fprintf(stderr, "This program decode %f fps\n", exe_fps);



  /* All done. */
  exit(jerr.num_warnings ? EXIT_WARNING : EXIT_SUCCESS);
  return 0;			/* suppress no-return-value warnings */
}
