/* modified by Kenichi Kurimoto */

/*
 * jidctint.c
 *
 * Copyright (C) 1991-1998, Thomas G. Lane.
 * This file is part of the Independent JPEG Group's software.
 * For conditions of distribution and use, see the accompanying README file.
 *
 * This file contains a slow-but-accurate integer implementation of the
 * inverse DCT (Discrete Cosine Transform).  In the IJG code, this routine
 * must also perform dequantization of the input coefficients.
 *
 * A 2-D IDCT can be done by 1-D IDCT on each column followed by 1-D IDCT
 * on each row (or vice versa, but it's more convenient to emit a row at
 * a time).  Direct algorithms are also available, but they are much more
 * complex and seem not to be any faster when reduced to code.
 *
 * This implementation is based on an algorithm described in
 *   C. Loeffler, A. Ligtenberg and G. Moschytz, "Practical Fast 1-D DCT
 *   Algorithms with 11 Multiplications", Proc. Int'l. Conf. on Acoustics,
 *   Speech, and Signal Processing 1989 (ICASSP '89), pp. 988-991.
 * The primary algorithm described there uses 11 multiplies and 29 adds.
 * We use their alternate method with 12 multiplies and 32 adds.
 * The advantage of this method is that no data path contains more than one
 * multiplication; this allows a very simple and accurate implementation in
 * scaled fixed-point arithmetic, with a minimal number of shifts.
 */

#define JPEG_INTERNALS
#include "jinclude.h"
#include "jpeglib.h"
#include "jdct.h"		/* Private declarations for DCT subsystem */


#define IDCTHW

/*
 * This module is specialized to the case DCTSIZE = 8.
 */

#if DCTSIZE != 8
  Sorry, this code only copes with 8x8 DCTs. /* deliberate syntax err */
#endif


/*
 * The poop on this scaling stuff is as follows:
 *
 * Each 1-D IDCT step produces outputs which are a factor of sqrt(N)
 * larger than the true IDCT outputs.  The final outputs are therefore
 * a factor of N larger than desired; since N=8 this can be cured by
 * a simple right shift at the end of the algorithm.  The advantage of
 * this arrangement is that we save two multiplications per 1-D IDCT,
 * because the y0 and y4 inputs need not be divided by sqrt(N).
 *
 * We have to do addition and subtraction of the integer inputs, which
 * is no problem, and multiplication by fractional constants, which is
 * a problem to do in integer arithmetic.  We multiply all the constants
 * by CONST_SCALE and convert them to integer constants (thus retaining
 * CONST_BITS bits of precision in the constants).  After doing a
 * multiplication we have to divide the product by CONST_SCALE, with proper
 * rounding, to produce the correct output.  This division can be done
 * cheaply as a right shift of CONST_BITS bits.  We postpone shifting
 * as long as possible so that partial sums can be added together with
 * full fractional precision.
 *
 * The outputs of the first pass are scaled up by PASS1_BITS bits so that
 * they are represented to better-than-integral precision.  These outputs
 * require BITS_IN_JSAMPLE + PASS1_BITS + 3 bits; this fits in a 16-bit word
 * with the recommended scaling.  (To scale up 12-bit sample data further, an
 * intermediate INT32 array would be needed.)
 *
 * To avoid overflow of the 32-bit intermediate results in pass 2, we must
 * have BITS_IN_JSAMPLE + CONST_BITS + PASS1_BITS <= 26.  Error analysis
 * shows that the values given below are the most effective.
 */

#if BITS_IN_JSAMPLE == 8
#define CONST_BITS  13
#define PASS1_BITS  2
#else
#define CONST_BITS  13
#define PASS1_BITS  1		/* lose a little precision to avoid overflow */
#endif

/* Some C compilers fail to reduce "FIX(constant)" at compile time, thus
 * causing a lot of useless floating-point operations at run time.
 * To get around this we use the following pre-calculated constants.
 * If you change CONST_BITS you may want to add appropriate values.
 * (With a reasonable C compiler, you can just rely on the FIX() macro...)
 */

#if CONST_BITS == 13
#define FIX_0_298631336  ((INT32)  2446)	/* FIX(0.298631336) */
#define FIX_0_390180644  ((INT32)  3196)	/* FIX(0.390180644) */
#define FIX_0_541196100  ((INT32)  4433)	/* FIX(0.541196100) */
#define FIX_0_765366865  ((INT32)  6270)	/* FIX(0.765366865) */
#define FIX_0_899976223  ((INT32)  7373)	/* FIX(0.899976223) */
#define FIX_1_175875602  ((INT32)  9633)	/* FIX(1.175875602) */
#define FIX_1_501321110  ((INT32)  12299)	/* FIX(1.501321110) */
#define FIX_1_847759065  ((INT32)  15137)	/* FIX(1.847759065) */
#define FIX_1_961570560  ((INT32)  16069)	/* FIX(1.961570560) */
#define FIX_2_053119869  ((INT32)  16819)	/* FIX(2.053119869) */
#define FIX_2_562915447  ((INT32)  20995)	/* FIX(2.562915447) */
#define FIX_3_072711026  ((INT32)  25172)	/* FIX(3.072711026) */
#else
#define FIX_0_298631336  FIX(0.298631336)
#define FIX_0_390180644  FIX(0.390180644)
#define FIX_0_541196100  FIX(0.541196100)
#define FIX_0_765366865  FIX(0.765366865)
#define FIX_0_899976223  FIX(0.899976223)
#define FIX_1_175875602  FIX(1.175875602)
#define FIX_1_501321110  FIX(1.501321110)
#define FIX_1_847759065  FIX(1.847759065)
#define FIX_1_961570560  FIX(1.961570560)
#define FIX_2_053119869  FIX(2.053119869)
#define FIX_2_562915447  FIX(2.562915447)
#define FIX_3_072711026  FIX(3.072711026)
#endif


/* Multiply an INT32 variable by an INT32 constant to yield an INT32 result.
 * For 8-bit samples with the recommended scaling, all the variable
 * and constant values involved are no more than 16 bits wide, so a
 * 16x16->32 bit multiply can be used instead of a full 32x32 multiply.
 * For 12-bit samples, a full 32-bit multiplication will be needed.
 */

#if BITS_IN_JSAMPLE == 8
#define MULTIPLY(var,const)  MULTIPLY16C16(var,const)
#else
#define MULTIPLY(var,const)  ((var) * (const))
#endif


/* Dequantize a coefficient by multiplying it by the multiplier-table
 * entry; produce an int result.  In this module, both inputs and result
 * are 16 bits or less, so either int or short multiply will work.
 */

#define DEQUANTIZE(coef,quantval)  (((ISLOW_MULT_TYPE) (coef)) * (quantval))

static double coeff_matrix[8][8] =
    {{0.35355339059,  0.35355339059,  0.35355339059,  0.35355339059,  0.35355339059,  0.35355339059,  0.35355339059,  0.35355339059},
     {0.49039264020,  0.41573480615,  0.27778511651,  0.09754516101, -0.09754516101, -0.27778511651, -0.41573480615, -0.49039264020},
     {0.46193976626,  0.19134171618, -0.19134171618, -0.46193976626, -0.46193976626, -0.19134171618,  0.19134171618,  0.46193976626},
     {0.41573480615, -0.09754516101, -0.49039264020, -0.27778511651,  0.27778511651,  0.49039264020,  0.09754516101, -0.41573480615},
     {0.35355339059, -0.35355339059, -0.35355339059,  0.35355339059,  0.35355339059, -0.35355339059, -0.35355339059,  0.35355339059},
     {0.27778511615, -0.49039264020,  0.09754516101,  0.41573480615, -0.41573480615, -0.09754516101,  0.49039264020, -0.27778511615},
     {0.19134171618, -0.46193976626,  0.46193976626, -0.19134171618, -0.19134171618,  0.46193976626, -0.46193976626,  0.19134171618},
     {0.09754516101, -0.27778511651,  0.41573480615, -0.49039264020,  0.49039264020, -0.41573480615,  0.27778511651,  0.09754516101}};

static int coeff_int[8][4] =
  {{0x2d413d, 0x2d413d, 0x2d413d, 0x2d413d},
   {0x3ec530, 0x3536cc, 0x238e76, 0xc7c5c},
   {0x3b20d8, 0x187de3, 0xffe7821e, 0xffc4df29},
   {0x3536cc, 0xfff383a4, 0xffc13ad1, 0xffdc718a},
   {0x2d413d, 0xffd2bec4, 0xffd2bec4, 0x2d413d},
   {0x238e76, 0xffc13ad1, 0xc7c5c, 0x3536cc},
   {0x187de3, 0xffc4df29, 0x3b20d8, 0xffe7821e},
   {0xc7c5c, 0xffdc718a, 0x3536cc, 0xffc13ad1}};
static int coeff_int2[8][4] =
  {{0xb505, 0xb505, 0xb505, 0xb505},
   {0xfb15, 0xd4db, 0x8e3a, 0x31f1},
   {0xec83, 0x61f8, 0xffff9e09, 0xffff137d},
   {0xd4db, 0xffffce0f, 0xffff04ec, 0xffff71c7},
   {0xb505, 0xffff4afc, 0xffff4afc, 0xb505},
   {0x8e3a, 0xffff04ec, 0x31f1, 0xd4db},
   {0x61f8, 0xffff137d, 0xec83, 0xffff9e09},
   {0x31f1, 0xffff71c7, 0xd4db, 0xffff04ec}};

int left_cutsi(int data, int num)
{
  int ret;

  //  printf("left_cutsi is called. data-%x(%d), num-%d      ",data,data,num);
  ret = data << num;
  ret = ret >> num;
  if(ret != data){
    printf("left_cutsi is called. data-%x(%d), num-%d      ",data,data,num);
    printf("ret = %x(%d)\n",ret,ret);
  }
  return(ret);
} 

int right_cutsi(int data, int num)
{
  int ret;
  ret = data >> num;
  return(ret);
}

/*n bit-me sisya-gonyu n+1 bit output*/
int round_cutsi(int data, int num)
{
  int ret;
  int mask;
  int five;

  //  printf("round_cutsi is called. data-%x(%d), num-%d     ",data,data,num);
  mask = 1;
  mask = mask << (num-1);
  if(data & mask){
    ret = (data >> num) +1;
  }else{
    ret = data >> num;
  }
  //  printf("ret = %x(%d)\n",ret,ret);
  return(ret);
}

long round_cutsl(long long data, int num)
{
  long long ret;
  long long mask;

  //    printf("round_cutsl is called. data-%x(%d), num-%d     ",data,data,num);

  mask = 1;
  mask = mask << (num-1);
  if(data & mask){
    ret = (data >> num) +1;
  }else{
    ret = data >> num;
  }
  //    printf("ret = %x(%d)\n",ret,ret);
  return(ret);
}


int ranlim(int data)
{
  int ret;

  ret = data;
  if(data < 0)ret = 0;
  if(data > 255)ret = 255;
  return(ret);
}

int ranlim_new(int data)
{
  int ret;
  ret = data;
  if(data < -128) ret = -128;
  if(data > 127) ret = 127;
  return(ret);
}

/*
 * Perform dequantization and inverse DCT on one block of coefficients.
 */

GLOBAL(void)
jpeg_idct_islow (j_decompress_ptr cinfo, jpeg_component_info * compptr,
		 JCOEFPTR coef_block,
		 JSAMPARRAY output_buf, JDIMENSION output_col)
{
  INT32 tmp0, tmp1, tmp2, tmp3;
  INT32 tmp10, tmp11, tmp12, tmp13;
  INT32 z1, z2, z3, z4, z5;
  JCOEFPTR inptr;
  ISLOW_MULT_TYPE * quantptr;
  //  int * wsptr;
  JSAMPROW outptr;
  JSAMPLE *range_limit = IDCT_range_limit(cinfo);
  int ctr;
  //  int workspace[DCTSIZE2];	/* buffers data between passes */

#ifdef IDCTSOFT
  int * wsptr;
  int workspace[DCTSIZE2];	/* buffers data between passes */
#endif
#ifdef IDCTHARD
  double * wsptr;
  double workspace[DCTSIZE2];	/* buffers data between passes */
#endif
#ifdef IDCTHW
  int * wsptr;
  int workspace[DCTSIZE2];
#endif

  SHIFT_TEMPS

    int i;
  JSAMPROW koutptr;

#ifdef IDCTHW
  int accumulator[8]={0,0,0,0,0,0,0,0};



  /* Pass 1: process columns from input, store into work array. */
  /* Note results are scaled up by sqrt(8) compared to a true IDCT; */
  /* furthermore, we scale the results by 2**PASS1_BITS. */

  inptr = coef_block;
  quantptr = (ISLOW_MULT_TYPE *) compptr->dct_table;
  wsptr = workspace;

  for(i=0;i<64;i++){
    fprintf(cinfo->in_2ddct,"%03X\n",(0x00000fff & inptr[i]));
  }
  for(i=0;i<64;i++){
    fprintf(cinfo->in_2dquant,"%02x\n",quantptr[i]);
  }

  for(ctr = 0; ctr< 8; ctr++){
    fprintf(cinfo->in_dct,"%03X\n",(0x00000fff & inptr[0]));
    fprintf(cinfo->in_dct,"%03X\n",(0x00000fff & inptr[8]));
    fprintf(cinfo->in_dct,"%03X\n",(0x00000fff & inptr[16]));
    fprintf(cinfo->in_dct,"%03X\n",(0x00000fff & inptr[24]));
    fprintf(cinfo->in_dct,"%03X\n",(0x00000fff & inptr[32]));
    fprintf(cinfo->in_dct,"%03X\n",(0x00000fff & inptr[40]));
    fprintf(cinfo->in_dct,"%03X\n",(0x00000fff & inptr[48]));
    fprintf(cinfo->in_dct,"%03X\n",(0x00000fff & inptr[56]));
    
    fprintf(cinfo->in_quant,"%02x\n",quantptr[0]);
    fprintf(cinfo->in_quant,"%02x\n",quantptr[8]);
    fprintf(cinfo->in_quant,"%02x\n",quantptr[16]);
    fprintf(cinfo->in_quant,"%02x\n",quantptr[24]);
    fprintf(cinfo->in_quant,"%02x\n",quantptr[32]);
    fprintf(cinfo->in_quant,"%02x\n",quantptr[40]);
    fprintf(cinfo->in_quant,"%02x\n",quantptr[48]);
    fprintf(cinfo->in_quant,"%02x\n",quantptr[56]);
    
    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(coeff_int[0][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[0],20) * left_cutsi((int)quantptr[0],23),20)),12); 
     accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(coeff_int[0][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[0],20) * left_cutsi((int)quantptr[0],23),20)),12); 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(coeff_int[0][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[0],20) * left_cutsi((int)quantptr[0],23),20)),12); 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(coeff_int[0][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[0],20) * left_cutsi((int)quantptr[0],23),20)),12); 
    accumulator[4] = accumulator[4] + (int)round_cutsl((long long)left_cutsi(coeff_int[0][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[0],20) * left_cutsi((int)quantptr[0],23),20)),12); 
    accumulator[5] = accumulator[5] + (int)round_cutsl((long long)left_cutsi(coeff_int[0][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[0],20) * left_cutsi((int)quantptr[0],23),20)),12); 
    accumulator[6] = accumulator[6] + (int)round_cutsl((long long)left_cutsi(coeff_int[0][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[0],20) * left_cutsi((int)quantptr[0],23),20)),12); 
    accumulator[7] = accumulator[7] + (int)round_cutsl((long long)left_cutsi(coeff_int[0][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[0],20) * left_cutsi((int)quantptr[0],23),20)),12); 

    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(coeff_int[1][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[8],20) * left_cutsi((int)quantptr[8],23),20)),12); 
     accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(coeff_int[1][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[8],20) * left_cutsi((int)quantptr[8],23),20)),12); 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(coeff_int[1][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[8],20) * left_cutsi((int)quantptr[8],23),20)),12); 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(coeff_int[1][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[8],20) * left_cutsi((int)quantptr[8],23),20)) ,12); 
    accumulator[4] = accumulator[4] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[1][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[8],20) * left_cutsi((int)quantptr[8],23),20)),12); 
    accumulator[5] = accumulator[5] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[1][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[8],20) * left_cutsi((int)quantptr[8],23),20)),12); 
    accumulator[6] = accumulator[6] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[1][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[8],20) * left_cutsi((int)quantptr[8],23),20)),12); 
    accumulator[7] = accumulator[7] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[1][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[8],20) * left_cutsi((int)quantptr[8],23),20)),12); 

    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(coeff_int[2][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[16],20) * left_cutsi((int)quantptr[16],23),20)),12); 
     accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(coeff_int[2][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[16],20) * left_cutsi((int)quantptr[16],23),20)),12); 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(coeff_int[2][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[16],20) * left_cutsi((int)quantptr[16],23),20)),12); 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(coeff_int[2][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[16],20) * left_cutsi((int)quantptr[16],23),20)),12); 
    accumulator[4] = accumulator[4] + (int)round_cutsl((long long)left_cutsi(coeff_int[2][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[16],20) * left_cutsi((int)quantptr[16],23),20)),12); 
    accumulator[5] = accumulator[5] + (int)round_cutsl((long long)left_cutsi(coeff_int[2][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[16],20) * left_cutsi((int)quantptr[16],23),20)),12); 
    accumulator[6] = accumulator[6] + (int)round_cutsl((long long)left_cutsi(coeff_int[2][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[16],20) * left_cutsi((int)quantptr[16],23),20)),12); 
    accumulator[7] = accumulator[7] + (int)round_cutsl((long long)left_cutsi(coeff_int[2][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[16],20) * left_cutsi((int)quantptr[16],23),20)),12); 
 
    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(coeff_int[3][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[24],20) * left_cutsi((int)quantptr[24],23),20)),12); 
     accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(coeff_int[3][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[24],20) * left_cutsi((int)quantptr[24],23),20)),12); 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(coeff_int[3][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[24],20) * left_cutsi((int)quantptr[24],23),20)),12); 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(coeff_int[3][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[24],20) * left_cutsi((int)quantptr[24],23),20)),12); 
    accumulator[4] = accumulator[4] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[3][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[24],20) * left_cutsi((int)quantptr[24],23),20)),12); 
    accumulator[5] = accumulator[5] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[3][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[24],20) * left_cutsi((int)quantptr[24],23),20)),12); 
    accumulator[6] = accumulator[6] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[3][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[24],20) * left_cutsi((int)quantptr[24],23),20)),12); 
    accumulator[7] = accumulator[7] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[3][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[24],20) * left_cutsi((int)quantptr[24],23),20)),12); 
 
    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(coeff_int[4][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[32],20) * left_cutsi((int)quantptr[32],23),20)),12); 
     accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(coeff_int[4][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[32],20) * left_cutsi((int)quantptr[32],23),20)),12); 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(coeff_int[4][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[32],20) * left_cutsi((int)quantptr[32],23),20)),12); 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(coeff_int[4][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[32],20) * left_cutsi((int)quantptr[32],23),20)),12); 
    accumulator[4] = accumulator[4] + (int)round_cutsl((long long)left_cutsi(coeff_int[4][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[32],20) * left_cutsi((int)quantptr[32],23),20)),12); 
    accumulator[5] = accumulator[5] + (int)round_cutsl((long long)left_cutsi(coeff_int[4][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[32],20) * left_cutsi((int)quantptr[32],23),20)),12); 
    accumulator[6] = accumulator[6] + (int)round_cutsl((long long)left_cutsi(coeff_int[4][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[32],20) * left_cutsi((int)quantptr[32],23),20)),12); 
    accumulator[7] = accumulator[7] + (int)round_cutsl((long long)left_cutsi(coeff_int[4][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[32],20) * left_cutsi((int)quantptr[32],23),20)),12); 
 
    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(coeff_int[5][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[40],20) * left_cutsi((int)quantptr[40],23),20)),12); 
     accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(coeff_int[5][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[40],20) * left_cutsi((int)quantptr[40],23),20)),12); 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(coeff_int[5][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[40],20) * left_cutsi((int)quantptr[40],23),20)),12); 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(coeff_int[5][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[40],20) * left_cutsi((int)quantptr[40],23),20)),12); 
    accumulator[4] = accumulator[4] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[5][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[40],20) * left_cutsi((int)quantptr[40],23),20)),12); 
    accumulator[5] = accumulator[5] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[5][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[40],20) * left_cutsi((int)quantptr[40],23),20)),12); 
    accumulator[6] = accumulator[6] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[5][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[40],20) * left_cutsi((int)quantptr[40],23),20)),12); 
    accumulator[7] = accumulator[7] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[5][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[40],20) * left_cutsi((int)quantptr[40],23),20)),12); 
 
    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(coeff_int[6][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[48],20) * left_cutsi((int)quantptr[48],23),20)),12); 
     accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(coeff_int[6][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[48],20) * left_cutsi((int)quantptr[48],23),20)),12); 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(coeff_int[6][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[48],20) * left_cutsi((int)quantptr[48],23),20)),12); 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(coeff_int[6][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[48],20) * left_cutsi((int)quantptr[48],23),20)),12); 
    accumulator[4] = accumulator[4] + (int)round_cutsl((long long)left_cutsi(coeff_int[6][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[48],20) * left_cutsi((int)quantptr[48],23),20)),12); 
    accumulator[5] = accumulator[5] + (int)round_cutsl((long long)left_cutsi(coeff_int[6][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[48],20) * left_cutsi((int)quantptr[48],23),20)),12); 
    accumulator[6] = accumulator[6] + (int)round_cutsl((long long)left_cutsi(coeff_int[6][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[48],20) * left_cutsi((int)quantptr[48],23),20)),12); 
    accumulator[7] = accumulator[7] + (int)round_cutsl((long long)left_cutsi(coeff_int[6][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[48],20) * left_cutsi((int)quantptr[48],23),20)),12); 

    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(coeff_int[7][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[56],20) * left_cutsi((int)quantptr[56],23),20)),12); 
     accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(coeff_int[7][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[56],20) * left_cutsi((int)quantptr[56],23),20)),12); 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(coeff_int[7][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[56],20) * left_cutsi((int)quantptr[56],23),20)),12); 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(coeff_int[7][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[56],20) * left_cutsi((int)quantptr[56],23),20)),12); 
    accumulator[4] = accumulator[4] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[7][3],9) *((long long)left_cutsi(left_cutsi((int)inptr[56],20) * left_cutsi((int)quantptr[56],23),20)),12); 
    accumulator[5] = accumulator[5] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[7][2],9) *((long long)left_cutsi(left_cutsi((int)inptr[56],20) * left_cutsi((int)quantptr[56],23),20)),12); 
    accumulator[6] = accumulator[6] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[7][1],9) *((long long)left_cutsi(left_cutsi((int)inptr[56],20) * left_cutsi((int)quantptr[56],23),20)),12); 
    accumulator[7] = accumulator[7] + (int)round_cutsl((-1)*(long long)left_cutsi(coeff_int[7][0],9) *((long long)left_cutsi(left_cutsi((int)inptr[56],20) * left_cutsi((int)quantptr[56],23),20)),12); 
 
    wsptr[0] = round_cutsi(left_cutsi((int)accumulator[0],11),5);
    wsptr[8] = round_cutsi(left_cutsi((int)accumulator[1],11),5);
    wsptr[16] = round_cutsi(left_cutsi((int)accumulator[2],11),5);
    wsptr[24] = round_cutsi(left_cutsi((int)accumulator[3],11),5);
    wsptr[32] = round_cutsi(left_cutsi((int)accumulator[4],11),5);
    wsptr[40] = round_cutsi(left_cutsi((int)accumulator[5],11),5);
    wsptr[48] = round_cutsi(left_cutsi((int)accumulator[6],11),5);
    wsptr[56] = round_cutsi(left_cutsi((int)accumulator[7],11),5);

    fprintf(cinfo->inter_dct,"%04x\n",(0x0000ffff & wsptr[0]));
    fprintf(cinfo->inter_dct,"%04x\n",(0x0000ffff & wsptr[8]));
    fprintf(cinfo->inter_dct,"%04x\n",(0x0000ffff & wsptr[16]));
    fprintf(cinfo->inter_dct,"%04x\n",(0x0000ffff & wsptr[24]));
    fprintf(cinfo->inter_dct,"%04x\n",(0x0000ffff & wsptr[32]));
    fprintf(cinfo->inter_dct,"%04x\n",(0x0000ffff & wsptr[40]));
    fprintf(cinfo->inter_dct,"%04x\n",(0x0000ffff & wsptr[48]));
    fprintf(cinfo->inter_dct,"%04x\n",(0x0000ffff & wsptr[56]));

    for(i=0; i<8; i++){
      accumulator[i] = 0;
    }
    inptr++;
    quantptr++;
    wsptr++;
  }

  wsptr = workspace;
  for(ctr = 0; ctr<8; ctr++){
    outptr = output_buf[ctr] + output_col;
    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(wsptr[0],16) * (long long)left_cutsi(coeff_int2[0][0],3),18) ;
    accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(wsptr[0],16) * (long long)left_cutsi(coeff_int2[0][1],3),18) ;
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(wsptr[0],16) * (long long)left_cutsi(coeff_int2[0][2],3),18) ;
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(wsptr[0],16) * (long long)left_cutsi(coeff_int2[0][3],3),18) ;
    accumulator[4] = accumulator[4] + (int)round_cutsl((long long)left_cutsi(wsptr[0],16) * (long long)left_cutsi(coeff_int2[0][3],3),18) ;
    accumulator[5] = accumulator[5] + (int)round_cutsl((long long)left_cutsi(wsptr[0],16) * (long long)left_cutsi(coeff_int2[0][2],3),18) ;
    accumulator[6] = accumulator[6] + (int)round_cutsl((long long)left_cutsi(wsptr[0],16) * (long long)left_cutsi(coeff_int2[0][1],3),18) ;
    accumulator[7] = accumulator[7] + (int)round_cutsl((long long)left_cutsi(wsptr[0],16) * (long long)left_cutsi(coeff_int2[0][0],3),18) ;
    
    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(wsptr[1],16) * (long long)left_cutsi(coeff_int2[1][0],3),18) ; 
    accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(wsptr[1],16) * (long long)left_cutsi(coeff_int2[1][1],3),18) ; 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(wsptr[1],16) * (long long)left_cutsi(coeff_int2[1][2],3),18) ; 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(wsptr[1],16) * (long long)left_cutsi(coeff_int2[1][3],3),18) ; 
    accumulator[4] = accumulator[4] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[1],16) * (long long)left_cutsi(coeff_int2[1][3],3),18) ;  
    accumulator[5] = accumulator[5] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[1],16) * (long long)left_cutsi(coeff_int2[1][2],3),18) ; 
    accumulator[6] = accumulator[6] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[1],16) * (long long)left_cutsi(coeff_int2[1][1],3),18) ; 
    accumulator[7] = accumulator[7] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[1],16) * (long long)left_cutsi(coeff_int2[1][0],3),18) ; 
    
    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(wsptr[2],16) * (long long)left_cutsi(coeff_int2[2][0],3),18) ; 
    accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(wsptr[2],16) * (long long)left_cutsi(coeff_int2[2][1],3),18) ; 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(wsptr[2],16) * (long long)left_cutsi(coeff_int2[2][2],3),18) ; 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(wsptr[2],16) * (long long)left_cutsi(coeff_int2[2][3],3),18) ; 
    accumulator[4] = accumulator[4] + (int)round_cutsl((long long)left_cutsi(wsptr[2],16) * (long long)left_cutsi(coeff_int2[2][3],3),18) ; 
    accumulator[5] = accumulator[5] + (int)round_cutsl((long long)left_cutsi(wsptr[2],16) * (long long)left_cutsi(coeff_int2[2][2],3),18) ; 
    accumulator[6] = accumulator[6] + (int)round_cutsl((long long)left_cutsi(wsptr[2],16) * (long long)left_cutsi(coeff_int2[2][1],3),18) ; 
    accumulator[7] = accumulator[7] + (int)round_cutsl((long long)left_cutsi(wsptr[2],16) * (long long)left_cutsi(coeff_int2[2][0],3),18) ; 
    
    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(wsptr[3],16) * (long long)left_cutsi(coeff_int2[3][0],3),18) ; 
    accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(wsptr[3],16) * (long long)left_cutsi(coeff_int2[3][1],3),18) ; 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(wsptr[3],16) * (long long)left_cutsi(coeff_int2[3][2],3),18) ; 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(wsptr[3],16) * (long long)left_cutsi(coeff_int2[3][3],3),18) ; 
    accumulator[4] = accumulator[4] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[3],16) * (long long)left_cutsi(coeff_int2[3][3],3),18) ; 
    accumulator[5] = accumulator[5] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[3],16) * (long long)left_cutsi(coeff_int2[3][2],3),18) ; 
    accumulator[6] = accumulator[6] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[3],16) * (long long)left_cutsi(coeff_int2[3][1],3),18) ; 
    accumulator[7] = accumulator[7] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[3],16) * (long long)left_cutsi(coeff_int2[3][0],3),18) ; 
    
    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(wsptr[4],16) * (long long)left_cutsi(coeff_int2[4][0],3),18) ; 
    accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(wsptr[4],16) * (long long)left_cutsi(coeff_int2[4][1],3),18) ; 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(wsptr[4],16) * (long long)left_cutsi(coeff_int2[4][2],3),18) ; 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(wsptr[4],16) * (long long)left_cutsi(coeff_int2[4][3],3),18) ; 
    accumulator[4] = accumulator[4] + (int)round_cutsl((long long)left_cutsi(wsptr[4],16) * (long long)left_cutsi(coeff_int2[4][3],3),18) ; 
    accumulator[5] = accumulator[5] + (int)round_cutsl((long long)left_cutsi(wsptr[4],16) * (long long)left_cutsi(coeff_int2[4][2],3),18) ; 
    accumulator[6] = accumulator[6] + (int)round_cutsl((long long)left_cutsi(wsptr[4],16) * (long long)left_cutsi(coeff_int2[4][1],3),18) ; 
    accumulator[7] = accumulator[7] + (int)round_cutsl((long long)left_cutsi(wsptr[4],16) * (long long)left_cutsi(coeff_int2[4][0],3),18) ; 
    
    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(wsptr[5],16) * (long long)left_cutsi(coeff_int2[5][0],3),18) ; 
    accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(wsptr[5],16) * (long long)left_cutsi(coeff_int2[5][1],3),18) ; 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(wsptr[5],16) * (long long)left_cutsi(coeff_int2[5][2],3),18) ; 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(wsptr[5],16) * (long long)left_cutsi(coeff_int2[5][3],3),18) ; 
    accumulator[4] = accumulator[4] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[5],16) * (long long)left_cutsi(coeff_int2[5][3],3),18) ; 
    accumulator[5] = accumulator[5] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[5],16) * (long long)left_cutsi(coeff_int2[5][2],3),18) ; 
    accumulator[6] = accumulator[6] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[5],16) * (long long)left_cutsi(coeff_int2[5][1],3),18) ; 
    accumulator[7] = accumulator[7] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[5],16) * (long long)left_cutsi(coeff_int2[5][0],3),18) ; 
    
    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(wsptr[6],16) * (long long)left_cutsi(coeff_int2[6][0],3),18) ; 
    accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(wsptr[6],16) * (long long)left_cutsi(coeff_int2[6][1],3),18) ; 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(wsptr[6],16) * (long long)left_cutsi(coeff_int2[6][2],3),18) ; 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(wsptr[6],16) * (long long)left_cutsi(coeff_int2[6][3],3),18) ; 
    accumulator[4] = accumulator[4] + (int)round_cutsl((long long)left_cutsi(wsptr[6],16) * (long long)left_cutsi(coeff_int2[6][3],3),18) ; 
    accumulator[5] = accumulator[5] + (int)round_cutsl((long long)left_cutsi(wsptr[6],16) * (long long)left_cutsi(coeff_int2[6][2],3),18) ; 
    accumulator[6] = accumulator[6] + (int)round_cutsl((long long)left_cutsi(wsptr[6],16) * (long long)left_cutsi(coeff_int2[6][1],3),18) ; 
    accumulator[7] = accumulator[7] + (int)round_cutsl((long long)left_cutsi(wsptr[6],16) * (long long)left_cutsi(coeff_int2[6][0],3),18) ; 
    
    accumulator[0] = accumulator[0] + (int)round_cutsl((long long)left_cutsi(wsptr[7],16) * (long long)left_cutsi(coeff_int2[7][0],3),18) ; 
    accumulator[1] = accumulator[1] + (int)round_cutsl((long long)left_cutsi(wsptr[7],16) * (long long)left_cutsi(coeff_int2[7][1],3),18) ; 
    accumulator[2] = accumulator[2] + (int)round_cutsl((long long)left_cutsi(wsptr[7],16) * (long long)left_cutsi(coeff_int2[7][2],3),18) ; 
    accumulator[3] = accumulator[3] + (int)round_cutsl((long long)left_cutsi(wsptr[7],16) * (long long)left_cutsi(coeff_int2[7][3],3),18) ; 
    accumulator[4] = accumulator[4] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[7],16) * (long long)left_cutsi(coeff_int2[7][3],3),18) ; 
    accumulator[5] = accumulator[5] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[7],16) * (long long)left_cutsi(coeff_int2[7][2],3),18) ; 
    accumulator[6] = accumulator[6] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[7],16) * (long long)left_cutsi(coeff_int2[7][1],3),18) ; 
    accumulator[7] = accumulator[7] + (int)round_cutsl((-1)*(long long)left_cutsi(wsptr[7],16) * (long long)left_cutsi(coeff_int2[7][0],3),18) ; 
    
    outptr[0] = (JSAMPLE)(ranlim_new(round_cutsi(left_cutsi(accumulator[0],18),5))+128);
    outptr[1] = (JSAMPLE)(ranlim_new(round_cutsi(left_cutsi(accumulator[1],18),5))+128);
    outptr[2] = (JSAMPLE)(ranlim_new(round_cutsi(left_cutsi(accumulator[2],18),5))+128);
    outptr[3] = (JSAMPLE)(ranlim_new(round_cutsi(left_cutsi(accumulator[3],18),5))+128);
    outptr[4] = (JSAMPLE)(ranlim_new(round_cutsi(left_cutsi(accumulator[4],18),5))+128);
    outptr[5] = (JSAMPLE)(ranlim_new(round_cutsi(left_cutsi(accumulator[5],18),5))+128);
    outptr[6] = (JSAMPLE)(ranlim_new(round_cutsi(left_cutsi(accumulator[6],18),5))+128);
    outptr[7] = (JSAMPLE)(ranlim_new(round_cutsi(left_cutsi(accumulator[7],18),5))+128);
       
    fprintf(cinfo->out_dct,"%02X\n",(0x0000ffff & outptr[0]));
    fprintf(cinfo->out_dct,"%02X\n",(0x0000ffff & outptr[1]));
    fprintf(cinfo->out_dct,"%02X\n",(0x0000ffff & outptr[2]));
    fprintf(cinfo->out_dct,"%02X\n",(0x0000ffff & outptr[3]));
    fprintf(cinfo->out_dct,"%02X\n",(0x0000ffff & outptr[4]));
    fprintf(cinfo->out_dct,"%02X\n",(0x0000ffff & outptr[5]));
    fprintf(cinfo->out_dct,"%02X\n",(0x0000ffff & outptr[6]));
    fprintf(cinfo->out_dct,"%02X\n",(0x0000ffff & outptr[7]));
    
    for(i=0; i<8; i++){
      accumulator[i] = 0.0;
    }
    wsptr+= 8;
  }
#endif
  

#ifdef IDCTHARD
  double accumulator[8]={0,0,0,0,0,0,0,0};
 
  volatile JCOEFPTR kinptr;
  volatile ISLOW_MULT_TYPE *kquantptr;
 
  /* Pass 1: process columns from input, store into work array. */
  /* Note results are scaled up by sqrt(8) compared to a true IDCT; */
  /* furthermore, we scale the results by 2**PASS1_BITS. */

  kinptr = coef_block;
  kquantptr = (ISLOW_MULT_TYPE *)compptr->dct_table;
  inptr = coef_block;
  quantptr = (ISLOW_MULT_TYPE *) compptr->dct_table;
  wsptr = workspace;

   for(ctr = 0; ctr< 8; ctr++){
     
      accumulator[0] = accumulator[0] + coeff_matrix[0][0] * (inptr[0] * quantptr[0]);
      accumulator[1] = accumulator[1] + coeff_matrix[0][1] * (inptr[0] * quantptr[0]); 
      accumulator[2] = accumulator[2] + coeff_matrix[0][2] * (inptr[0] * quantptr[0]);
      accumulator[3] = accumulator[3] + coeff_matrix[0][3] * (inptr[0] * quantptr[0]);
      accumulator[4] = accumulator[4] + coeff_matrix[0][3] * (inptr[0] * quantptr[0]); 
      accumulator[5] = accumulator[5] + coeff_matrix[0][2] * (inptr[0] * quantptr[0]); 
      accumulator[6] = accumulator[6] + coeff_matrix[0][1] * (inptr[0] * quantptr[0]); 
      accumulator[7] = accumulator[7] + coeff_matrix[0][0] * (inptr[0] * quantptr[0]);

      accumulator[0] = accumulator[0] + coeff_matrix[1][0] * (inptr[8] * quantptr[8]);
      accumulator[1] = accumulator[1] + coeff_matrix[1][1] * (inptr[8] * quantptr[8]); 
      accumulator[2] = accumulator[2] + coeff_matrix[1][2] * (inptr[8] * quantptr[8]);
      accumulator[3] = accumulator[3] + coeff_matrix[1][3] * (inptr[8] * quantptr[8]);
      accumulator[4] = accumulator[4] - coeff_matrix[1][3] * (inptr[8] * quantptr[8]); 
      accumulator[5] = accumulator[5] - coeff_matrix[1][2] * (inptr[8] * quantptr[8]); 
      accumulator[6] = accumulator[6] - coeff_matrix[1][1] * (inptr[8] * quantptr[8]); 
      accumulator[7] = accumulator[7] - coeff_matrix[1][0] * (inptr[8] * quantptr[8]);

      accumulator[0] = accumulator[0] + coeff_matrix[2][0] * (inptr[16] * quantptr[16]);
      accumulator[1] = accumulator[1] + coeff_matrix[2][1] * (inptr[16] * quantptr[16]); 
      accumulator[2] = accumulator[2] + coeff_matrix[2][2] * (inptr[16] * quantptr[16]);
      accumulator[3] = accumulator[3] + coeff_matrix[2][3] * (inptr[16] * quantptr[16]);
      accumulator[4] = accumulator[4] + coeff_matrix[2][3] * (inptr[16] * quantptr[16]); 
      accumulator[5] = accumulator[5] + coeff_matrix[2][2] * (inptr[16] * quantptr[16]); 
      accumulator[6] = accumulator[6] + coeff_matrix[2][1] * (inptr[16] * quantptr[16]); 
      accumulator[7] = accumulator[7] + coeff_matrix[2][0] * (inptr[16] * quantptr[16]); 

      accumulator[0] = accumulator[0] + coeff_matrix[3][0] * (inptr[24] * quantptr[24]);
      accumulator[1] = accumulator[1] + coeff_matrix[3][1] * (inptr[24] * quantptr[24]); 
      accumulator[2] = accumulator[2] + coeff_matrix[3][2] * (inptr[24] * quantptr[24]);
      accumulator[3] = accumulator[3] + coeff_matrix[3][3] * (inptr[24] * quantptr[24]);
      accumulator[4] = accumulator[4] - coeff_matrix[3][3] * (inptr[24] * quantptr[24]); 
      accumulator[5] = accumulator[5] - coeff_matrix[3][2] * (inptr[24] * quantptr[24]); 
      accumulator[6] = accumulator[6] - coeff_matrix[3][1] * (inptr[24] * quantptr[24]); 
      accumulator[7] = accumulator[7] - coeff_matrix[3][0] * (inptr[24] * quantptr[24]); 
 
      accumulator[0] = accumulator[0] + coeff_matrix[4][0] * (inptr[32] * quantptr[32]);
      accumulator[1] = accumulator[1] + coeff_matrix[4][1] * (inptr[32] * quantptr[32]); 
      accumulator[2] = accumulator[2] + coeff_matrix[4][2] * (inptr[32] * quantptr[32]);
      accumulator[3] = accumulator[3] + coeff_matrix[4][3] * (inptr[32] * quantptr[32]);
      accumulator[4] = accumulator[4] + coeff_matrix[4][3] * (inptr[32] * quantptr[32]); 
      accumulator[5] = accumulator[5] + coeff_matrix[4][2] * (inptr[32] * quantptr[32]); 
      accumulator[6] = accumulator[6] + coeff_matrix[4][1] * (inptr[32] * quantptr[32]); 
      accumulator[7] = accumulator[7] + coeff_matrix[4][0] * (inptr[32] * quantptr[32]);  

      accumulator[0] = accumulator[0] + coeff_matrix[5][0] * (inptr[40] * quantptr[40]);
      accumulator[1] = accumulator[1] + coeff_matrix[5][1] * (inptr[40] * quantptr[40]); 
      accumulator[2] = accumulator[2] + coeff_matrix[5][2] * (inptr[40] * quantptr[40]);
      accumulator[3] = accumulator[3] + coeff_matrix[5][3] * (inptr[40] * quantptr[40]);
      accumulator[4] = accumulator[4] - coeff_matrix[5][3] * (inptr[40] * quantptr[40]); 
      accumulator[5] = accumulator[5] - coeff_matrix[5][2] * (inptr[40] * quantptr[40]); 
      accumulator[6] = accumulator[6] - coeff_matrix[5][1] * (inptr[40] * quantptr[40]); 
      accumulator[7] = accumulator[7] - coeff_matrix[5][0] * (inptr[40] * quantptr[40]); 

      accumulator[0] = accumulator[0] + coeff_matrix[6][0] * (inptr[48] * quantptr[48]);
      accumulator[1] = accumulator[1] + coeff_matrix[6][1] * (inptr[48] * quantptr[48]); 
      accumulator[2] = accumulator[2] + coeff_matrix[6][2] * (inptr[48] * quantptr[48]);
      accumulator[3] = accumulator[3] + coeff_matrix[6][3] * (inptr[48] * quantptr[48]);
      accumulator[4] = accumulator[4] + coeff_matrix[6][3] * (inptr[48] * quantptr[48]); 
      accumulator[5] = accumulator[5] + coeff_matrix[6][2] * (inptr[48] * quantptr[48]); 
      accumulator[6] = accumulator[6] + coeff_matrix[6][1] * (inptr[48] * quantptr[48]); 
      accumulator[7] = accumulator[7] + coeff_matrix[6][0] * (inptr[48] * quantptr[48]); 

      accumulator[0] = accumulator[0] + coeff_matrix[7][0] * (inptr[56] * quantptr[56]);
      accumulator[1] = accumulator[1] + coeff_matrix[7][1] * (inptr[56] * quantptr[56]); 
      accumulator[2] = accumulator[2] + coeff_matrix[7][2] * (inptr[56] * quantptr[56]);
      accumulator[3] = accumulator[3] + coeff_matrix[7][3] * (inptr[56] * quantptr[56]);
      accumulator[4] = accumulator[4] - coeff_matrix[7][3] * (inptr[56] * quantptr[56]); 
      accumulator[5] = accumulator[5] - coeff_matrix[7][2] * (inptr[56] * quantptr[56]); 
      accumulator[6] = accumulator[6] - coeff_matrix[7][1] * (inptr[56] * quantptr[56]); 
      accumulator[7] = accumulator[7] - coeff_matrix[7][0] * (inptr[56] * quantptr[56]); 

      wsptr[0] = accumulator[0];
      wsptr[1] = accumulator[1];
      wsptr[2] = accumulator[2];
      wsptr[3] = accumulator[3];
      wsptr[4] = accumulator[4];
      wsptr[5] = accumulator[5];
      wsptr[6] = accumulator[6];
      wsptr[7] = accumulator[7];
      for(i=0; i<8; i++){
	accumulator[i] = 0.0;
      }

    inptr++;
    quantptr++;
    wsptr+=8;
  }
 
  wsptr = workspace;
  for(ctr = 0; ctr<8; ctr++){
    outptr = output_buf[ctr] + output_col;
    accumulator[0] = accumulator[0] + wsptr[0] * coeff_matrix[0][0] ;
    accumulator[1] = accumulator[1] + wsptr[0] * coeff_matrix[0][1] ;
    accumulator[2] = accumulator[2] + wsptr[0] * coeff_matrix[0][2] ;
    accumulator[3] = accumulator[3] + wsptr[0] * coeff_matrix[0][3] ;
    accumulator[4] = accumulator[4] + wsptr[0] * coeff_matrix[0][3] ;
    accumulator[5] = accumulator[5] + wsptr[0] * coeff_matrix[0][2] ;
    accumulator[6] = accumulator[6] + wsptr[0] * coeff_matrix[0][1] ;
    accumulator[7] = accumulator[7] + wsptr[0] * coeff_matrix[0][0] ;
    
    accumulator[0] = accumulator[0] + wsptr[8] * coeff_matrix[1][0] ;
    accumulator[1] = accumulator[1] + wsptr[8] * coeff_matrix[1][1] ;
    accumulator[2] = accumulator[2] + wsptr[8] * coeff_matrix[1][2] ;
    accumulator[3] = accumulator[3] + wsptr[8] * coeff_matrix[1][3] ;
    accumulator[4] = accumulator[4] - wsptr[8] * coeff_matrix[1][3] ;
    accumulator[5] = accumulator[5] - wsptr[8] * coeff_matrix[1][2] ;
    accumulator[6] = accumulator[6] - wsptr[8] * coeff_matrix[1][1] ;
    accumulator[7] = accumulator[7] - wsptr[8] * coeff_matrix[1][0] ;
    
    accumulator[0] = accumulator[0] + wsptr[16] * coeff_matrix[2][0] ;
    accumulator[1] = accumulator[1] + wsptr[16] * coeff_matrix[2][1] ;
    accumulator[2] = accumulator[2] + wsptr[16] * coeff_matrix[2][2] ;
    accumulator[3] = accumulator[3] + wsptr[16] * coeff_matrix[2][3] ;
    accumulator[4] = accumulator[4] + wsptr[16] * coeff_matrix[2][3] ;
    accumulator[5] = accumulator[5] + wsptr[16] * coeff_matrix[2][2] ;
    accumulator[6] = accumulator[6] + wsptr[16] * coeff_matrix[2][1] ;
    accumulator[7] = accumulator[7] + wsptr[16] * coeff_matrix[2][0] ;
    
    accumulator[0] = accumulator[0] + wsptr[24] * coeff_matrix[3][0] ;
    accumulator[1] = accumulator[1] + wsptr[24] * coeff_matrix[3][1] ;
    accumulator[2] = accumulator[2] + wsptr[24] * coeff_matrix[3][2] ;
    accumulator[3] = accumulator[3] + wsptr[24] * coeff_matrix[3][3] ;
    accumulator[4] = accumulator[4] - wsptr[24] * coeff_matrix[3][3] ;
    accumulator[5] = accumulator[5] - wsptr[24] * coeff_matrix[3][2] ;
    accumulator[6] = accumulator[6] - wsptr[24] * coeff_matrix[3][1] ;
    accumulator[7] = accumulator[7] - wsptr[24] * coeff_matrix[3][0] ;
    
    accumulator[0] = accumulator[0] + wsptr[32] * coeff_matrix[4][0] ;
    accumulator[1] = accumulator[1] + wsptr[32] * coeff_matrix[4][1] ;
    accumulator[2] = accumulator[2] + wsptr[32] * coeff_matrix[4][2] ;
    accumulator[3] = accumulator[3] + wsptr[32] * coeff_matrix[4][3] ;
    accumulator[4] = accumulator[4] + wsptr[32] * coeff_matrix[4][3] ;
    accumulator[5] = accumulator[5] + wsptr[32] * coeff_matrix[4][2] ;
    accumulator[6] = accumulator[6] + wsptr[32] * coeff_matrix[4][1] ;
    accumulator[7] = accumulator[7] + wsptr[32] * coeff_matrix[4][0] ;
    
    accumulator[0] = accumulator[0] + wsptr[40] * coeff_matrix[5][0] ;
    accumulator[1] = accumulator[1] + wsptr[40] * coeff_matrix[5][1] ;
    accumulator[2] = accumulator[2] + wsptr[40] * coeff_matrix[5][2] ;
    accumulator[3] = accumulator[3] + wsptr[40] * coeff_matrix[5][3] ;
    accumulator[4] = accumulator[4] - wsptr[40] * coeff_matrix[5][3] ;
    accumulator[5] = accumulator[5] - wsptr[40] * coeff_matrix[5][2] ;
    accumulator[6] = accumulator[6] - wsptr[40] * coeff_matrix[5][1] ;
    accumulator[7] = accumulator[7] - wsptr[40] * coeff_matrix[5][0] ;
    
    accumulator[0] = accumulator[0] + wsptr[48] * coeff_matrix[6][0] ;
    accumulator[1] = accumulator[1] + wsptr[48] * coeff_matrix[6][1] ;
    accumulator[2] = accumulator[2] + wsptr[48] * coeff_matrix[6][2] ;
    accumulator[3] = accumulator[3] + wsptr[48] * coeff_matrix[6][3] ;
    accumulator[4] = accumulator[4] + wsptr[48] * coeff_matrix[6][3] ;
    accumulator[5] = accumulator[5] + wsptr[48] * coeff_matrix[6][2] ;
    accumulator[6] = accumulator[6] + wsptr[48] * coeff_matrix[6][1] ;
    accumulator[7] = accumulator[7] + wsptr[48] * coeff_matrix[6][0] ;
    
    accumulator[0] = accumulator[0] + wsptr[56] * coeff_matrix[7][0] ;
    accumulator[1] = accumulator[1] + wsptr[56] * coeff_matrix[7][1] ;
    accumulator[2] = accumulator[2] + wsptr[56] * coeff_matrix[7][2] ;
    accumulator[3] = accumulator[3] + wsptr[56] * coeff_matrix[7][3] ;
    accumulator[4] = accumulator[4] - wsptr[56] * coeff_matrix[7][3] ;
    accumulator[5] = accumulator[5] - wsptr[56] * coeff_matrix[7][2] ;
    accumulator[6] = accumulator[6] - wsptr[56] * coeff_matrix[7][1] ;
    accumulator[7] = accumulator[7] - wsptr[56] * coeff_matrix[7][0] ;
    
    for(i=0;i<8;i++){
      if(accumulator[i]> 127){
	accumulator[i] = 127;
      }
      if(accumulator[i] < -128){
	accumulator[i] = -128;
      }
    }
    outptr[0] = (JSAMPLE)accumulator[0]+128;
    outptr[1] = (JSAMPLE)accumulator[1]+128;
    outptr[2] = (JSAMPLE)accumulator[2]+128;
    outptr[3] = (JSAMPLE)accumulator[3]+128;
    outptr[4] = (JSAMPLE)accumulator[4]+128;
    outptr[5] = (JSAMPLE)accumulator[5]+128;
    outptr[6] = (JSAMPLE)accumulator[6]+128;
    outptr[7] = (JSAMPLE)accumulator[7]+128;

    for(i=0; i<8; i++){
      accumulator[i] = 0.0;
    }
    wsptr++;
  }
#endif

#ifdef IDCTSOFT
  /* Pass 1: process columns from input, store into work array. */
  /* Note results are scaled up by sqrt(8) compared to a true IDCT; */
  /* furthermore, we scale the results by 2**PASS1_BITS. */

  inptr = coef_block;
  quantptr = (ISLOW_MULT_TYPE *) compptr->dct_table;
  wsptr = workspace;


  for (ctr = DCTSIZE; ctr > 0; ctr--) {
    /* Due to quantization, we will usually find that many of the input
     * coefficients are zero, especially the AC terms.  We can exploit this
     * by short-circuiting the IDCT calculation for any column in which all
     * the AC terms are zero.  In that case each output is equal to the
     * DC coefficient (with scale factor as needed).
     * With typical images and quantization tables, half or more of the
     * column DCT calculations can be simplified this way.
     */
    
    if (inptr[DCTSIZE*1] == 0 && inptr[DCTSIZE*2] == 0 &&
	inptr[DCTSIZE*3] == 0 && inptr[DCTSIZE*4] == 0 &&
	inptr[DCTSIZE*5] == 0 && inptr[DCTSIZE*6] == 0 &&
	inptr[DCTSIZE*7] == 0) {
      /* AC terms all zero */
      int dcval = DEQUANTIZE(inptr[DCTSIZE*0], quantptr[DCTSIZE*0]) << PASS1_BITS;
      
      wsptr[DCTSIZE*0] = dcval;
      wsptr[DCTSIZE*1] = dcval;
      wsptr[DCTSIZE*2] = dcval;
      wsptr[DCTSIZE*3] = dcval;
      wsptr[DCTSIZE*4] = dcval;
      wsptr[DCTSIZE*5] = dcval;
      wsptr[DCTSIZE*6] = dcval;
      wsptr[DCTSIZE*7] = dcval;
      
      inptr++;			/* advance pointers to next column */
      quantptr++;
      wsptr++;
      continue;
    }
    
    /* Even part: reverse the even part of the forward DCT. */
    /* The rotator is sqrt(2)*c(-6). */
    
    z2 = DEQUANTIZE(inptr[DCTSIZE*2], quantptr[DCTSIZE*2]);
    z3 = DEQUANTIZE(inptr[DCTSIZE*6], quantptr[DCTSIZE*6]);
    
    z1 = MULTIPLY(z2 + z3, FIX_0_541196100);
    tmp2 = z1 + MULTIPLY(z3, - FIX_1_847759065);
    tmp3 = z1 + MULTIPLY(z2, FIX_0_765366865);
    
    z2 = DEQUANTIZE(inptr[DCTSIZE*0], quantptr[DCTSIZE*0]);
    z3 = DEQUANTIZE(inptr[DCTSIZE*4], quantptr[DCTSIZE*4]);

    tmp0 = (z2 + z3) << CONST_BITS;
    tmp1 = (z2 - z3) << CONST_BITS;
    
    tmp10 = tmp0 + tmp3;
    tmp13 = tmp0 - tmp3;
    tmp11 = tmp1 + tmp2;
    tmp12 = tmp1 - tmp2;
    
    /* Odd part per figure 8; the matrix is unitary and hence its
     * transpose is its inverse.  i0..i3 are y7,y5,y3,y1 respectively.
     */
    
    tmp0 = DEQUANTIZE(inptr[DCTSIZE*7], quantptr[DCTSIZE*7]);
    tmp1 = DEQUANTIZE(inptr[DCTSIZE*5], quantptr[DCTSIZE*5]);
    tmp2 = DEQUANTIZE(inptr[DCTSIZE*3], quantptr[DCTSIZE*3]);
    tmp3 = DEQUANTIZE(inptr[DCTSIZE*1], quantptr[DCTSIZE*1]);
    
    z1 = tmp0 + tmp3;
    z2 = tmp1 + tmp2;
    z3 = tmp0 + tmp2;
    z4 = tmp1 + tmp3;
    z5 = MULTIPLY(z3 + z4, FIX_1_175875602); /* sqrt(2) * c3 */
    
    tmp0 = MULTIPLY(tmp0, FIX_0_298631336); /* sqrt(2) * (-c1+c3+c5-c7) */
    tmp1 = MULTIPLY(tmp1, FIX_2_053119869); /* sqrt(2) * ( c1+c3-c5+c7) */
    tmp2 = MULTIPLY(tmp2, FIX_3_072711026); /* sqrt(2) * ( c1+c3+c5-c7) */
    tmp3 = MULTIPLY(tmp3, FIX_1_501321110); /* sqrt(2) * ( c1+c3-c5-c7) */
    z1 = MULTIPLY(z1, - FIX_0_899976223); /* sqrt(2) * (c7-c3) */
    z2 = MULTIPLY(z2, - FIX_2_562915447); /* sqrt(2) * (-c1-c3) */
    z3 = MULTIPLY(z3, - FIX_1_961570560); /* sqrt(2) * (-c3-c5) */
    z4 = MULTIPLY(z4, - FIX_0_390180644); /* sqrt(2) * (c5-c3) */
    
    z3 += z5;
    z4 += z5;
    
    tmp0 += z1 + z3;
    tmp1 += z2 + z4;
    tmp2 += z2 + z3;
    tmp3 += z1 + z4;
    
    /* Final output stage: inputs are tmp10..tmp13, tmp0..tmp3 */
    
    wsptr[DCTSIZE*0] = (int) DESCALE(tmp10 + tmp3, CONST_BITS-PASS1_BITS);
    wsptr[DCTSIZE*7] = (int) DESCALE(tmp10 - tmp3, CONST_BITS-PASS1_BITS);
    wsptr[DCTSIZE*1] = (int) DESCALE(tmp11 + tmp2, CONST_BITS-PASS1_BITS);
    wsptr[DCTSIZE*6] = (int) DESCALE(tmp11 - tmp2, CONST_BITS-PASS1_BITS);
    wsptr[DCTSIZE*2] = (int) DESCALE(tmp12 + tmp1, CONST_BITS-PASS1_BITS);
    wsptr[DCTSIZE*5] = (int) DESCALE(tmp12 - tmp1, CONST_BITS-PASS1_BITS);
    wsptr[DCTSIZE*3] = (int) DESCALE(tmp13 + tmp0, CONST_BITS-PASS1_BITS);
    wsptr[DCTSIZE*4] = (int) DESCALE(tmp13 - tmp0, CONST_BITS-PASS1_BITS);
    
    inptr++;			/* advance pointers to next column */
    quantptr++;
    wsptr++;
  }
  
  /* Pass 2: process rows from work array, store into output array. */
  /* Note that we must descale the results by a factor of 8 == 2**3, */
  /* and also undo the PASS1_BITS scaling. */

  wsptr = workspace;
  for (ctr = 0; ctr < DCTSIZE; ctr++) {
    outptr = output_buf[ctr] + output_col;
    /* Rows of zeroes can be exploited in the same way as we did with columns.
     * However, the column calculation has created many nonzero AC terms, so
     * the simplification applies less often (typically 5% to 10% of the time).
     * On machines with very fast multiplication, it's possible that the
     * test takes more time than it's worth.  In that case this section
     * may be commented out.
     */
    
#ifndef NO_ZERO_ROW_TEST
    if (wsptr[1] == 0 && wsptr[2] == 0 && wsptr[3] == 0 && wsptr[4] == 0 &&
	wsptr[5] == 0 && wsptr[6] == 0 && wsptr[7] == 0) {
      /* AC terms all zero */
      JSAMPLE dcval = range_limit[(int) DESCALE((INT32) wsptr[0], PASS1_BITS+3)
				  & RANGE_MASK];
      
      outptr[0] = dcval;
      outptr[1] = dcval;
      outptr[2] = dcval;
      outptr[3] = dcval;
      outptr[4] = dcval;
      outptr[5] = dcval;
      outptr[6] = dcval;
      outptr[7] = dcval;

      wsptr += DCTSIZE;		/* advance pointer to next row */


    fprintf(cinfo->in_upycc,"%02X\n",outptr[0]);
    fprintf(cinfo->in_upycc,"%02X\n",outptr[1]);
    fprintf(cinfo->in_upycc,"%02X\n",outptr[2]);
    fprintf(cinfo->in_upycc,"%02X\n",outptr[3]);
    fprintf(cinfo->in_upycc,"%02X\n",outptr[4]);
    fprintf(cinfo->in_upycc,"%02X\n",outptr[5]);
    fprintf(cinfo->in_upycc,"%02X\n",outptr[6]);
    fprintf(cinfo->in_upycc,"%02X\n",outptr[7]);
    
      continue;
    }
#endif
    
    /* Even part: reverse the even part of the forward DCT. */
    /* The rotator is sqrt(2)*c(-6). */
    
    z2 = (INT32) wsptr[2];
    z3 = (INT32) wsptr[6];
    
    z1 = MULTIPLY(z2 + z3, FIX_0_541196100);
    tmp2 = z1 + MULTIPLY(z3, - FIX_1_847759065);
    tmp3 = z1 + MULTIPLY(z2, FIX_0_765366865);
    
    tmp0 = ((INT32) wsptr[0] + (INT32) wsptr[4]) << CONST_BITS;
    tmp1 = ((INT32) wsptr[0] - (INT32) wsptr[4]) << CONST_BITS;
    
    tmp10 = tmp0 + tmp3;
    tmp13 = tmp0 - tmp3;
    tmp11 = tmp1 + tmp2;
    tmp12 = tmp1 - tmp2;
    
    /* Odd part per figure 8; the matrix is unitary and hence its
     * transpose is its inverse.  i0..i3 are y7,y5,y3,y1 respectively.
     */
    
    tmp0 = (INT32) wsptr[7];
    tmp1 = (INT32) wsptr[5];
    tmp2 = (INT32) wsptr[3];
    tmp3 = (INT32) wsptr[1];
    
    z1 = tmp0 + tmp3;
    z2 = tmp1 + tmp2;
    z3 = tmp0 + tmp2;
    z4 = tmp1 + tmp3;
    z5 = MULTIPLY(z3 + z4, FIX_1_175875602); /* sqrt(2) * c3 */
    
    tmp0 = MULTIPLY(tmp0, FIX_0_298631336); /* sqrt(2) * (-c1+c3+c5-c7) */
    tmp1 = MULTIPLY(tmp1, FIX_2_053119869); /* sqrt(2) * ( c1+c3-c5+c7) */
    tmp2 = MULTIPLY(tmp2, FIX_3_072711026); /* sqrt(2) * ( c1+c3+c5-c7) */
    tmp3 = MULTIPLY(tmp3, FIX_1_501321110); /* sqrt(2) * ( c1+c3-c5-c7) */
    z1 = MULTIPLY(z1, - FIX_0_899976223); /* sqrt(2) * (c7-c3) */
    z2 = MULTIPLY(z2, - FIX_2_562915447); /* sqrt(2) * (-c1-c3) */
    z3 = MULTIPLY(z3, - FIX_1_961570560); /* sqrt(2) * (-c3-c5) */
    z4 = MULTIPLY(z4, - FIX_0_390180644); /* sqrt(2) * (c5-c3) */
    
    z3 += z5;
    z4 += z5;
    
    tmp0 += z1 + z3;
    tmp1 += z2 + z4;
    tmp2 += z2 + z3;
    tmp3 += z1 + z4;
    
    /* Final output stage: inputs are tmp10..tmp13, tmp0..tmp3 */
    
    outptr[0] = range_limit[(int) DESCALE(tmp10 + tmp3,
					  CONST_BITS+PASS1_BITS+3)
			    & RANGE_MASK];
    outptr[7] = range_limit[(int) DESCALE(tmp10 - tmp3,
					  CONST_BITS+PASS1_BITS+3)
			    & RANGE_MASK];
    outptr[1] = range_limit[(int) DESCALE(tmp11 + tmp2,
					  CONST_BITS+PASS1_BITS+3)
			    & RANGE_MASK];
    outptr[6] = range_limit[(int) DESCALE(tmp11 - tmp2,
					  CONST_BITS+PASS1_BITS+3)
			    & RANGE_MASK];
    outptr[2] = range_limit[(int) DESCALE(tmp12 + tmp1,
					  CONST_BITS+PASS1_BITS+3)
			    & RANGE_MASK];
    outptr[5] = range_limit[(int) DESCALE(tmp12 - tmp1,
					  CONST_BITS+PASS1_BITS+3)
			    & RANGE_MASK];
    outptr[3] = range_limit[(int) DESCALE(tmp13 + tmp0,
					  CONST_BITS+PASS1_BITS+3)
			    & RANGE_MASK];
    outptr[4] = range_limit[(int) DESCALE(tmp13 - tmp0,
					  CONST_BITS+PASS1_BITS+3)
			    & RANGE_MASK];
    
    wsptr += DCTSIZE;		/* advance pointer to next row */

    fprintf(cinfo->in_upycc,"%02X\n",outptr[0]);
    fprintf(cinfo->in_upycc,"%02X\n",outptr[1]);
    fprintf(cinfo->in_upycc,"%02X\n",outptr[2]);
    fprintf(cinfo->in_upycc,"%02X\n",outptr[3]);
    fprintf(cinfo->in_upycc,"%02X\n",outptr[4]);
    fprintf(cinfo->in_upycc,"%02X\n",outptr[5]);
    fprintf(cinfo->in_upycc,"%02X\n",outptr[6]);
    fprintf(cinfo->in_upycc,"%02X\n",outptr[7]);
  }
#endif

}

