﻿//#define TEST_ASYNC

using System;
using System.IO;
using System.Runtime.Serialization;
using System.Text.RegularExpressions;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Shell;
using EncodeDetect;
using Prop = FooEditor.Properties;
using FooEditEngine;
using FooEditEngine.WPF;
using System.Runtime.CompilerServices;
using Microsoft.WindowsAPICodePack.Dialogs;

namespace FooEditor
{
    /// <summary>
    /// 文章タイプが変化している時に送られてくるイベント
    /// </summary>
    public class DocumentTypeChangeingEventArgs : EventArgs
    {
        /// <summary>
        /// 変更後の文章タイプ
        /// </summary>
        public string Type;
        /// <summary>
        /// イベント処理後にハイライター、フォールディングを割り当てる必要がないなら真を設定する
        /// </summary>
        public bool Handled = false;
        public DocumentTypeChangeingEventArgs(string type)
        {
            this.Type = type;
        }
    }
    /// <summary>
    /// DocumentWindow.xaml の相互作用ロジック
    /// </summary>
    [Serializable]
    public partial class DocumentWindow : UserControl,INotifyPropertyChanged,ISerializable,IDisposable
    {
        DocumentWindowModel model;

        /// <summary>
        /// コンストラクター
        /// </summary>
        public DocumentWindow()
        {
            InitializeComponent();
            this.PropertyChanged +=new PropertyChangedEventHandler((s,e)=>{});

            this.model = new DocumentWindowModel(this.TextBox);
            this.model.DirtyChanged += model_DirtyChanged;
            this.model.DocumentTypeChangeing += (s, e) => {
                if(this.DocumentTypeChangeing != null)
                    this.DocumentTypeChangeing(this, e);
            };
            this.model.DocumentTypeChanged += (s, e) => {
                if(this.DocumentTypeChanged != null)
                    this.DocumentTypeChanged(this, e);
            };

            Config config = Config.GetInstance();
            this.ApplyConfig(config);

            this.ExtraDataCollection = new Dictionary<string, object>();

            this.Progress += (s, e) => { };

        }

        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="number"></param>
        public DocumentWindow(int number) : this()
        {
            this.Title = string.Format(Prop.Resources.NewDocumentTitle,number);
        }

        /// <summary>
        /// 現在開いているドキュメントのパス
        /// </summary>
        public string FilePath
        {
            get { return this.model.FilePath; }
            set
            {
                this.model.FilePath = value;
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// ドキュメントのタイトル
        /// </summary>
        public string Title
        {
            get { return this.model.Title; }
            set
            {
                this.model.Title = value;
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// 現在開いているファイルのエンコーディング
        /// </summary>
        public Encoding Encoding
        {
            get { return this.model.Encoding; }
            set
            {
                this.model.Encoding = value;
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// 現在開いているファイルの改行コード
        /// </summary>
        public LineFeedType LineFeed
        {
            get { return this.model.LineFeed; }
            set
            {
                this.model.LineFeed = value;
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// 現在開いているファイルの文章タイプ
        /// </summary>
        public string DocumentType
        {
            get { return this.model.DocumentType; }
            set
            {
                this.model.DocumentType = value;
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// ドキュメントが変更されたなら真
        /// </summary>
        public bool Dirty
        {
            get { return this.model.Dirty; }
            set
            {
                this.model.Dirty = value;
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// DockPanelを表す
        /// </summary>
        internal DockPanel LayoutRoot
        {
            get { return this.DockPanel; }
        }

        /// <summary>
        /// TextBoxを表す
        /// </summary>
        public FooTextBox TextBox
        {
            get { return this.FooTextBox; }
        }

        /// <summary>
        /// 保持しているキーワードを表す
        /// </summary>
        public SyntaxDefnition SynataxDefnition
        {
            get
            {
                return this.model.SynataxDefnition;
            }
        }

        /// <summary>
        /// <summary>
        /// オートコンプリートボックスへのインスタンスを表す
        /// </summary>
        public AutocompleteBox CompleteBox
        {
            get;
            set;
        }

        /// <summary>
        /// プロパティの変更を通知する
        /// </summary>
        /// <param name="name"></param>
        public void OnPropertyChanged([CallerMemberName] string name = "")
        {
            if (this.PropertyChanged != null)
                this.PropertyChanged(this, new PropertyChangedEventArgs(name));
        }

        /// <summary>
        /// DocumentWindowに対応する付属のデーターを表す
        /// </summary>
        public Dictionary<string, object> ExtraDataCollection
        {
            get;
            private set;
        }

        /// <summary>
        /// 文章タイプが変更されたことを通知する
        /// </summary>
        public event EventHandler DocumentTypeChanged;

        /// <summary>
        /// 文章タイプが変更されようとしていることを通知する
        /// </summary>
        public event EventHandler<DocumentTypeChangeingEventArgs> DocumentTypeChangeing;

        /// <summary>
        /// 非同期操作の進行状況を表す
        /// </summary>
        public event EventHandler<ProgressEventArgs> Progress;

        /// <summary>
        /// 設定を適用する
        /// </summary>
        /// <param name="config">Configインスタンス</param>
        public void ApplyConfig(Config config)
        {
            this.model.ApplyConfig(config);
        }

        /// <summary>
        /// ファイルを読み取る
        /// </summary>
        /// <param name="filepath">対象となるファイルへのフルパス</param>
        /// <param name="enc">エンコーディング。nullの場合は自動判定</param>
        /// <returns>Taskオブジェクト</returns>
        public async Task LoadAsync(string filepath, Encoding enc)
        {
            this.Progress(this, new ProgressEventArgs(ProgressState.Start));
            await this.model.LoadAsync(filepath, enc);
            this.OnPropertyChanged("FilePath"); //モデル内で更新したので
            this.OnPropertyChanged("Title");    //モデル内で更新したので
            this.Progress(this, new ProgressEventArgs(ProgressState.Complete));
        }

        /// <summary>
        /// ファイルを保存する
        /// </summary>
        /// <param name="filepath">対象となるファイル</param>
        /// <param name="enc">文字コードをしていする</param>
        /// <returns>Taskオブジェクト</returns>
        public async Task SaveAsync(string filepath, Encoding enc)
        {
            await this.model.SaveAsync(filepath, enc);
            this.OnPropertyChanged("FilePath"); //モデル内で更新したので
            this.OnPropertyChanged("Title");    //モデル内で更新したので
        }

        void model_DirtyChanged(object sender, EventArgs e)
        {
            this.PropertyChanged(this, new PropertyChangedEventArgs("Dirty"));
        }

        /// <summary>
        /// オブジェクトを破棄する
        /// </summary>
        public void Dispose()
        {
            this.FooTextBox.Dispose();
        }

        public event PropertyChangedEventHandler PropertyChanged;

        public DocumentWindow(SerializationInfo info, StreamingContext context) : this()
        {
            this.FilePath = info.GetString("FilePath");
            
            this.Title = info.GetString("Title");
            
            this.DocumentType = info.GetString("DocumentType");
            
            this.Encoding = Encoding.GetEncoding(info.GetInt32("Encoding"));
            
            this.TextBox.InsertMode = info.GetBoolean("InsertMode");
            
            this.TextBox.RectSelectMode = info.GetBoolean("RectSelectMode");

            int maxCount = info.GetInt32("LineCount");
            this.TextBox.Document.FireUpdateEvent = false;
            this.TextBox.Document.UndoManager.BeginLock();
            for (int i = 0; i < maxCount; i++)
                this.TextBox.Document.Append(info.GetString(i.ToString()));
            this.TextBox.Document.UndoManager.EndLock();
            this.TextBox.Document.FireUpdateEvent = true;
            
            TextPoint tp = new TextPoint();
            tp.row = info.GetInt32("row");
            tp.col = info.GetInt32("col");
            this.TextBox.Loaded += new RoutedEventHandler((s,e)=>{
                this.TextBox.JumpCaret(tp.row, tp.col);
                this.TextBox.Refresh();
            });
        }

        public void GetObjectData(SerializationInfo info, StreamingContext context)
        {
            info.AddValue("row", this.TextBox.CaretPostion.row);
            
            info.AddValue("col", this.TextBox.CaretPostion.col);
            
            info.AddValue("RectSelectMode", this.TextBox.RectSelectMode);
            
            info.AddValue("Title", this.Title);
            
            info.AddValue("FilePath", this.FilePath);
            
            info.AddValue("DocumentType", this.DocumentType);
            
            info.AddValue("Encoding", this.Encoding.CodePage);
            
            info.AddValue("InsertMode", this.TextBox.InsertMode);
            
            info.AddValue("LineCount", this.TextBox.LayoutLineCollection.Count);
            for (int i = 0; i < this.TextBox.LayoutLineCollection.Count; i++)
                info.AddValue(i.ToString(), this.TextBox.LayoutLineCollection[i]);
        }
    }
}
