﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Globalization;

using DotNetTextStore.UnmanagedAPI.TSF;
using DotNetTextStore.UnmanagedAPI.TSF.TextStore;
using DotNetTextStore.UnmanagedAPI.WinDef;
using DotNetTextStore.UnmanagedAPI.WinError;

namespace DotNetTextStore 
{


    //========================================================================================

    /// <summary>
    ///  テキストストアの実装を担うクラス。
    /// <pre>
    ///   各ドキュメント毎に実装すべき部分をイベントとして切り離して、テキストストアの実装を楽に
    ///   させる。
    /// </pre>
    /// <pre>
    ///   使用側としては各イベントハンドラの実装、フォーカスを取得した時に SetFocus() メソッドの
    ///   呼び出し、選択やテキストが変更された時に NotifySelectionChanged() メソッドや
    ///   NotifyTextChange() メソッドの呼び出しを行う必要がある。
    /// </pre>
    public sealed class TextStore :  TextStoreBase,IDisposable, ITextStoreACP, ITfContextOwnerCompositionSink
    {
        ITfCategoryMgr _categoryMgr;
        ITfDisplayAttributeMgr _displayAttributeMgr;
        ITfThreadMgr _threadMgr;
        ITfDocumentMgr _documentMgr;
        ITfContext _context;
        ITfFunctionProvider _functionProvider;
        uint _editCookie = 0;
        ITfFnReconversion _reconversion;

        public delegate IntPtr GetHWndHandler();
        public event GetHWndHandler GetHWnd;

        #region "生成と破棄"
        /// <summary>コンストラクタ</summary>
        public TextStore()
        {
#if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", DebugOut.GetCaller()) )
#endif
            {
                try
                {
                    // スレッドマネージャ－の生成
                    CreateThreadMgr();
                    // カテゴリマネージャーの生成
                    CreateCategoryMgr();
                    // 表示属性マネージャーの生成
                    CreateDisplayAttributeMgr();

                    // ドキュメントマネージャーの生成
                    _threadMgr.CreateDocumentMgr(out _documentMgr);

                    // スレッドマネージャのアクティブ化
                    int clientId = 0;
                    _threadMgr.Activate(out clientId);

                    // コンテキストの生成
                    _documentMgr.CreateContext(clientId, 0, this, out _context, out _editCookie);

                    // コンテキストの push
                    _documentMgr.Push(_context);

                    // ファンクションプロバイダーを取得する。
                    Guid guid = TfDeclarations.GUID_SYSTEM_FUNCTIONPROVIDER;
                    _threadMgr.GetFunctionProvider(ref guid, out _functionProvider);

                    // ITfReconversion オブジェクトを取得する。
                    var guidNull = new Guid();
                    var guidReconversion = new Guid("4cea93c0-0a58-11d3-8df0-00105a2799b5");    //ITfFnReconversionの定義から
                    object reconversion = null;
                    _functionProvider.GetFunction(
                        ref guidNull,
                        ref guidReconversion,
                        out reconversion
                    );
                    _reconversion = reconversion as ITfFnReconversion;

                    // MODEBIAS の初期化
                    uint guidAtom = 0;
                    Guid guidModebiasNone = TfDeclarations.GUID_MODEBIAS_NONE;
                    _categoryMgr.RegisterGUID(ref guidModebiasNone, out guidAtom);
                    _attributeInfo[0].attrID = TfDeclarations.GUID_PROP_MODEBIAS;
                    _attributeInfo[0].flags = AttributeInfoFlags.None;
                    _attributeInfo[0].currentValue.vt = (short)VarEnum.VT_EMPTY;
                    _attributeInfo[0].defaultValue.vt = (short)VarEnum.VT_I4;
                    _attributeInfo[0].defaultValue.data1 = (IntPtr)guidAtom;
                }
                catch (Exception exception)
                {
                    Debug.WriteLine(exception.Message);
                    Dispose();
                }
            }
        }

        /// <summary>
        /// 後処理
        /// </summary>
        public void Dispose()
        {
#if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", DebugOut.GetCaller()) )
#endif
            {
                // ITfReconversion の解放
                ReleaseComObject("_reconversion", ref _reconversion);
                // ファンクションプロバイダの解放
                ReleaseComObject("_functionProvider", ref _functionProvider);
                // コンテキストの解放
                ReleaseComObject("_context", ref _context);
                // ドキュメントマネージャの解放
                DestroyDocumentMgr();
                // 表示属性マネージャーの解放
                DestroyDisplayAttributeMgr();
                // カテゴリマネージャーの解放
                DestroyCategoryMgr();
                // スレッドマネージャーの解放
                DestroyThreadMgr();

                GC.SuppressFinalize(this);
            }
        }
        #endregion

        public override void SetFocus()
        {
            if (_threadMgr != null)
                _threadMgr.SetFocus(_documentMgr);
        }

        #region ITextStoreACPの実装
        public void GetWnd(int i_viewCookie, out IntPtr o_hwnd)
        {
#if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", DebugOut.GetCaller()) )
#endif
            {
                if (GetHWnd != null)
                    o_hwnd = GetHWnd();
                else
                    o_hwnd = IntPtr.Zero;
            }
        }

        public void RequestSupportedAttrs(
            AttributeFlags i_flags,
            int i_length,
            Guid[] i_filterAttributes
        )
        {
#if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", DebugOut.GetCaller()) )
#endif
            {
                for (int i = 0; i < i_length; i++)
                {
                    if (i_filterAttributes[i].Equals(_attributeInfo[0].attrID))
                    {
                        _attributeInfo[0].flags = AttributeInfoFlags.Requested;
                        if ((i_flags & AttributeFlags.TS_ATTR_FIND_WANT_VALUE) != 0)
                        {
                            _attributeInfo[0].flags |= AttributeInfoFlags.Default;
                        }
                        else
                        {
                            _attributeInfo[0].currentValue = _attributeInfo[0].defaultValue;
                        }
                    }
                }
            }
        }

        public void RequestAttrsAtPosition(
            int i_position,
            int i_length,
            Guid[] i_filterAttributes,
            AttributeFlags i_flags
        )
        {
#if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", DebugOut.GetCaller()) )
#endif
            {
                for (int i = 0; i < i_length; i++)
                {
                    if (i_filterAttributes[i].Equals(_attributeInfo[0].attrID))
                    {
                        _attributeInfo[0].flags = AttributeInfoFlags.Requested;
                        if ((i_flags & AttributeFlags.TS_ATTR_FIND_WANT_VALUE) != 0)
                        {
                            _attributeInfo[0].flags |= AttributeInfoFlags.Default;
                        }
                        else
                        {
                            _attributeInfo[0].currentValue = _attributeInfo[0].defaultValue;
                        }
                    }
                }
            }
        }

        public void RequestAttrsTransitioningAtPosition(
            int i_position,
            int i_length,
            Guid[] i_filterAttributes,
            AttributeFlags i_flags
        )
        {
#if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", DebugOut.GetCaller()))
#endif
            {
                // 何もしない。
            }
        }

        public void FindNextAttrTransition(
            int i_start,
            int i_halt,
            int i_length,
            Guid[] i_filterAttributes,
            AttributeFlags i_flags,
            out int o_nextIndex,
            out bool o_found,
            out int o_foundOffset
        )
        {
#if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", DebugOut.GetCaller()) )
#endif
            {
                throw new NotImplementedException();
            }
        }

        public void RetrieveRequestedAttrs(
            int i_length,
            TS_ATTRVAL[] o_attributeVals,
            out int o_fetchedLength
        )
        {
#if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", DebugOut.GetCaller()) )
#endif
            {
                o_fetchedLength = 0;
                for (int i = 0; i < _attributeInfo.Length && o_fetchedLength < i_length; i++)
                {
                    if ((_attributeInfo[i].flags & AttributeInfoFlags.Requested) != 0)
                    {
                        o_attributeVals[o_fetchedLength].overlappedId = 0;
                        o_attributeVals[o_fetchedLength].attributeId = _attributeInfo[i].attrID;
                        o_attributeVals[o_fetchedLength].val = _attributeInfo[i].currentValue;

                        if ((_attributeInfo[i].flags & AttributeInfoFlags.Default) != 0)
                            _attributeInfo[i].currentValue = _attributeInfo[i].defaultValue;

                        o_fetchedLength++;
                        _attributeInfo[i].flags = AttributeInfoFlags.None;
                    }
                }
            }
        }
        #endregion

        #region 表示属性
        /// <summary>表示属性の取得</summary>
        public IEnumerable<TextDisplayAttribute> EnumAttributes(int start, int end)
        {
            ITfRangeACP allRange;
            _services.CreateRange(start, end, out allRange);

            foreach (TextDisplayAttribute attr in this.EnumAttributes((ITfRange)allRange))
                yield return attr;

            ReleaseComObject("allRange", ref allRange);
        }

        IEnumerable<TextDisplayAttribute> EnumAttributes(ITfRange range)
        {
#if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", DebugOut.GetCaller()))
#endif
            {
                ITfProperty property = null;    // プロパティインターフェイス
                IEnumTfRanges enumRanges;         // 範囲の列挙子
                Guid guidPropAttribute = TfDeclarations.GUID_PROP_ATTRIBUTE;

                if (_context == null || _services == null)
                    yield break;

                // GUID_PROP_ATTRIBUTE プロパティを取得。
                _context.GetProperty(ref guidPropAttribute, out property);
                if (property == null)
                    yield break;

                // 全範囲の中で表示属性プロパティをもつ範囲を列挙する。
                property.EnumRanges((int)_editCookie, out enumRanges, range);

                ITfRange[] ranges = new ITfRange[1];
                int fetchedLength = 0;
                while (HRESULT.Succeeded(enumRanges.Next(1, ranges, out fetchedLength))
                    && fetchedLength > 0)
                {
                    // ItfRange から ItfRangeACP を取得。
                    ITfRangeACP rangeACP = ranges[0] as ITfRangeACP;
                    if (rangeACP == null)
                        continue;   // 普通はあり得ない

                    // 範囲の開始位置と文字数を取得。
                    int start, count;
                    rangeACP.GetExtent(out start, out count);

                    // VARIANT 値としてプロパティ値を取得。VT_I4 の GUID ATOM がとれる。
                    VARIANT value = new VARIANT();
                    property.GetValue((int)_editCookie, ranges[0], ref value);
                    if (value.vt == (short)VarEnum.VT_I4)
                    {
                        Guid guid, clsid;
                        ITfDisplayAttributeInfo info;
                        TF_DISPLAYATTRIBUTE attribute;

                        // GUID ATOM から GUID を取得する。
                        _categoryMgr.GetGUID((int)value.data1, out guid);
                        // その GUID から IDisplayAttributeInfo インターフェイスを取得。
                        _displayAttributeMgr.GetDisplayAttributeInfo(
                            ref guid,
                            out info,
                            out clsid
                        );
                        // さらに IDisplayAttributeInfo インターフェイスから表示属性を取得する。
                        info.GetAttributeInfo(out attribute);
                        ReleaseComObject("info", ref info);

                        yield return new TextDisplayAttribute
                        {
                            startIndex = start,
                            endIndex = start + count,
                            attribute = attribute
                        };

#if TSF_DEBUG_OUTPUT
                            Debug.WriteLine(
                                "*******:::: DisplayAttribute: {0} ~ {1} :::::: *********",
                                start, start + count
                            );
                            Debug.WriteLine(attribute.bAttr);
                            Debug.WriteLine(
                                "LineColorType: {0}, {1}",
                                attribute.crLine.type, attribute.crLine.indexOrColorRef
                            );
                            Debug.WriteLine(
                                "TextColorType: {0}, {1}",
                                attribute.crText.type, attribute.crText.indexOrColorRef
                            );
                            Debug.WriteLine(
                                "BackColorType: {0}, {1}",
                                attribute.crBk.type, attribute.crBk.indexOrColorRef
                            );
                            Debug.WriteLine(
                                "Bold, Style  : {0}, {1}",
                                attribute.fBoldLine, attribute.lsStyle
                            );
#endif
                    }

                    ReleaseComObject("rangeACP", ref rangeACP);
                }

                ReleaseComObject("ranges[0]", ref ranges[0]);
                ReleaseComObject("enumRanges", ref enumRanges);
                ReleaseComObject("property", ref property);
            }
        }
        #endregion

        //========================================================================================
        //  TSF 関連のラッパメソッド
        //========================================================================================
        #region TSF 関連のラッパメソッド
        /// <summary>
        /// スレッドマネージャの生成
        /// </summary>
        /// 
        /// <exception cref="COMException">
        /// スレッドマネージャーの生成に失敗した場合。
        /// </exception>
        void CreateThreadMgr()
        {
            if( _threadMgr == null )
            {
                TextFrameworkFunctions.TF_GetThreadMgr(out _threadMgr);
                if( _threadMgr == null )
                {
                    Type clsid = Marshal.GetTypeFromCLSID(TfDeclarations.CLSID_TF_ThreadMgr);
                    _threadMgr = Activator.CreateInstance(clsid) as ITfThreadMgr;

                    if( _threadMgr == null )
                    {
                        const string message = "スレッドマネージャーの生成に失敗しました。";
                        Debug.WriteLine(message);
                        throw new COMException(message, HRESULT.E_NOTIMPL);
                    }
                }
            }
        }


        //====================================================================


        /// <summary>
        /// スレッドマネージャーの解放。
        /// </summary>
        void DestroyThreadMgr()
        {
            if( _threadMgr != null )
            {
                try { _threadMgr.Deactivate(); } catch(Exception) {}

                ReleaseComObject("_threadMgr", ref _threadMgr);
            }
        }


        //====================================================================


        /// <summary>
        /// カテゴリマネージャーの生成。
        /// </summary>
        /// 
        /// <exception cref="COMException">
        /// カテゴリマネージャーの生成に失敗した場合。
        /// </exception>
        void CreateCategoryMgr()
        {
            if( _categoryMgr == null )
            {
                var clsid = Marshal.GetTypeFromCLSID(TfDeclarations.CLSID_TF_CategoryMgr);
                _categoryMgr = Activator.CreateInstance(clsid) as ITfCategoryMgr;

                if( _categoryMgr == null )
                {
                    const string message = "カテゴリマネージャーの生成に失敗しました。";
                    Debug.WriteLine(message);
                    throw new COMException(message, HRESULT.E_NOTIMPL);
                }
            }
        }


        //====================================================================


        /// <summary>
        /// カテゴリマネージャーの解放。
        /// </summary>
        void DestroyCategoryMgr()
        {
            ReleaseComObject("_categoryMgr", ref _categoryMgr);
        }

        
        //====================================================================


        /// <summary>
        /// 表示属性マネージャーの生成。
        /// </summary>
        /// 
        /// <exception cref="COMException">
        /// 表示属性マネージャーの生成に失敗した場合。
        /// </exception>
        void CreateDisplayAttributeMgr()
        {
            if( _displayAttributeMgr == null )
            {
                var clsid = Marshal.GetTypeFromCLSID(TfDeclarations.CLSID_TF_DisplayAttributeMgr);
                _displayAttributeMgr = Activator.CreateInstance(clsid) as ITfDisplayAttributeMgr;

                if( _displayAttributeMgr == null )
                {
                    const string message = "表示属性マネージャーの生成に失敗しました。";
                    Debug.WriteLine(message);
                    throw new COMException(message, HRESULT.E_NOTIMPL);
                }
            }
        }


        //====================================================================


        /// <summary>
        /// 表示属性マネージャーの解放。
        /// </summary>
        void DestroyDisplayAttributeMgr()
        {
            ReleaseComObject("_displayAttributeMgr", ref _displayAttributeMgr);
        }

        
        //====================================================================


        /// <summary>
        /// ドキュメントマネージャーの解放
        /// </summary>
        void DestroyDocumentMgr()
        {
            if( _documentMgr != null )
            {
                try { _documentMgr.Pop(PopFlags.TF_POPF_ALL); } catch(Exception) {}

                ReleaseComObject("_documentMgr", ref _documentMgr);
            }
        }

        
        //====================================================================


        #endregion

    }
}
