﻿using System;
using System.Linq;
using System.Collections.Generic;

namespace FooEditEngine
{
    enum AdjustFlow
    {
        Row,
        Col,
        Both,
    }

    /// <summary>
    /// キャレットとドキュメントの表示を担当します。レイアウト関連もこちらで行います
    /// </summary>
    class View : IDisposable
    {
        const int SpiltCharCount = 1024;

        protected Document Document;

        Point _CaretLocation = new Point();
        Point2 _Src = new Point2();
        TextPoint _CaretPostion = new TextPoint();
        LineToIndexTable _LayoutLines;
        double MarginLeftAndRight;
        double _LongestWidth;
        long tickCount;
        Rectangle _Rect;
        bool _DrawLineNumber, _CaretBlink, _isLineBreak;
        int LineCountOnScreenWithInVisible;

        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <remarks>Graphicsプロパティをセットした後で必ずDocument.Clear(false)を呼び出してください。呼び出さない場合、例外が発生します</remarks>
        public View(Document doc, ITextRender r,int MarginLeft = 5)
        {
            this.Document = doc;
            this.Document.Update += new DocumentUpdateEventHandler(doc_Update);
            this._LayoutLines = new LineToIndexTable(this.Document,r);
            this._LayoutLines.SpilitString = new SpilitStringEventHandler(LayoutLines_SpilitStringByChar);
            this.render = r;
            this.render.ChangedRenderResource += new ChangedRenderResourceEventHandler(render_ChangedRenderResource);
            this.render.ChangedRightToLeft += render_ChangedRightToLeft;
            this._CaretLocation.X = this.MarginLeftAndRight = MarginLeft;
            this.CaretBlinkTime = 500;
            this.CaretWidthOnInsertMode = 1;
            this.SrcChanged += new EventHandler((s, e) => { });
            this.LineBreakChanged += new EventHandler((s, e) => { });
            this.PageBoundChanged += new EventHandler((s, e) => { });
        }

        void render_ChangedRightToLeft(object sender, EventArgs e)
        {
            this._Src.X = 0;
        }

        public event EventHandler SrcChanged;

        public event EventHandler LineBreakChanged;

        public event EventHandler PageBoundChanged;

        /// <summary>
        /// テキストレンダラ―
        /// </summary>
        public ITextRender render
        {
            get;
            set;
        }

        /// <summary>
        /// URLをハイパーリンクとして表示するなら真。そうでないなら偽
        /// </summary>
        public bool UrlMark
        {
            get { return this.LayoutLines.UrlMark; }
            set { this.LayoutLines.UrlMark = value; }
        }

        /// <summary>
        /// ラインマーカーを描くなら偽。そうでなければ真
        /// </summary>
        public bool HideLineMarker
        {
            get;
            set;
        }

        /// <summary>
        /// キャレットを描くなら偽。そうでなければ真
        /// </summary>
        public bool HideCaret
        {
            get;
            set;
        }

        /// <summary>
        /// 挿入モードなら真を返し、上書きモードなら偽を返す
        /// </summary>
        public bool InsertMode
        {
            get;
            set;
        }

        /// <summary>
        /// キャレットの点滅間隔
        /// </summary>
        public int CaretBlinkTime
        {
            get;
            set;
        }

        /// <summary>
        /// 挿入モード時のキャレットの幅
        /// </summary>
        public double CaretWidthOnInsertMode
        {
            get;
            set;
        }

        /// <summary>
        /// キャレットを点滅させるなら真。そうでないなら偽
        /// </summary>
        /// <remarks>キャレット点滅タイマーもリセットされます</remarks>
        public bool CaretBlink
        {
            get { return this._CaretBlink; }
            set
            {
                this._CaretBlink = value;
                if (value)
                    this.tickCount = DateTime.Now.Ticks + this.To100nsTime(this.CaretBlinkTime);
            }
        }

        /// <summary>
        /// 一ページの高さに収まる行数を返す
        /// </summary>
        public int LineCountOnScreen
        {
            get;
            protected set;
        }

        /// <summary>
        /// 折り返し時の右マージン
        /// </summary>
        public double LineBreakingMarginWidth
        {
            get { return this.PageBound.Width * 5 / 100; }
        }

        /// <summary>
        /// スクロール時に確保するマージン幅
        /// </summary>
        public double ScrollMarginWidth
        {
            get { return this.PageBound.Width * 20 / 100; }
        }

        /// <summary>
        /// 保持しているレイアウト行
        /// </summary>
        public LineToIndexTable LayoutLines
        {
            get { return this._LayoutLines; }
        }

        /// <summary>
        /// 行番号を表示するかどうか
        /// </summary>
        public bool DrawLineNumber
        {
            get { return this._DrawLineNumber; }
            set
            {
                this._DrawLineNumber = value;
                this.LayoutLines.ClearLayoutCache();
                CalculateClipRect();
            }
        }

        /// <summary>
        /// ページ全体を表す領域
        /// </summary>
        public Rectangle PageBound
        {
            get { return this._Rect; }
            set
            {
                if (value.Width < 0 || value.Height < 0)
                    throw new ArgumentOutOfRangeException("");
                this._Rect = value;
                CalculateClipRect();
                CalculateLineCountOnScreen();
                this.PageBoundChanged(this, null);
            }
        }

        /// <summary>
        /// Draw()の対象となる領域の左上を表す
        /// </summary>
        public Point2 Src
        {
            get { return this._Src; }
            set { this._Src = value; }
        }

        /// <summary>
        /// 最も長い行の幅
        /// </summary>
        public double LongestWidth
        {
            get { return this._LongestWidth; }
        }

        /// <summary>
        /// キャレットがある領域を示す
        /// </summary>
        public Point CaretLocation
        {
            get { return this._CaretLocation; }
        }

        /// <summary>
        /// レイアウト行のどこにキャレットがあるかを表す
        /// </summary>
        public TextPoint CaretPostion
        {
            get { return this._CaretPostion; }
        }

        /// <summary>
        /// 桁折り処理を行うかどうか
        /// </summary>
        /// <remarks>
        /// 変更した場合、呼び出し側で再描写を行う必要があります
        /// </remarks>
        public bool isLineBreak
        {
            get
            {
                return this._isLineBreak;
            }
            set
            {
                this._isLineBreak = value;
                if (value)
                    this._LayoutLines.SpilitString = new SpilitStringEventHandler(LayoutLines_SpilitStringByPixelbase);
                else
                    this._LayoutLines.SpilitString = new SpilitStringEventHandler(LayoutLines_SpilitStringByChar);
                this.PerfomLayouts();
                this.LineBreakChanged(this, null);
            }
        }

        /// <summary>
        /// シンタックスハイライター
        /// </summary>
        public IHilighter Hilighter
        {
            get { return this.LayoutLines.Hilighter; }
            set { this.LayoutLines.Hilighter = value; }
        }

        /// <summary>
        /// タブの幅
        /// </summary>
        /// <remarks>変更した場合、呼び出し側で再描写する必要があります</remarks>
        public int TabStops
        {
            get { return this.render.TabWidthChar; }
            set { this.render.TabWidthChar = value; }
        }

        /// <summary>
        /// すべてのレイアウト行を破棄し、再度レイアウトをやり直す
        /// </summary>
        public virtual void PerfomLayouts()
        {
            this.doc_Update(this.Document,new DocumentUpdateEventArgs(UpdateType.Clear,-1,-1,-1));
            this.doc_Update(this.Document, new DocumentUpdateEventArgs(UpdateType.Replace, 0, 0, this.Document.Length));
        }

        /// <summary>
        /// ヒットテストを行う
        /// </summary>
        /// <param name="x">x座標</param>
        /// <param name="y">y座標</param>
        /// <returns>テキストエリア内にあれば真。そうでなければ偽</returns>
        public bool HitTextArea(double x, double y)
        {
            if (x >= this.render.ClipRect.X && x <= this.render.ClipRect.Right &&
                y >= this.render.ClipRect.Y && y <= this.render.ClipRect.Bottom)
                return true;
            else
                return false;
        }

        /// <summary>
        /// ヒットテストを行う
        /// </summary>
        /// <param name="x">x座標</param>
        /// <param name="row">行</param>
        /// <returns>ヒットした場合はFoldingDataオブジェクトが返され、そうでない場合はnullが返る</returns>
        public FoldingItem HitFoldingData(double x, int row)
        {
            if (x >= this.PageBound.X && x <= this.PageBound.X + this.render.FoldingWidth + this.MarginLeftAndRight)
            {
                LineToIndexTableData lineData = this.LayoutLines.GetData(row);
                FoldingItem foldingData = this.LayoutLines.FoldingCollection.Get(lineData.Index, lineData.Length);
                if (foldingData != null && foldingData.IsFirstLine(this.LayoutLines,row))
                    return foldingData;
            }
            return null;
        }

        /// <summary>
        /// Rectで指定された範囲にドキュメントを描く
        /// </summary>
        /// <param name="updateRect">描写する範囲</param>
        /// <remarks>キャレットを点滅させる場合、定期的のこのメソッドを呼び出してください</remarks>
        public virtual void Draw(Rectangle updateRect)
        {
            if (this.LayoutLines.Count == 0)
                return;

            if ((updateRect.Height < this.PageBound.Height ||
                updateRect.Width < this.PageBound.Width) && 
                this.render.IsVaildCache())
            {
                this.render.DrawCachedBitmap(updateRect);
            }
            else
            {
                Rectangle background = this.PageBound;
                this.render.FillBackground(background);

                if (this.HideLineMarker == false)
                    this.DrawLineMarker(this.CaretPostion.row);

                Point pos = new Point(this.PageBound.X, this.PageBound.Y);
                pos.X -= this.Src.X;
                double endposy = pos.Y + this.PageBound.Height;
                for (int i = this.Src.Row; pos.Y < endposy && i < this.LayoutLines.Count; i++)
                {
                    LineToIndexTableData lineData = this.LayoutLines.GetData(i);

                    FoldingItem foldingData = this.LayoutLines.FoldingCollection.Get(lineData.Index,lineData.Length);

                    if (foldingData != null)
                    {
                        if ((foldingData.Parent == null || foldingData.Parent.Expand) && foldingData.IsFirstLine(this.LayoutLines, i))
                            this.render.DrawFoldingMark(foldingData.Expand, this.PageBound.X + this.MarginLeftAndRight, pos.Y);
                        if (foldingData.IsHidden(lineData.Index))
                            continue;
                    }

                    var selectRange = from s in this.Document.Selections.Get(lineData.Index, lineData.Length)
                                      let n = Util.ConvertAbsIndexToRelIndex(s, lineData.Index,lineData.Length)
                                      select n;

                    this.render.DrawOneLine(this.LayoutLines, i, pos.X + this.render.ClipRect.X, pos.Y, selectRange);

                    if (this.DrawLineNumber)
                        this.render.DrawLineNumber(i + 1, this.PageBound.X + this.render.FoldingWidth, pos.Y); //行番号は１からはじまる

                    pos.Y += this.LayoutLines.GetLayout(i).Height;
                }

                this.render.CacheContent();
            }

            if (this.HideCaret == false)
                this.DrawCaret();
        }

        void DrawCaret()
        {
            long diff = DateTime.Now.Ticks - this.tickCount;
            long blinkTime = this.To100nsTime(this.CaretBlinkTime);

            if (this._CaretBlink && diff % blinkTime >= blinkTime / 2)
                return;

            Rectangle CaretRect = new Rectangle();

            int row = this.CaretPostion.row;
            double lineHeight = this.LayoutLines.GetLayout(row).Height;
            double charWidth = this.LayoutLines.GetLayout(row).GetWidthFromIndex(this.CaretPostion.col);
            bool transparent = false;

            if (this.InsertMode || charWidth == 0)
            {
                CaretRect.Size = new Size(CaretWidthOnInsertMode, lineHeight);
                CaretRect.Location = new Point(this.CaretLocation.X, this.CaretLocation.Y + this.render.ClipRect.Y);
            }
            else
            {
                double height = lineHeight / 3;
                CaretRect.Size = new Size(charWidth, height);
                CaretRect.Location = new Point(this.CaretLocation.X, this.CaretLocation.Y + lineHeight - height + this.render.ClipRect.Y);
                transparent = true;
            }
            render.DrawCaret(CaretRect,transparent);
        }

        long To100nsTime(int ms)
        {
            return ms * 10000;
        }

        public virtual void DrawLineMarker(int row)
        {
            Point p = this.CaretLocation;
            double height = this.LayoutLines.GetData(this.CaretPostion.row).Layout.Height;
            double width = this.render.ClipRect.Width;
            this.render.DrawLineMarker(new Rectangle(this.render.ClipRect.X,this.CaretLocation.Y,width,height));
        }

        /// <summary>
        /// 指定した座標の一番近くにあるTextPointを取得する
        /// </summary>
        /// <param name="p">(PageBound.X,PageBound.Y)を左上とする相対位置</param>
        /// <returns>レイアウトラインを指し示すTextPoint</returns>
        public virtual TextPoint GetTextPointFromPostion(Point p)
        {
            TextPoint tp = new TextPoint();

            if (this.LayoutLines.Count == 0)
                return tp;

            double y = 0;
            tp.row = this.LayoutLines.Count - 1;
            for (int i = this._Src.Row; i < this.LayoutLines.Count; i++)
            {
                double height = this.LayoutLines.GetData(i).Layout.Height;
                
                LineToIndexTableData lineData = this.LayoutLines.GetData(i);
                FoldingItem foldingData = this.LayoutLines.FoldingCollection.Get(lineData.Index,lineData.Length);

                if (foldingData != null && foldingData.IsHidden(lineData.Index))
                    continue;

                if (y + height > p.Y)
                {
                    tp.row = i;
                    break;
                }
                y += height;
            }

            if (p.X < this.render.ClipRect.X)
                return tp;

            tp.col = GetIndexFromX(tp.row, p.X);

            int lineLength = this.LayoutLines[tp.row].Length;
            if (tp.col > lineLength)
                tp.col = lineLength;

            return tp;
        }

        /// <summary>
        /// X座標に対応するインデックスを取得する
        /// </summary>
        /// <param name="row">対象となる行</param>
        /// <param name="x">PageBound.Xからの相対位置</param>
        /// <returns></returns>
        public virtual int GetIndexFromX(int row, double x)
        {
            x -= this.render.ClipRect.X - this.PageBound.X;
            LineToIndexTableData data = this.LayoutLines.GetData(row);
            if (data.Length == 0)
                return 0;
            int index = data.Layout.GetIndexFromX(this.Src.X + x);
            return index;
        }

        /// <summary>
        /// インデックスに対応するＸ座標を得る
        /// </summary>
        /// <param name="row">対象となる行</param>
        /// <param name="index">インデックス</param>
        /// <returns>PageBound.Xからの相対位置を返す</returns>
        public virtual double GetXFromIndex(int row, int index)
        {
            double x = this.LayoutLines.GetData(row).Layout.GetXFromIndex(index);
            return x - Src.X + this.render.ClipRect.X;
        }

        /// <summary>
        /// TextPointに対応する座標を得る
        /// </summary>
        /// <param name="tp">レイアウトライン上の位置</param>
        /// <returns>(PageBound.X,PageBound.Y)を左上とする相対位置</returns>
        public virtual Point GetPostionFromTextPoint(TextPoint tp)
        {
            Point p = new Point();
            for (int i = this.Src.Row; i < tp.row; i++)
            {
                LineToIndexTableData lineData = this.LayoutLines.GetData(i);
                FoldingItem foldingData = this.LayoutLines.FoldingCollection.Get(lineData.Index,lineData.Length);
                if (foldingData != null && foldingData.IsHidden(lineData.Index))
                    continue;
                p.Y += this.LayoutLines.GetData(i).Layout.Height;
            }
            p.X = this.LayoutLines.GetData(tp.row).Layout.GetXFromIndex(tp.col);
            p.Y += this.render.ClipRect.Y;
            return p;
        }

        /// <summary>
        /// キャレットを指定した位置に移動させる
        /// </summary>
        /// <param name="row"></param>
        /// <param name="col"></param>
        /// <param name="autoExpand">折り畳みを展開するなら真</param>
        public virtual void JumpCaret(int row, int col, bool autoExpand = true)
        {
            if (autoExpand)
            {
                LineToIndexTableData lineData = this.LayoutLines.GetData(row);
                FoldingItem foldingData = this.LayoutLines.FoldingCollection.Get(lineData.Index,lineData.Length);
                if(foldingData != null && !foldingData.IsFirstLine(this.LayoutLines,row))
                    this.LayoutLines.FoldingCollection.Expand(foldingData);
            }

            this._CaretPostion.row = row;
            this._CaretPostion.col = col;
        }

        /// <summary>
        /// キャレットがあるところまでスクロールする
        /// </summary>
        /// <return>再描写する必要があるなら真を返す</return>
        /// <remarks>Document.Update(type == UpdateType.Clear)イベント時に呼び出した場合、例外が発生します</remarks>
        public virtual bool AdjustCaretAndSrc(AdjustFlow flow = AdjustFlow.Both)
        {
            if (this.PageBound.Width == 0 || this.PageBound.Height == 0)
            {
                this.SetCaretPostion(this.MarginLeftAndRight + this.render.FoldingWidth, 0);
                return false;
            }

            bool result = false;
            TextPoint tp = this.CaretPostion;
            double x = this.CaretLocation.X;
            double y = this.CaretLocation.Y;

            if (flow == AdjustFlow.Col || flow == AdjustFlow.Both)
            {
                x = this.LayoutLines.GetData(tp.row).Layout.GetXFromIndex(tp.col);

                double left = this.Src.X;
                double right = this.Src.X + this.render.ClipRect.Width;

                if (x >= left && x <= right)    //xは表示領域にないにある
                {
                    x -= left;
                }
                else if (x > right) //xは表示領域の右側にある
                {
                    this._Src.X = x - this.render.ClipRect.Width + this.ScrollMarginWidth;
                    if (this.render.RightToLeft && this._Src.X > 0)
                    {
                        System.Diagnostics.Debug.Assert(x > 0);
                        this._Src.X = 0;
                    }
                    else
                    {
                        x = this.render.ClipRect.Width - this.ScrollMarginWidth;
                    }
                    result = true;
                }
                else if (x < left)    //xは表示領域の左側にある
                {
                    this._Src.X = x - this.ScrollMarginWidth;
                    if (!this.render.RightToLeft && this._Src.X < this.render.ClipRect.X)
                    {
                        this._Src.X = 0;
                        x =0;
                    }
                    else
                    {
                        x = this.ScrollMarginWidth;
                    }
                    result = true;
                }
                x += this.render.ClipRect.X - this.PageBound.X;
            }

            if (flow == AdjustFlow.Row || flow == AdjustFlow.Both)
            {
                CalculateLineCountOnScreen();
                int caretRow = 0;
                int lineCount = this.LineCountOnScreenWithInVisible;
                if (tp.row >= this.Src.Row && tp.row < this.Src.Row + lineCount)
                    caretRow = tp.row - this.Src.Row;
                else if (tp.row >= this.Src.Row + lineCount)
                {
                    if (this.HasCollapsedFolding(tp.row,lineCount))
                        this._Src.Row = this.AdjustRow(tp.row - lineCount,true);
                    else
                        this._Src.Row = tp.row - this.LineCountOnScreen;
                    caretRow = tp.row - this._Src.Row;
                    result = true;
                }
                else if (tp.row < this.Src.Row)
                {
                    this._Src.Row = tp.row;
                    result = true;
                }

                y = 0;

                if (caretRow > 0)
                {
                    for (int i = 0; i < caretRow; i++)
                    {
                        LineToIndexTableData lineData = this.LayoutLines.GetData(this.Src.Row + i);

                        FoldingItem foldingData = this.LayoutLines.FoldingCollection.Get(lineData.Index, lineData.Length);

                        if (foldingData != null && foldingData.IsHidden(lineData.Index))
                            continue;

                        y += lineData.Layout.Height;
                    }
                }
            }

            this.SetCaretPostion(x, y);

            if(result)
                this.SrcChanged(this, null);

            return result;
        }

        bool HasCollapsedFolding(int row, int count)
        {
            if (this.LayoutLines.FoldingStrategy == null)
                return false;
            int end = row - count + 1;
            for(; row >= end; row--)
            {
            LineToIndexTableData lineData = this.LayoutLines.GetData(row);
            FoldingItem foldingData = this.LayoutLines.FoldingCollection.GetFarestHiddenFoldingData(lineData.Index, lineData.Length);
            if (foldingData != null && !foldingData.Expand)
                return true;
            }
            return false;
        }

        /// <summary>
        /// レイアウト行をテキストポイントからインデックスに変換する
        /// </summary>
        /// <param name="tp">テキストポイント表す</param>
        /// <returns>インデックスを返す</returns>
        public virtual int GetIndexFromLayoutLine(TextPoint tp)
        {
            return this.LayoutLines.GetIndexFromTextPoint(tp);
        }

        /// <summary>
        /// インデックスからレイアウト行を指し示すテキストポイントに変換する
        /// </summary>
        /// <param name="index">インデックスを表す</param>
        /// <returns>テキストポイント返す</returns>
        public virtual TextPoint GetLayoutLineFromIndex(int index)
        {
            return this.LayoutLines.GetTextPointFromIndex(index);
        }

        /// <summary>
        /// 指定した座標までスクロールする
        /// </summary>
        /// <param name="x"></param>
        /// <param name="row"></param>
        /// <returns>trueの場合、スクロールに失敗したことを表す</returns>
        public virtual bool TryScroll(double x, int row)
        {
            if (row < 0)
                return true;
            if (row > this.LayoutLines.Count - 1)
                return true;
            this._Src.X = x;
            this._Src.Row = row;
            this.HideCaret = true;
            CalculateLineCountOnScreen();
            this.SrcChanged(this, null);
            return false;
        }

        /// <summary>
        /// 指定した座標までスクロールする
        /// </summary>
        /// <param name="x"></param>
        /// <param name="row"></param>
        /// <remarks>
        /// 範囲外の座標を指定した場合、範囲内に収まるように調整されます
        /// </remarks>
        public virtual void Scroll(double x, int row)
        {
            if (x < 0 || row < 0)
            {
                x = 0;
                row = 0;
            }
            int endRow = this.LayoutLines.Count - 1 - this.LineCountOnScreen;
            if (endRow < 0)
                endRow = 0;
            if (row > endRow)
            {
                row = endRow;
            }
            this._Src.X = x;
            this._Src.Row = row;
            CalculateLineCountOnScreen();
            this.SrcChanged(this, null);
        }

        public int AdjustRow(int row, bool isMoveNext)
        {
            if (this.LayoutLines.FoldingStrategy == null)
                return row;
            LineToIndexTableData lineData = this.LayoutLines.GetData(row);
            FoldingItem foldingData = this.LayoutLines.FoldingCollection.GetFarestHiddenFoldingData(lineData.Index, lineData.Length);
            if (foldingData != null && !foldingData.Expand)
            {
                if (foldingData.End == this.Document.Length)
                    return row;
                if (isMoveNext && lineData.Index > foldingData.Start)
                    row = this.LayoutLines.GetLineNumberFromIndex(foldingData.End) + 1;
                else
                    row = this.LayoutLines.GetLineNumberFromIndex(foldingData.Start);
            }
            return row;
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (disposing)
                this.Document.Update -= new DocumentUpdateEventHandler(this.doc_Update);    //これをしないと複数のビューを作成した時に妙なエラーが発生する
        }

        protected virtual void CalculateClipRect()
        {
            if (this.DrawLineNumber)
                this.render.ClipRect = new Rectangle(this.render.LineNemberWidth + this.render.FoldingWidth,
                    0,
                    this.PageBound.Width - this.render.LineNemberWidth - this.MarginLeftAndRight - this.render.FoldingWidth,
                    this.PageBound.Height);
            else
                this.render.ClipRect = new Rectangle(this.MarginLeftAndRight + this.render.FoldingWidth,
                    0,
                    this.PageBound.Width - this.MarginLeftAndRight * 2 - this.render.FoldingWidth,
                    this.PageBound.Height);
            if (this.render.ClipRect.Width < 0)
                this.render.ClipRect = this.PageBound;
        }

        protected virtual void CalculateLineCountOnScreen()
        {
            if (this.LayoutLines.Count == 0)
                return;

            double y = 0;
            int i = this.Src.Row;
            int visualCount = this.Src.Row;
            for (; true; i++)
            {
                LineToIndexTableData lineData = i < this.LayoutLines.Count ?
                    this.LayoutLines.GetData(i) :
                    this.LayoutLines.GetData(this.LayoutLines.Count - 1);

                FoldingItem foldingData = this.LayoutLines.FoldingCollection.Get(lineData.Index, lineData.Length);

                if (foldingData != null && foldingData.End < this.Document.Length - 1 && foldingData.IsHidden(lineData.Index))
                    continue;

                ITextLayout layout = lineData.Layout;

                double width = layout.Width;

                if (width > this._LongestWidth)
                    this._LongestWidth = width;

                double lineHeight = layout.Height;

                y += lineHeight;

                if (y >= this.PageBound.Height)
                    break;
                visualCount++;
            }
            this.LineCountOnScreen = Math.Max(visualCount - this.Src.Row - 1, 0);
            this.LineCountOnScreenWithInVisible = Math.Max(i - this.Src.Row - 1, 0);
        }

        void render_ChangedRenderResource(object sender, EventArgs e)
        {
            this.LayoutLines.ClearLayoutCache();
            this.CalculateClipRect();
            this.CalculateLineCountOnScreen();
        }

        protected void SetCaretPostion(double x, double y)
        {
            this._CaretLocation = new Point(x, y);
            this.HideCaret = false;
        }

        void doc_Update(object sender, DocumentUpdateEventArgs e)
        {
            switch (e.type)
            {
                case UpdateType.Replace:
                    this._LayoutLines.UpdateAsReplace(e.startIndex, e.removeLength, e.insertLength);
                    this.CalculateLineCountOnScreen();
                    break;
                case UpdateType.Clear:
                    this.LayoutLines.Clear();
                    this._LongestWidth = 0;
                    break;
            }
        }

        IList<LineToIndexTableData> LayoutLines_SpilitStringByPixelbase(object sender, SpilitStringEventArgs e)
        {
            double WrapWidth;
            WrapWidth = this.render.ClipRect.Width - LineBreakingMarginWidth;  //余白を残さないと欠ける

            if (WrapWidth < 0 && this.isLineBreak)
                throw new InvalidOperationException();

            int startIndex = e.index;
            int endIndex = e.index + e.length - 1;

            return this.render.BreakLine(e.buffer, startIndex, endIndex, WrapWidth);
        }

        IList<LineToIndexTableData> LayoutLines_SpilitStringByChar(object sender, SpilitStringEventArgs e)
        {
            int startIndex = e.index;
            int endIndex = e.index + e.length - 1;
            List<LineToIndexTableData> output = new List<LineToIndexTableData>();

            foreach (string str in e.buffer.GetLines(startIndex, endIndex, SpiltCharCount))
            {
                char c = str.Last();
                bool hasNewLine = c == Document.NewLine;
                output.Add(new LineToIndexTableData(startIndex, str.Length, hasNewLine, null));
                startIndex += str.Length;
            }

            if(output.Count > 0)
                output.Last().LineEnd = true;

            return output;
        }

    }
}
