﻿using System;
using System.Collections.Generic;

namespace FooEditEngine
{
    /// <summary>
    /// 選択領域を表すクラス
    /// </summary>
    struct Selection : IRange,IEqualityComparer<Selection>
    {
        #region IMarker メンバー

        public int start
        {
            get;
            set;
        }

        public int length
        {
            get;
            set;
        }

        public HilightType hilight
        {
            get;
            set;
        }

        #endregion

        public static Selection Create(int start, int length)
        {
            return new Selection { start = start, length = length, hilight = HilightType.Select };
        }

        public bool Equals(Selection x, Selection y)
        {
            return x.length == y.length && x.start == y.start;
        }

        public int GetHashCode(Selection obj)
        {
            return this.start ^ this.length;
        }
    }

    /// <summary>
    /// 選択範囲が更新されたことを通知するデリゲート
    /// </summary>
    /// <param name="sender">送信元クラス</param>
    /// <param name="e">イベントデータ</param>
    public delegate void SelectChangeEventHandler(object sender,EventArgs e);

    /// <summary>
    /// 選択範囲を格納するコレクションを表します
    /// </summary>
    class SelectCollection : IEnumerable<Selection>
    {
        RangeCollection<Selection> collection;

        /// <summary>
        /// コンストラクター
        /// </summary>
        public SelectCollection()
            : this(null)
        {
        }

        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="selections">コレクションを表します</param>
        public SelectCollection(IEnumerable<Selection> selections)
        {
            if (selections != null)
                collection = new RangeCollection<Selection>(selections);
            else
                collection = new RangeCollection<Selection>();
            this.SelectChange += new SelectChangeEventHandler((s,e)=>{});
        }

        /// <summary>
        /// 選択領域が更新されたことを通知します
        /// </summary>
        public event SelectChangeEventHandler SelectChange;

        /// <summary>
        /// インデクサー
        /// </summary>
        /// <param name="i">0から始まるインデックス</param>
        /// <returns>選択領域を返します</returns>
        public Selection this[int i]
        {
            get
            {
                return this.collection[i];
            }
        }

        /// <summary>
        /// 追加します
        /// </summary>
        /// <param name="sel">選択領域</param>
        public void Add(Selection sel)
        {
            this.collection.Add(sel);
            this.SelectChange(this, null);
        }

        /// <summary>
        /// すべて削除します
        /// </summary>
        public void Clear()
        {
            this.collection.Clear();
            this.SelectChange(this, null);
        }

        /// <summary>
        /// 要素を取得します
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <param name="length">長さ</param>
        /// <returns>要素を表すイテレーター</returns>
        public IEnumerable<Selection> Get(int index, int length)
        {
            return this.collection.Get(index, length);
        }

        /// <summary>
        /// 格納されている選択領域の数を返します
        /// </summary>
        public int Count
        {
            get
            {
                return this.collection.Count;
            }
        }

        #region IEnumerable<Selection> メンバー

        /// <summary>
        /// 列挙子を返します
        /// </summary>
        /// <returns>IEnumeratorオブジェクトを返す</returns>
        public IEnumerator<Selection> GetEnumerator()
        {
            for (int i = 0; i < this.collection.Count; i++)
                yield return this.collection[i];
        }

        #endregion

        #region IEnumerable メンバー

        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            throw new NotImplementedException();
        }

        #endregion
    }
}
