﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using System.Runtime.InteropServices;
using DWriteWarpper;
using FooEditEngine.Windows.Properties;

namespace FooEditEngine.Windows
{
    internal class D2DTextRender : ITextRender,IDisposable
    {
        const int LineNumberFiledLength = 6;    //行番号表示時の最大桁数

        int _TabWidthChar;

        DFactory Factory = new DFactory();
        DWindowRender render;
        FooTextBox TextBox;
        ColorTable _ColorTable = new ColorTable();
        ResourceManager<HilightType, DDrawingEffect> effects = new ResourceManager<HilightType, DDrawingEffect>();
        ResourceManager<TokenType, DColorBrush> brushs = new ResourceManager<TokenType, DColorBrush>();
        CacheManager<string, DTextLayout> cache = new CacheManager<string, DTextLayout>();
        DColorBrush InsertCaret,OverwriteCaret,LineMarker;
        DTextFormat Format;
        DBitmap bitmap;
        IntPtr currentMonitor;
        bool hasCache = false;
        
        public D2DTextRender(FooTextBox textbox)
        {
            InitRenderResource(Factory,textbox);

            textbox.SizeChanged += new EventHandler(textbox_SizeChanged);
            textbox.FontChanged += new EventHandler(textbox_FontChanged);
            textbox.HandleCreated += new EventHandler(textbox_HandleCreated);
            textbox.HandleDestroyed += new EventHandler(textbox_HandleDestroyed);
            textbox.Document.Markers.Updated += new EventHandler(Markers_Updated);
            textbox.Move += new EventHandler(textbox_Move);
        }

        [DllImport("user32.dll")]
        static extern IntPtr MonitorFromWindow(IntPtr hwnd, uint dwFlags);
        
        void textbox_Move(object sender, EventArgs e)
        {
            FooTextBox textbox = (FooTextBox)sender;
            IntPtr monitor = MonitorFromWindow(textbox.Handle, 0);
            if (monitor != this.currentMonitor)
            {
                this.currentMonitor = monitor;
                DRenderingParams param = this.Factory.CreateMonitorRenderingParams(monitor);
                this.render.SetTextRenderingParams(param);
                param.Dispose();
            }
        }

        Color2F ToColor2F(Color color)
        {
            return new Color2F(color.R, color.G, color.B, 255);
        }

        void InitRenderResource(DFactory Factory,FooTextBox textbox)
        {
            if (textbox == null || Factory == null)
                throw new ArgumentNullException();

            this.render = Factory.CreateWindowRender(textbox.Handle,true);
            DColorBrush defalut = this.render.CreateBrush(ToColor2F(this.TokenToColor.Fore));
            DColorBrush control = this.render.CreateBrush(ToColor2F(this.TokenToColor.Control));
            DColorBrush hilight = this.render.CreateBrush(ToColor2F(this.TokenToColor.Hilight));
            this.brushs.Add(TokenType.None, defalut);
            this.brushs.Add(TokenType.Control,control);
            this.brushs.Add(TokenType.Literal, this.render.CreateBrush(ToColor2F(this.TokenToColor.Literal)));
            this.brushs.Add(TokenType.Keyword1, this.render.CreateBrush(ToColor2F(this.TokenToColor.Keyword1)));
            this.brushs.Add(TokenType.Keyword2, this.render.CreateBrush(ToColor2F(this.TokenToColor.Keyword2)));
            this.brushs.Add(TokenType.Comment, this.render.CreateBrush(ToColor2F(this.TokenToColor.Comment)));
            this.render.InitTextRender(defalut, control);
            this.render.AddControlSymbol('\n',UInt32.Parse(Resources.NewLineSymbol));
            this.render.AddControlSymbol('\t', UInt32.Parse(Resources.TabSymbol));
            this.render.AddControlSymbol('　', UInt32.Parse(Resources.FullSpaceSymbol));
            this.InsertCaret = this.render.CreateBrush(ToColor2F(this.TokenToColor.InsertCaret));
            this.OverwriteCaret = this.render.CreateBrush(ToColor2F(this.TokenToColor.OverwriteCaret));
            this.LineMarker = this.render.CreateBrush(ToColor2F(this.TokenToColor.LineMarker));
            this.Format = Factory.CreateTextFormat(textbox.Font.Name, textbox.Font.Size);
            this.Format.WordWrapping = DWordWrapping.NoWrapping;
            this.TextBox = textbox;
            this.effects.Add(HilightType.Url, new DDrawingEffect(this.render.CreateBrush(ToColor2F(this.TokenToColor.URL)), null, this.Factory.CrateStrokeStyke(DLineStyle.Solid)));
            this.effects.Add(HilightType.Sold, new DDrawingEffect(this.brushs[TokenType.None], null, this.Factory.CrateStrokeStyke(DLineStyle.Solid)));
            this.effects.Add(HilightType.Dot, new DDrawingEffect(this.brushs[TokenType.None], null, this.Factory.CrateStrokeStyke(DLineStyle.Dot)));
            this.effects.Add(HilightType.Dash, new DDrawingEffect(this.brushs[TokenType.None], null, this.Factory.CrateStrokeStyke(DLineStyle.Dash)));
            this.effects.Add(HilightType.DashDot, new DDrawingEffect(this.brushs[TokenType.None], null, this.Factory.CrateStrokeStyke(DLineStyle.DashDot)));
            this.effects.Add(HilightType.DashDotDot, new DDrawingEffect(this.brushs[TokenType.None], null, this.Factory.CrateStrokeStyke(DLineStyle.DashDotDot)));
            this.effects.Add(HilightType.Select, new DDrawingEffect(this.brushs[TokenType.None], this.render.CreateBrush(ToColor2F(this.TokenToColor.Hilight)), null));
            this.TabWidthChar = this.TabWidthChar;
            this.bitmap = render.CreateBitmap(new SizeU((uint)this.ClipRect.Width, (uint)this.ClipRect.Height));
            this.hasCache = false;
        }

        void DestructRenderResource()
        {
            this.cache.Clear();
            this.brushs.Clear();
            this.effects.Clear();
            this.Format.Dispose();
            this.bitmap.Dispose();
            this.InsertCaret.Dispose();
            this.LineMarker.Dispose();
            if(this.render != null)
                this.render.Dispose();
            this.render = null;
        }

        void textbox_HandleDestroyed(object sender, EventArgs e)
        {
            DestructRenderResource();
        }

        void textbox_HandleCreated(object sender, EventArgs e)
        {
            FooTextBox textbox = (FooTextBox)sender;
            InitRenderResource(this.Factory, textbox);
        }

        void textbox_FontChanged(object sender, EventArgs e)
        {
            FooTextBox textbox = (FooTextBox)sender;
            DestructRenderResource();
            InitRenderResource(this.Factory, this.TextBox);
            if (this.ChangedRenderResource != null)
                ChangedRenderResource(this, null);
        }

        void textbox_SizeChanged(object sender, EventArgs e)
        {
            FooTextBox textbox = (FooTextBox)sender;
            if (render != null)
            {
                render.Resize((uint)textbox.Size.Width, (uint)textbox.Size.Height);
                this.bitmap.Dispose();
                this.bitmap = render.CreateBitmap(new SizeU((uint)this.ClipRect.Width, (uint)this.ClipRect.Height));
                this.hasCache = false;
            }
        }

        void Markers_Updated(object sender, EventArgs e)
        {
            this.PurgeLayoutCache();
        }

        public void BeginDraw()
        {
            render.BeginDraw();
        }

        public void EndDraw()
        {
            if (render.EndDraw())
            {
                DestructRenderResource();
                InitRenderResource(this.Factory, this.TextBox);
            }
        }

        public double LineNemberWidth
        {
            get
            {
                int length = Int32.Parse(Resources.LineNumberFiledLength);
                if(length < 0)
                    throw new InvalidOperationException("LineNumberFiledLengthは正の値を指定してください");
                length++;   //余白を確保する
                DTextLayout layout = Factory.CreateTextLayout(this.Format, "0", (float)this.ClipRect.Size.Width, (float)this.ClipRect.Size.Height);
                return (int)layout.metrics.widthIncludingTrailingWhitespace * length;
            }
        }

        public event ChangedRenderResourceEventHandler ChangedRenderResource;

        public int TabWidthChar
        {
            get { return this._TabWidthChar; }
            set
            {
                if (value == 0)
                    return;

                this._TabWidthChar = value;

                string str = Util.Generate('a', this._TabWidthChar);
                DTextLayout layout = Factory.CreateTextLayout(this.Format, str, (float)this.ClipRect.Size.Width, (float)this.ClipRect.Size.Height);
                this.Format.TabWidth = (int)layout.metrics.widthIncludingTrailingWhitespace;
                layout.Dispose();
            }
        }

        public ColorTable TokenToColor
        {
            get { return _ColorTable; }
            set
            {
                _ColorTable = value;
                DestructRenderResource();
                InitRenderResource(this.Factory, this.TextBox);
            }
        }

        public Rectangle ClipRect
        {
            get;
            set;
        }

        public void PurgeLayoutCache()
        {
            this.cache.Clear();
        }

        public void FillBackground(Rectangle rect)
        {
            this.render.Clear(ToColor2F(this.TokenToColor.Back));
        }

        public void DrawCachedBitmap(Rectangle rect)
        {
            render.DrawBitmap(this.bitmap, rect, 1.0f, rect);
        }

        public void CacheContent()
        {
            render.Flush();
            this.bitmap.CopyFromRenderTarget(new Point2U(), this.render, new RectU(0, 0, (uint)this.ClipRect.Width, (uint)this.ClipRect.Height));
            this.hasCache = true;
        }

        public bool IsVaildCache()
        {
            return this.hasCache;
        }

        public void DrawLineNumber(int lineNumber, double x, double y)
        {
            string lineNumberFormat = "{0," + Resources.LineNumberFiledLength +  "}";
            DTextLayout layout = Factory.CreateTextLayout(this.Format, string.Format(lineNumberFormat, lineNumber), (float)this.ClipRect.Size.Width, (float)this.ClipRect.Size.Height);
            render.DrawTextLayout(layout, (float)x, (float)y);
            layout.Dispose();
        }

        public double GetXFromIndex(string str, int index)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.Format, str, this.ClipRect.Size, out layout);
            float x, y;
            DHitTestMetrics metrics;
            metrics = layout.HitTestTextPostion(index, false, out x, out y);
            return Util.RoundUp(x);
        }

        public double GetWidthFromIndex(string str, int index)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.Format, str, this.ClipRect.Size,out layout);
            float x, y;
            DHitTestMetrics metrics;
            metrics = layout.HitTestTextPostion(index, false, out x, out y);
            float x2;
            layout.HitTestTextPostion(index, true, out x2, out y);

            return Util.RoundUp(x2 - x);
        }

        public double GetWidth(string str)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.Format, str, this.ClipRect.Size, out layout);
            return Util.RoundUp(layout.metrics.widthIncludingTrailingWhitespace);
        }

        public int AlignIndexToNearestCluster(string str, int index, AlignDirection flow)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.Format, str, this.ClipRect.Size, out layout);
            float x, y;
            DHitTestMetrics metrics;
            metrics = layout.HitTestTextPostion(index, false, out x, out y);

            if (flow == AlignDirection.Forward)
                return Util.RoundUp(metrics.textPosition + metrics.length);
            else if (flow == AlignDirection.Back) 
                return Util.RoundUp(metrics.textPosition);
            throw new ArgumentOutOfRangeException();
        }


        public int GetIndexFromX(string str, double x)
        {
            if (str == string.Empty || str == null || str[0] == Document.EndOfFile)
                return 0;

            DTextLayout layout;
            this.CreateTextLayout(this.Format, str, this.ClipRect.Size, out layout);
            bool isTrailing, isInsed;
            DHitTestMetrics metrics;
            metrics = layout.HitTextPoint((float)x, 0, out isTrailing, out isInsed);
            if (isTrailing)
                return Util.RoundUp(metrics.textPosition + metrics.length);
            else
                return Util.RoundUp(metrics.textPosition);
        }

        public double GetHeight(string str)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.Format, str, this.ClipRect.Size,out layout);
            int height = 0;
            DLineMetrics[] metrics = layout.LineMetrics;
            if(metrics != null && metrics.Length > 0)
                height = Util.RoundUp(metrics[0].height);
            return height;
        }

        public List<LineToIndexTableData> BreakLine(Document doc, int startIndex, int endIndex, double wrapwidth)
        {
            List<LineToIndexTableData> output = new List<LineToIndexTableData>();

            this.Format.WordWrapping = DWordWrapping.Wrapping;

            foreach (string str in Util.GetLines(doc, startIndex, endIndex))
            {
                DTextLayout layout = Factory.CreateTextLayout(this.Format, str, (float)wrapwidth, Int32.MaxValue);

                int i = startIndex;
                foreach (DLineMetrics metrics in layout.LineMetrics)
                {
                    if (metrics.length == 0 && metrics.newlineLength == 0)
                        continue;

                    bool lineend = false;
                    if (metrics.newlineLength > 0 || doc[i + (int)metrics.length - 1] == Document.EndOfFile)
                        lineend = true;

                    output.Add(new LineToIndexTableData(i, (int)metrics.length, lineend, null));
                    i += Util.RoundUp(metrics.length);
                }

                layout.Dispose();

                startIndex += str.Length;
            }

            this.Format.WordWrapping = DWordWrapping.NoWrapping;

            return output;
        }

        public void DrawLineMarker(Rectangle rect)
        {
            this.render.AntialiasMode = DAntialias.Alias;

            this.render.FillRectangle(rect, this.LineMarker);

            this.render.AntialiasMode = DAntialias.Antialias;
        }

        public void DrawCaret(Rectangle rect,bool transparent)
        {
            this.render.AntialiasMode = DAntialias.Alias;

            if (transparent == false)
                this.render.FillRectangle(rect, this.InsertCaret);
            else
                this.render.FillRectangle(rect, this.InsertCaret);

            this.render.AntialiasMode = DAntialias.Antialias;
        }

        public void DrawOneLine(LineToIndexTable lti, int row, double x, double y, IEnumerable<Selection> SelectRanges, IEnumerable<Marker> MarkerRanges)
        {
            string str = lti[row];

            if (str == string.Empty || str == null)
                return;

            DTextLayout layout;
            this.CreateTextLayout(this.Format, str, this.ClipRect.Size, out layout);
            if ((bool)layout.Tag == false)
            {
                LineToIndexTableData lineData = lti.GetData(row);
                if (lineData.syntax != null)
                {
                    foreach (SyntaxInfo s in lineData.syntax)
                    {
                            layout.SetTextEffect(s.index, s.length, this.brushs[s.type]);
                    }
                }

                foreach (Marker m in MarkerRanges)
                {
                    if (m.start == -1 || m.length == 0)
                        continue;
                    layout.SetTextEffect(m.start, m.length, this.effects[m.hilight]);
                    if (m.hilight != HilightType.Select && m.hilight != HilightType.None)
                        layout.SetUnderline(m.start, m.length, true);
                }
                layout.Tag = true;
            }

            foreach (Selection sel in SelectRanges)
            {
                if (sel.length == 0 || sel.start == -1)
                    continue;

                this.FillSelectArea(layout, sel.start, sel.length, (float)x , (float)y);
            }

            this.render.PushAxisAlignedClip(this.ClipRect, DAntialias.Alias);

            render.DrawTextLayout(layout, (float)x, (float)y);

            this.render.PopAxisAlignedClip();
        }

        public void Dispose()
        {
            DestructRenderResource();
            Factory.Dispose();
        }

        void FillSelectArea(DTextLayout layout, int start, int length,float x,float y)
        {
            this.render.AntialiasMode = DAntialias.Alias;

            float height = layout.LineMetrics[0].height;
            for (int i = start; i < start + length;)
            {
                float x1,x2,dummy;
                DHitTestMetrics metrics = layout.HitTestTextPostion(i, false, out x1, out dummy);
                layout.HitTestTextPostion(i, true, out x2, out dummy);
                this.render.FillRectangle(new RectF(x + x1, y, x + x2, y + height), this.effects[HilightType.Select].Back);
                int delta = (int)metrics.length;
                if(delta == 0)  delta = 1;
                i += delta;
            }

            this.render.AntialiasMode = DAntialias.Antialias;
        }

        bool CreateTextLayout(DTextFormat format, string str, Size size,out DTextLayout layout)
        {
            bool has = this.cache.TryGetValue(str, out layout);
            if (has == false)
            {
                layout = this.Factory.CreateGdiTextLayout(format, str, (float)size.Width, (float)size.Height, true);
                layout.Tag = false;
                this.cache.Add(str, layout);
            }
            return has;
        }
    }

}
