﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Media;
using FooEditEngine;
using FooEditEngine.WPF.Properties;
using DotNetTextStore;
using DotNetTextStore.UnmanagedAPI.TSF;
using DotNetTextStore.UnmanagedAPI.WinDef;
using DWriteWarpper;
using FooEditEngine.WPF.Direct2D;

namespace FooEditEngine.WPF
{
    class WPFRender : ITextRender,IDisposable
    {
        CacheManager<string, DTextLayout> Cache = new CacheManager<string, DTextLayout>();
        ResourceManager<TokenType, DColorBrush> SyntaxResources = new ResourceManager<TokenType, DColorBrush>();
        ResourceManager<HilightType, DDrawingEffect> MarkerEffects = new ResourceManager<HilightType, DDrawingEffect>();
        Size textureSize;
        Color ForegroundColor, BackgroundColor, HilightColor, Keyword1Color, Keyword2Color, LiteralColor, UrlColor, ControlCharColor, CommentColor, InsertCaretColor,OverwriteCaretColor,LineMarkerColor;
        DColorBrush InsertCaretBrush,OverwriteCaretBrush,LineMarkerBrush;
        TextStore store;
        DFactory factory;
        DTextFormat format;
        DBitmap bitmap;
        D2DTexture texture;
        DRenderBase render;
        DXGISurface surface;
        D3D10Device device;
        D3D9Device device9;
        D3D9Surface surface9;
        double fontSize;
        float tabWidth = 64.0f;
        FontFamily fontFamily;
        int tabLength;
        bool hasCache;

        public WPFRender(FooTextBox textbox, double width, double height)
        {
            this.fontFamily = textbox.FontFamily;
            this.fontSize = textbox.FontSize;
            this.ForegroundColor = textbox.Foreground;
            this.BackgroundColor = textbox.Background;
            this.ControlCharColor = textbox.ControlChar;
            this.HilightColor = textbox.Hilight;
            this.CommentColor = textbox.Comment;
            this.UrlColor = textbox.URL;
            this.Keyword1Color = textbox.Keyword1;
            this.Keyword2Color = textbox.Keyword2;
            this.LiteralColor = textbox.Literal;
            this.InsertCaretColor = textbox.InsertCaret;
            this.OverwriteCaretColor = textbox.OverwriteCaret;
            this.LineMarkerColor = textbox.LineMarker;
            this.store = textbox.TextStore;

            textbox.Document.Markers.Updated += new EventHandler(Markers_Updated);

            this.factory = new DFactory();

            this.ConstructResource();
            this.ConstructDeviceResource(width, height);
        }

        public D3D9Surface Surface
        {
            get { return this.surface9; }
        }
        
        public FontFamily FontFamily
        {
            get { return this.fontFamily; }
            set
            {
                this.fontFamily = value;
                this.format.Dispose();
                this.format = this.factory.CreateTextFormat(this.fontFamily.Source, (float)this.fontSize);
                this.format.WordWrapping = DWordWrapping.NoWrapping;
                this.TabWidthChar = this.TabWidthChar;
                this.Cache.Clear();
                this.ChangedRenderResource(this, null);
            }
        }

        public double FontSize
        {
            get { return this.fontSize; }
            set
            {
                this.fontSize = value;
                this.format.Dispose();
                this.format = this.factory.CreateTextFormat(this.fontFamily.Source, (float)this.fontSize);
                this.format.WordWrapping = DWordWrapping.NoWrapping;
                this.TabWidthChar = this.TabWidthChar;
                this.Cache.Clear();
                this.ChangedRenderResource(this, null);
            }
        }

        public Color Foreground
        {
            get
            {
                return this.ForegroundColor;
            }
            set
            {
                this.ForegroundColor = value;
                if (this.render == null)
                    return;
                Color2F color = new Color2F(value.R, value.G, value.B, value.A);
                this.MarkerEffects[HilightType.Sold] = new DDrawingEffect(this.render.CreateBrush(color),null,this.factory.CrateStrokeStyke(DLineStyle.Solid));
                this.MarkerEffects[HilightType.Dot] = new DDrawingEffect(this.MarkerEffects[HilightType.Sold].Fore, null, this.factory.CrateStrokeStyke(DLineStyle.Dot));
                this.MarkerEffects[HilightType.Dash] = new DDrawingEffect(this.MarkerEffects[HilightType.Sold].Fore, null, this.factory.CrateStrokeStyke(DLineStyle.Dash));
                this.MarkerEffects[HilightType.DashDot] = new DDrawingEffect(this.MarkerEffects[HilightType.Sold].Fore, null, this.factory.CrateStrokeStyke(DLineStyle.DashDot));
                this.MarkerEffects[HilightType.DashDotDot] = new DDrawingEffect(this.MarkerEffects[HilightType.Sold].Fore, null, this.factory.CrateStrokeStyke(DLineStyle.DashDotDot));
                if(this.render.HasTextRender)
                    this.render.SetDefalutBrush(this.MarkerEffects[HilightType.Sold].Fore);
                
                this.Cache.Clear();
            }
        }

        public Color Background
        {
            get
            {
                return this.BackgroundColor;
            }
            set
            {
                this.BackgroundColor = value;
            }
        }

        public Color InsertCaret
        {
            get
            {
                return this.InsertCaretColor;
            }
            set
            {
                this.InsertCaretColor = value;
                if (this.InsertCaretBrush != null)
                    this.InsertCaretBrush.Dispose();
                if (this.render == null)
                    return;
                this.InsertCaretBrush = this.render.CreateBrush(new Color2F(value.R, value.G, value.B, value.A));
            }
        }

        public Color OverwriteCaret
        {
            get
            {
                return this.OverwriteCaretColor;
            }
            set
            {
                this.OverwriteCaretColor = value;
                if (this.OverwriteCaretBrush != null)
                    this.OverwriteCaretBrush.Dispose();
                if (this.render == null)
                    return;
                this.OverwriteCaretBrush = this.render.CreateBrush(new Color2F(value.R, value.G, value.B, value.A));
            }
        }

        public Color LineMarker
        {
            get
            {
                return this.LineMarkerColor;
            }
            set
            {
                this.LineMarkerColor = value;
                if(this.LineMarkerBrush != null)
                    this.LineMarkerBrush.Dispose();
                if (this.render == null)
                    return;
                this.LineMarkerBrush = this.render.CreateBrush(new Color2F(value.R, value.G, value.B, value.A));
            }
        }

        public Color ControlChar
        {
            get
            {
                return this.ControlCharColor;
            }
            set
            {
                this.ControlCharColor = value;
                if (this.render == null)
                    return;
                Color2F color = new Color2F(value.R, value.G, value.B, value.A);
                this.SyntaxResources[TokenType.Control] = this.render.CreateBrush(color);
                if (this.render.HasTextRender)
                    this.render.SetDefalutBrush(this.SyntaxResources[TokenType.Control]);
                this.Cache.Clear();
            }
        }
        
        public Color Url
        {
            get
            {
                return this.UrlColor;
            }
            set
            {
                this.UrlColor = value;
                if (this.render == null)
                    return;
                Color2F color = new Color2F(value.R, value.G, value.B, value.A);
                this.MarkerEffects[HilightType.Url] = new DDrawingEffect(this.render.CreateBrush(color), null, this.factory.CrateStrokeStyke(DLineStyle.Solid));
                this.Cache.Clear();
            }
        }

        public Color Hilight
        {
            get
            {
                return this.HilightColor;
            }
            set
            {
                this.HilightColor = value;
                if (this.render == null)
                    return;
                Color2F color = new Color2F(value.R, value.G, value.B, value.A);
                Color2F fore = new Color2F(this.ForegroundColor.R, this.ForegroundColor.G, this.ForegroundColor.B, this.ForegroundColor.A);
                this.MarkerEffects[HilightType.Select] = new DDrawingEffect(this.render.CreateBrush(fore), this.render.CreateBrush(color), this.factory.CrateStrokeStyke(DLineStyle.Solid));
                this.Cache.Clear();
            }
        }

        public Color Comment
        {
            get
            {
                return this.CommentColor;
            }
            set
            {
                this.CommentColor = value;
                if (this.render == null)
                    return;
                Color2F color = new Color2F(value.R, value.G, value.B, value.A);
                this.SyntaxResources[TokenType.Comment] = this.render.CreateBrush(color);
                this.Cache.Clear();
            }
        }

        public Color Literal
        {
            get
            {
                return this.LiteralColor;
            }
            set
            {
                this.LiteralColor = value;
                if (this.render == null)
                    return;
                Color2F color = new Color2F(value.R, value.G, value.B, value.A);
                this.SyntaxResources[TokenType.Literal] = this.render.CreateBrush(color);
                this.Cache.Clear();
            }
        }

        public Color Keyword1
        {
            get
            {
                return this.Keyword1Color;
            }
            set
            {
                this.Keyword1Color = value;
                if (this.render == null)
                    return;
                Color2F color = new Color2F(value.R, value.G, value.B, value.A);
                this.SyntaxResources[TokenType.Keyword1] = this.render.CreateBrush(color);
                this.Cache.Clear();
            }
        }

        public Color Keyword2
        {
            get
            {
                return this.Keyword2Color;
            }
            set
            {
                this.Keyword2Color = value;
                if (this.render == null)
                    return;
                Color2F color = new Color2F(value.R, value.G, value.B, value.A);
                this.SyntaxResources[TokenType.Keyword2] = this.render.CreateBrush(color);
                this.Cache.Clear();
            }
        }

        public Rectangle ClipRect
        {
            get;
            set;
        }

        public double LineNemberWidth
        {
            get
            {
                double width = this.GetWidth("0");
                return width * (Int32.Parse(Resources.LineNumberLength) + 1);
            }
        }

        public int TabWidthChar
        {
            get { return this.tabLength; }
            set
            {
                this.tabLength = value;
                this.tabWidth = (float)this.GetWidth("a") * value;
                this.format.TabWidth = this.tabWidth;
                this.Cache.Clear();
            }
        }

        public event ChangedRenderResourceEventHandler ChangedRenderResource;

        public void Resize(double width,double height)
        {
            this.textureSize = new Size(width, height);
            this.DestructDeviceResource();
            this.ConstructDeviceResource(width, height);
        }

        public void ClearLayoutCache()
        {
            this.Cache.Clear();
        }

        public void DrawCachedBitmap(Rectangle rect)
        {
            if (this.render == null)
                return;
            render.DrawBitmap(this.bitmap, rect, 1.0f, rect);
        }

        public void CacheContent()
        {
            if (this.render == null || this.bitmap == null)
                return;
            render.Flush();
            this.bitmap.CopyFromRenderTarget(new Point2U(), this.render, new RectU(0, 0, (uint)this.textureSize.Width, (uint)this.textureSize.Height));
            this.hasCache = true;
        }

        public bool IsVaildCache()
        {
            return this.hasCache;
        }

        public void BegineDraw()
        {
            if (this.render == null)
                return;
            this.render.BeginDraw();
        }

        public void EndDraw()
        {
            if (this.render == null)
                return;
            if (this.render.EndDraw())
            {
                this.DestructDeviceResource();
                this.ConstructDeviceResource(this.textureSize.Width, this.textureSize.Height);
                System.Diagnostics.Debug.WriteLine("Reconstruct Device Resource");
            }
            else
            {
                this.device.Flush();
            }
        }

        public void DrawLineNumber(int lineNumber, double x, double y)
        {
            if (this.render == null)
                return;
            string lineNumberFormat = "{0," + Resources.LineNumberLength + "}";
            DTextLayout layout = this.factory.CreateTextLayout(this.format, string.Format(lineNumberFormat, lineNumber), Int32.MaxValue, Int32.MaxValue);
            this.render.DrawTextLayout(layout, (float)x, (float)y);
        }

        public void DrawLineMarker(Rectangle rect)
        {
            if (this.render == null)
                return;
            this.render.AntialiasMode = DAntialias.Alias;
            this.render.FillRectangle(rect, this.LineMarkerBrush);
            this.render.AntialiasMode = DAntialias.Antialias;
        }

        public void DrawCaret(Rectangle rect, bool transparent)
        {
            if (this.render == null)
                return;
            this.render.AntialiasMode = DAntialias.Alias;
            if (transparent == false)
                this.render.FillRectangle(rect, this.InsertCaretBrush);
            else
                this.render.FillRectangle(rect, this.OverwriteCaretBrush);
            this.render.AntialiasMode = DAntialias.Antialias;
        }

        public void FillBackground(Rectangle rect)
        {
            if (this.render == null)
                return;
            Color value = this.BackgroundColor;
            this.render.Clear(new Color2F(value.R, value.G, value.B, value.A));
        }

        public void DrawOneLine(LineToIndexTable lti, int row, double x, double y, IEnumerable<Selection> SelectRanges, IEnumerable<Marker> MarkerRanges)
        {
            string str = lti[row];

            if (str == string.Empty || str == null || this.render == null)
                return;

            DTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            if ((bool)layout.Tag == false)
            {
                LineToIndexTableData lineData = lti.GetData(row);
                if (lineData.syntax != null)
                {
                    foreach (SyntaxInfo s in lineData.syntax)
                    {
                        layout.SetTextEffect(s.index, s.length, this.SyntaxResources[s.type]);
                    }
                }

                foreach (Marker m in MarkerRanges)
                {
                    if (m.start == -1 || m.length == 0)
                        continue;
                    if(m.hilight != HilightType.None)
                        layout.SetTextEffect(m.start, m.length, this.MarkerEffects[m.hilight]);
                    if (m.hilight != HilightType.Select && m.hilight != HilightType.None)
                        layout.SetUnderline(m.start, m.length, true);
                }
                layout.Tag = true;
            }

            this.render.PushAxisAlignedClip(this.ClipRect, DAntialias.Alias);

            IMarkerEffecter effecter = new HilightMarker(this.MarkerEffects[HilightType.Select].Back);
            foreach (Selection sel in SelectRanges)
            {
                if (sel.length == 0 || sel.start == -1)
                    continue;

                effecter.Apply(this.render, layout, sel.start, sel.length, x, y);
            }

            using (Unlocker locker = this.store.LockDocument(false))
            {
                foreach (TextDisplayAttribute attr in this.store.EnumAttributes())
                {
                    if (attr.startIndex == attr.endIndex)
                        continue;
                    int lineIndex = lti.GetIndexFromLineNumber(row);
                    int lineLength = lti.GetLengthFromLineNumber(row);
                    int length = attr.endIndex - attr.startIndex;
                    int start = attr.startIndex - lineIndex;
                    if (start < 0)
                    {
                        if (attr.endIndex > lineIndex && attr.endIndex < lineIndex + lineLength)
                        {
                            start = 0;
                            length = attr.endIndex - lineIndex;
                        }
                        else if (attr.endIndex >= lineIndex + lineLength)
                        {
                            start = 0;
                            length = lineLength;
                        }
                        else
                        {
                            continue;
                        }
                    }
                    else if (start > lineLength)
                    {
                        continue;
                    }

                    effecter = null;

                    HilightType type = HilightType.None;
                    switch (attr.attribute.lsStyle)
                    {
                        case TF_DA_LINESTYLE.TF_LS_DOT:
                            type = HilightType.Dot;
                            break;
                        case TF_DA_LINESTYLE.TF_LS_SOLID:
                            type = HilightType.Sold;
                            break;
                        case TF_DA_LINESTYLE.TF_LS_DASH:
                            type = HilightType.Dash;
                            break;
                        case TF_DA_LINESTYLE.TF_LS_SQUIGGLE:
                            effecter = new SquilleLineMarker(this.MarkerEffects[HilightType.Sold], 1);
                            break;
                    }

                    if (attr.attribute.fBoldLine)
                        effecter = new LineMarker(this.MarkerEffects[type], 2);
                    else if(effecter == null)
                        effecter = new LineMarker(this.MarkerEffects[type], 1);
                    effecter.Apply(this.render, layout, start, length, x, y);
                }
            }

            render.DrawTextLayout(layout, (float)x, (float)y);

            this.render.PopAxisAlignedClip();
        }

        public int GetIndexFromX(string str, double x)
        {
            if (str == string.Empty || str == null || str[0] == Document.EndOfFile)
                return 0;

            DTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            bool isTrailing, isInsed;
            DHitTestMetrics metrics;
            metrics = layout.HitTextPoint((float)x, 0, out isTrailing, out isInsed);
            if (isTrailing)
                return Util.RoundUp(metrics.textPosition + metrics.length);
            else
                return Util.RoundUp(metrics.textPosition);
        }

        public double GetWidthFromIndex(string str, int index)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            float x, y;
            DHitTestMetrics metrics;
            metrics = layout.HitTestTextPostion(index, false, out x, out y);
            float x2;
            layout.HitTestTextPostion(index, true, out x2, out y);

            return x2 - x;
        }

        public double GetWidth(string str)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            return Util.RoundUp(layout.metrics.widthIncludingTrailingWhitespace);
        }

        public double GetXFromIndex(string str, int index)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            float x, y;
            DHitTestMetrics metrics;
            metrics = layout.HitTestTextPostion(index, false, out x, out y);
            return x;
        }

        public double GetHeight(string str)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            double height = 0;
            DLineMetrics[] metrics = layout.LineMetrics;
            if (metrics != null && metrics.Length > 0)
                height = metrics[0].height;
            return height;
        }

        public int AlignIndexToNearestCluster(string str, int index, AlignDirection flow)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            float x, y;
            DHitTestMetrics metrics;
            metrics = layout.HitTestTextPostion(index, false, out x, out y);

            if (flow == AlignDirection.Forward)
                return Util.RoundUp(metrics.textPosition + metrics.length);
            else if (flow == AlignDirection.Back)
                return Util.RoundUp(metrics.textPosition);
            throw new ArgumentOutOfRangeException();
        }

        public List<LineToIndexTableData> BreakLine(Document doc, int startIndex, int endIndex, double wrapwidth)
        {
            List<LineToIndexTableData> output = new List<LineToIndexTableData>();

            this.format.WordWrapping = DWordWrapping.Wrapping;

            foreach (string str in Util.GetLines(doc, startIndex, endIndex))
            {
                DTextLayout layout = this.factory.CreateTextLayout(this.format, str, (float)wrapwidth, Int32.MaxValue);

                int i = startIndex;
                foreach (DLineMetrics metrics in layout.LineMetrics)
                {
                    if (metrics.length == 0 && metrics.newlineLength == 0)
                        continue;

                    bool lineend = false;
                    if (metrics.newlineLength > 0)
                        lineend = true;

                    output.Add(new LineToIndexTableData(i, (int)metrics.length, lineend, null));
                    i += Util.RoundUp(metrics.length);
                }

                layout.Dispose();

                startIndex += str.Length;
            }

            this.format.WordWrapping = DWordWrapping.NoWrapping;

            if (output.Count > 0)
                output.Last().lineend = true;

            return output;
        }

        public void Dispose()
        {
            this.DestructDeviceResource();
            this.DestructResource();
        }

        void ConstructDeviceResource(double width, double height)
        {
            this.device = new D3D10Device();
            this.texture = this.device.CreateTexture2D((int)width, (int)height);
            this.surface = new DXGISurface(this.texture);
            this.render = this.factory.CreateDxgiSurfaceRenderTarget(this.surface);
            this.device9 = new D3D9Device();
            this.surface9 = this.device9.CreateTexture(this.texture);

            this.bitmap = render.CreateBitmap(new SizeU((uint)width, (uint)height));
            this.hasCache = false;

            this.Foreground = this.ForegroundColor;
            this.Background = this.BackgroundColor;
            this.ControlChar = this.ControlCharColor;
            this.Url = this.UrlColor;
            this.Keyword1 = this.Keyword1Color;
            this.Keyword2 = this.Keyword2Color;
            this.Literal = this.LiteralColor;
            this.Comment = this.CommentColor;
            this.Hilight = this.HilightColor;
            this.LineMarker = this.LineMarkerColor;
            this.InsertCaret = this.InsertCaretColor;
            this.OverwriteCaret = this.OverwriteCaretColor;
            this.render.InitTextRender(this.MarkerEffects[HilightType.Sold].Fore, this.SyntaxResources[TokenType.Control]);
            this.render.AddControlSymbol('\n', UInt32.Parse(Resources.NewLineSymbol));
            this.render.AddControlSymbol('\t', UInt32.Parse(Resources.TabSymbol));
            this.render.AddControlSymbol('　', UInt32.Parse(Resources.FullSpaceSymbol));

            this.textureSize.Width = width;
            this.textureSize.Height = height;
        }

        void DestructDeviceResource()
        {
            this.hasCache = false;
            if(this.bitmap != null)
                this.bitmap.Dispose();
            this.MarkerEffects.Clear();
            this.SyntaxResources.Clear();
            if(this.LineMarkerBrush != null)
                this.LineMarkerBrush.Dispose();
            if(this.InsertCaretBrush != null)
                this.InsertCaretBrush.Dispose();
            if(this.render != null)
                this.render.Dispose();
            if(this.device != null)
                this.device.Dispose();
            if(this.texture != null)
                this.texture.Dispose();
            if(this.surface != null)
                this.surface.Dispose();
            if(this.device9 != null)
                this.device9.Dispose();
            if (this.surface9 != null)
                this.surface9.Dispose();
        }
        
        void ConstructResource()
        {
            this.format = this.factory.CreateTextFormat(this.FontFamily.Source, (float)this.FontSize);
            this.format.TabWidth = this.tabWidth;
            this.format.WordWrapping = DWordWrapping.NoWrapping;
        }

        void DestructResource()
        {
            this.Cache.Clear();
            this.format.Dispose();
            this.factory.Dispose();
        }

        bool CreateTextLayout(DTextFormat format, string str, Size size, out DTextLayout layout)
        {
            bool has = this.Cache.TryGetValue(str, out layout);
            if (has == false)
            {
                layout = this.factory.CreateTextLayout(format, str, (float)size.Width, (float)size.Height);
                layout.Tag = false;
                this.Cache.Add(str, layout);
            }
            return has;
        }

        Color ToColor(Color2F color)
        {
            Color retval = new Color();
            retval.A = (byte)(color.A * 255.0f);
            retval.B = (byte)(color.B * 255.0f);
            retval.G = (byte)(color.G * 255.0f);
            retval.R = (byte)(color.R * 255.0f);
            return retval;
        }

        void Markers_Updated(object sender, EventArgs e)
        {
            this.ClearLayoutCache();
        }
    }
}
