﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Documents;
using System.Windows.Interop;
using System.Windows.Threading;
using DotNetTextStore;
using DotNetTextStore.UnmanagedAPI.WinDef;

namespace FooEditEngine.WPF
{
    public class FooTextBox : Control
    {
        [DllImport("user32.dll")]
        static extern uint GetCaretBlinkTime(); 
        
        View View;
        Controller Controller;
        WPFRender Render;
        Image image;
        D3DImage imageSource;
        ScrollBar verticalScrollBar, horizontalScrollBar;
        TextStore textStore;
        DispatcherTimer timer;
        
        static FooTextBox()
        {
            DefaultStyleKeyProperty.OverrideMetadata(typeof(FooTextBox), new FrameworkPropertyMetadata(typeof(FooTextBox)));
        }

        public FooTextBox()
        {
            this.textStore = new TextStore();
            this.textStore.IsReadOnly += () => false;
            this.textStore.GetStringLength += () => this.Document.Length;
            this.textStore.GetString += _textStore_GetString;
            this.textStore.GetSelectionIndex += _textStore_GetSelectionIndex;
            this.textStore.SetSelectionIndex += _textStore_SetSelectionIndex;
            this.textStore.InsertAtSelection += _textStore_InsertAtSelection;
            this.textStore.GetHWnd += _textStore_GetHWnd;
            this.textStore.GetScreenExtent += _textStore_GetScreenExtent;
            this.textStore.GetStringExtent += _textStore_GetStringExtent;
            this.textStore.CompositionStarted += textStore_CompositionStarted;
            this.textStore.CompositionEnded += textStore_CompositionEnded;

            this.Document = new Document();

            this.Render = new WPFRender(this, 200, 200);

            this.View = new View(this.Document, this.Render);
            this.View.InsertMode = this.InsertMode;
            this.View.CaretWidthOnInsertMode = SystemParameters.CaretWidth;
            this.View.CaretBlink = true;
            this.View.CaretBlinkTime = (int)GetCaretBlinkTime() * 2;
            this.View.CaretWidthOnInsertMode = SystemParameters.CaretWidth;
            this.View.DrawLineNumber = this.DrawLineNumber;
            this.View.HideLineMarker = !this.DrawCaretLine;
            this.View.UrlMark = this.MarkURL;
            this.View.TabStops = this.TabChars;

            this.Controller = new Controller(this.Document, this.View);
            this.Controller.CaretMoved += new EventHandler(Controller_CaretMoved);

            this.imageSource = new D3DImage();
            this.imageSource.Lock();
            this.imageSource.SetBackBuffer(D3DResourceType.IDirect3DSurface9, this.Render.Surface.Handel);
            this.imageSource.Unlock();

            this.image = new Image();
            this.image.Source = this.imageSource;
            this.image.Stretch = Stretch.None;
            this.image.HorizontalAlignment = HorizontalAlignment.Left;
            this.image.VerticalAlignment = VerticalAlignment.Top;

            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Copy, CopyCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Cut, CutCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Paste, PasteCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Delete, DeleteCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.SelectAll, SelectAllCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Undo, UndoCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Redo, RedoCommand));
            this.CommandBindings.Add(new CommandBinding(EditingCommands.ToggleInsert, ToggleInsertCommand));
            var RectSelectCommand = new RoutedCommand("ToggleRectSelect", typeof(FooTextBox));
            this.CommandBindings.Add(new CommandBinding(RectSelectCommand, ToggleRectSelectCommand));

            this.InputBindings.Add(new InputBinding(ApplicationCommands.Copy, new KeyGesture(Key.C, ModifierKeys.Control)));
            this.InputBindings.Add(new InputBinding(ApplicationCommands.Cut, new KeyGesture(Key.X, ModifierKeys.Control)));
            this.InputBindings.Add(new InputBinding(ApplicationCommands.Paste, new KeyGesture(Key.V, ModifierKeys.Control)));
            this.InputBindings.Add(new InputBinding(ApplicationCommands.Delete, new KeyGesture(Key.Delete, ModifierKeys.None)));
            this.InputBindings.Add(new InputBinding(ApplicationCommands.SelectAll, new KeyGesture(Key.A, ModifierKeys.Control)));
            this.InputBindings.Add(new InputBinding(ApplicationCommands.Undo, new KeyGesture(Key.Z, ModifierKeys.Control)));
            this.InputBindings.Add(new InputBinding(ApplicationCommands.Redo, new KeyGesture(Key.Y, ModifierKeys.Control)));
            this.InputBindings.Add(new InputBinding(EditingCommands.ToggleInsert, new KeyGesture(Key.Insert, ModifierKeys.None)));
            this.InputBindings.Add(new InputBinding(RectSelectCommand, new KeyGesture(Key.B, ModifierKeys.Control)));

            this.IsTabStop = false;

            this.timer = new DispatcherTimer();
            this.timer.Interval = new TimeSpan(0, 0, 0, 0, 100);
            this.timer.Tick += new EventHandler(timer_Tick);
        }

        /// <summary>
        /// テンプレートを適用します
        /// </summary>
        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            Grid grid = this.GetTemplateChild("PART_Grid") as Grid;
            if (grid != null)
            {
                Grid.SetRow(this.image, 0);
                Grid.SetColumn(this.image, 0);
                grid.Children.Add(this.image);
            }

            this.horizontalScrollBar = this.GetTemplateChild("PART_HorizontalScrollBar") as ScrollBar;
            if (this.horizontalScrollBar != null)
            {
                this.horizontalScrollBar.SmallChange = 10;
                this.horizontalScrollBar.LargeChange = 100;
                this.horizontalScrollBar.Maximum = this.horizontalScrollBar.LargeChange + 1;
                this.horizontalScrollBar.Scroll += new ScrollEventHandler(horizontalScrollBar_Scroll);
            }

            this.verticalScrollBar = this.GetTemplateChild("PART_VerticalScrollBar") as ScrollBar;
            if (this.verticalScrollBar != null)
            {
                this.verticalScrollBar.SmallChange = 1;
                this.verticalScrollBar.LargeChange = 1;
                this.verticalScrollBar.Maximum = this.verticalScrollBar.LargeChange + 1;
                this.verticalScrollBar.Scroll += new ScrollEventHandler(verticalScrollBar_Scroll);
            }
        }

        /// <summary>
        /// ドキュメントを選択する
        /// </summary>
        /// <param name="start">開始インデックス</param>
        /// <param name="length">長さ</param>
        public void Select(int start, int length)
        {
            this.Controller.Select(start, length);
        }

        /// <summary>
        /// キャレットを指定した行に移動させます
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <remarks>このメソッドを呼び出すと選択状態は解除されます</remarks>
        public void JumpCaret(int index)
        {
            this.Controller.JumpCaret(index);
        }
        /// <summary>
        /// キャレットを指定した行と桁に移動させます
        /// </summary>
        /// <param name="row">行番号</param>
        /// <param name="col">桁</param>
        /// <remarks>このメソッドを呼び出すと選択状態は解除されます</remarks>
        public void JumpCaret(int row, int col)
        {
            this.Controller.JumpCaret(row, col);
        }

        /// <summary>
        /// ドキュメント全体を選択する
        /// </summary>
        public void SelectAll()
        {
            this.Controller.Select(0, this.Document.Length - 1);
        }

        /// <summary>
        /// 選択を解除する
        /// </summary>
        public void DeSelectAll()
        {
            this.Controller.DeSelectAll();
        }

        /// <summary>
        /// 対応する座標を返します
        /// </summary>
        /// <param name="tp">テキストポイント</param>
        /// <returns>座標</returns>
        /// <remarks>テキストポイントがクライアント領域の原点より外にある場合、返される値は原点に丸められます</remarks>
        public System.Windows.Point GetPostionFromTextPoint(TextPoint tp)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            return this.View.GetPostionFromTextPoint(tp);
        }

        /// <summary>
        /// 対応するテキストポイントを返します
        /// </summary>
        /// <param name="p">クライアント領域の原点を左上とする座標</param>
        /// <returns>テキストポイント</returns>
        public TextPoint GetTextPointFromPostion(System.Windows.Point p)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            return this.View.GetTextPointFromPostion(p);
        }

        /// <summary>
        /// インデックスに対応する座標を得ます
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <returns>座標を返す</returns>
        public System.Windows.Point GetPostionFromIndex(int index)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            TextPoint tp = this.View.GetLayoutLineFromIndex(index);
            return this.View.GetPostionFromTextPoint(tp);
        }

        /// <summary>
        /// 座標からインデックスに変換します
        /// </summary>
        /// <param name="p">座標</param>
        /// <returns>インデックスを返す</returns>
        public int GetIndexFromPostion(System.Windows.Point p)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            TextPoint tp = this.View.GetTextPointFromPostion(p);
            return this.View.GetIndexFromLayoutLine(tp);
        }

        public void Refresh()
        {
            this.Refresh(this.View.PageBound);
        }

        void Refresh(Rectangle updateRect)
        {
            if (this.imageSource.IsFrontBufferAvailable == false)
                return;

            this.timer.Stop();

            this.imageSource.Lock();
            this.Render.BegineDraw();
            this.View.Draw(updateRect);
            this.Render.EndDraw();
            this.imageSource.AddDirtyRect(new Int32Rect(0, 0, (int)this.imageSource.Width, (int)this.imageSource.Height));
            this.imageSource.Unlock();

            this.timer.Start();
        }

        void timer_Tick(object sender, EventArgs e)
        {
            string str = this.View.LayoutLines[this.View.CaretPostion.row];
            double width = this.Render.GetWidthFromIndex(str, this.View.CaretPostion.col);
            double height = this.Render.GetHeight(str);
            Rectangle updateRect = new Rectangle(
                this.View.CaretLocation.X,
                this.View.CaretLocation.Y,
                width,
                height);

            this.Refresh(updateRect);
        }

        #region Commands
        void CopyCommand(object sender, RoutedEventArgs e)
        {
            string text = this.Controller.SelectedText;
            if (text != null && text != string.Empty)
                Clipboard.SetText(text);
        }

        void CutCommand(object sender, RoutedEventArgs e)
        {
            string text = this.Controller.SelectedText;
            if (text != null && text != string.Empty)
            {
                Clipboard.SetText(text);
                this.Controller.SelectedText = "";
                this.textStore.NotifySelectionChanged();
                this.textStore.NotifyTextChanged(this.Document.Length + text.Length, this.Document.Length);
            }
        }

        void PasteCommand(object sender, RoutedEventArgs e)
        {
            if (Clipboard.ContainsText() == false)
                return;
            string text = Clipboard.GetText();
            this.Controller.SelectedText = text;
            this.textStore.NotifySelectionChanged();
            this.textStore.NotifyTextChanged(this.Document.Length + text.Length, this.Document.Length);
            this.Refresh();
        }

        void DeleteCommand(object sender, RoutedEventArgs e)
        {
            int oldLength = this.Document.Length;
            this.Controller.DoDeleteAction();
            this.textStore.NotifySelectionChanged();
            this.textStore.NotifyTextChanged(oldLength, this.Document.Length);
            this.Refresh();
        }

        void SelectAllCommand(object sender, RoutedEventArgs e)
        {
            this.Controller.Select(0, this.Document.Length - 1);
            this.textStore.NotifySelectionChanged();
            this.Refresh();
        }

        void UndoCommand(object sender, RoutedEventArgs e)
        {
            int oldLength = this.Document.Length;
            this.Document.UndoManager.undo();
            this.textStore.NotifySelectionChanged();
            this.textStore.NotifyTextChanged(oldLength, this.Document.Length);
            this.Refresh();
        }

        void RedoCommand(object sender, RoutedEventArgs e)
        {
            int oldLength = this.Document.Length;
            this.Document.UndoManager.redo();
            this.textStore.NotifySelectionChanged();
            this.textStore.NotifyTextChanged(oldLength, this.Document.Length);
            this.Refresh();
        }

        void ToggleInsertCommand(object sender, RoutedEventArgs e)
        {
            if (this.View.InsertMode)
                this.View.InsertMode = false;
            else
                this.View.InsertMode = true;
            this.Refresh();
        }

        void ToggleRectSelectCommand(object sender, RoutedEventArgs e)
        {
            if (this.Controller.RectSelection)
            {
                this.Controller.RectSelection = false;
                InputMethod.SetIsInputMethodEnabled(this, true);
            }
            else
            {
                this.Controller.RectSelection = true;
                InputMethod.SetIsInputMethodEnabled(this, false);
            }
        }
        #endregion
        #region TSF
        int SelStart, SelEnd;   //Documentとは別に持たせないと例外が発生する
        bool ComstionNow = false;

        internal TextStore TextStore
        {
            get { return this.textStore; }
        }

        void textStore_CompositionEnded()
        {
            this.ComstionNow = false;
            this.Controller.JumpCaret(this.SelStart);
        }

        bool textStore_CompositionStarted()
        {
            if(this.Controller.RectSelection)
                return false;
            this.ComstionNow = true;
            return true;
        }

        string _textStore_GetString(int start, int length)
        {
            return this.Document.ToString(start, length);
        }

        IntPtr _textStore_GetHWnd()
        {
            var hwndSource = HwndSource.FromVisual(this) as HwndSource;
            if (hwndSource != null)
                return hwndSource.Handle;
            else
                return IntPtr.Zero;
        }

        void _textStore_GetStringExtent(
            int i_startIndex,
            int i_endIndex,
            out POINT o_topLeft,
            out POINT o_bottomRight
        )
        {
            var endIndex = i_endIndex < 0 ? this.Document.Length - 1 : i_endIndex;
            var startPos = new Point(0, 0);
            var endPos = new Point(0, 0);
            TextPoint endTextPoint;
            View view = this.View;

            TextPoint startTextPoint = view.GetLayoutLineFromIndex(i_startIndex);
            double x = this.Render.GetXFromIndex(view.LayoutLines[startTextPoint.row],
                i_startIndex - view.LayoutLines.GetIndexFromLineNumber(startTextPoint.row));
            if (this.ComstionNow)
            {
                if (x < view.Src.X ||
                    x > view.Src.X + view.PageBound.Width)
                {
                    view.TryScroll(x, this.View.Src.Row);
                }
            }
            startPos = this.TranslatePoint(view.GetPostionFromTextPoint(startTextPoint), this);

            endTextPoint = view.GetLayoutLineFromIndex(endIndex);
            endPos = view.GetPostionFromTextPoint(endTextPoint);
            endPos = this.TranslatePoint(endPos, this);
            endPos.Y += this.Render.GetHeight(view.LayoutLines[endTextPoint.row])
                + MarkerFactory.boldThickness;

            startPos = PointToScreen(startPos);
            endPos = PointToScreen(endPos);

            o_topLeft = new POINT((int)startPos.X, (int)startPos.Y);
            o_bottomRight = new POINT((int)endPos.X, (int)endPos.Y);
        }

        void _textStore_GetScreenExtent(out POINT o_topLeft, out POINT o_bottomRight)
        {
            var pointTopLeft = new Point(0, 0);
            var pointBottomRight = new Point(this.RenderSize.Width, this.RenderSize.Height);

            pointTopLeft = PointToScreen(pointTopLeft);
            pointBottomRight = PointToScreen(pointBottomRight);

            o_topLeft = new POINT((int)pointTopLeft.X, (int)pointTopLeft.Y);
            o_bottomRight = new POINT((int)pointBottomRight.X, (int)pointBottomRight.Y);
        }

        void _textStore_GetSelectionIndex(out int o_startIndex, out int o_endIndex)
        {
            o_startIndex = this.SelStart;
            o_endIndex = this.SelEnd;
        }

        void _textStore_SetSelectionIndex(int i_startIndex, int i_endIndex)
        {
            this.SelStart = i_startIndex;
            this.SelEnd = i_endIndex;
            if (i_startIndex == i_endIndex && this.ComstionNow == false)
                this.Controller.JumpCaret(i_startIndex);
        }

        void _textStore_InsertAtSelection(string i_value)
        {
            if (this.SelStart != this.SelEnd)
                this.Controller.Select(this.SelStart, this.SelEnd - this.SelStart);
            this.Controller.DoInputString(i_value);
        }

        /// <summary>
        /// キーボードフォーカスが取得されたときに呼ばれます
        /// </summary>
        /// <param name="e"></param>
        protected override void OnGotKeyboardFocus(KeyboardFocusChangedEventArgs e)
        {
            base.OnGotKeyboardFocus(e);
            this.textStore.SetFocus();
        }

        /// <summary>
        /// 文字列が入力されたときに呼ばれます
        /// </summary>
        /// <param name="e"></param>
        protected override void OnTextInput(TextCompositionEventArgs e)
        {
            int oldLength = this.Document.Length;
            if (e.Text == "\r")
            {
                this.Controller.DoEnterAction();
                this.textStore.NotifySelectionChanged();
                this.textStore.NotifyTextChanged(oldLength, this.Document.Length);
            }
            else if (e.Text == "\b")
            {
                this.Controller.DoBackSpaceAction();
                this.textStore.NotifySelectionChanged();
                this.textStore.NotifyTextChanged(oldLength, this.Document.Length);
            }
            else
                this.textStore.InsertTextAtSelection(e.Text);
            this.Refresh();
            base.OnTextInput(e);
        }
        #endregion
        #region Event
        /// <summary>
        /// キーが押されたときに呼ばれます。
        /// </summary>
        /// <param name="e">イベントパラメーター</param>
        protected override void OnPreviewKeyDown(System.Windows.Input.KeyEventArgs e)
        {
            int toRow;
            ModifierKeys modiferKeys = e.KeyboardDevice.Modifiers;
            switch (e.Key)
            {
                case Key.Up:
                    this.Controller.MoveCaretVertical(-1, this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift));
                    this.textStore.NotifySelectionChanged();
                    this.Refresh();
                    break;
                case Key.Down:
                    this.Controller.MoveCaretVertical(+1, this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift));
                    this.textStore.NotifySelectionChanged();
                    this.Refresh();
                    break;
                case Key.Left:
                    this.Controller.MoveCaretHorizontical(-1, this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift), this.IsPressedModifierKey(modiferKeys, ModifierKeys.Control));
                    this.textStore.NotifySelectionChanged();
                    this.Refresh();
                    break;
                case Key.Right:
                    this.Controller.MoveCaretHorizontical(1, this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift), this.IsPressedModifierKey(modiferKeys, ModifierKeys.Control));
                    this.textStore.NotifySelectionChanged();
                    this.Refresh();
                    break;
                case Key.PageUp:
                    toRow = Math.Max(0, this.View.Src.Row - this.View.LineCountOnScreen);
                    this.Controller.ScrollWithCaretMove(this.View.Src.X, toRow, this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift));
                    this.textStore.NotifySelectionChanged();
                    this.Refresh();
                    break;
                case Key.PageDown:
                    toRow = Math.Min(this.View.Src.Row + this.View.LineCountOnScreen, this.View.LayoutLines.Count - 1);
                    this.Controller.ScrollWithCaretMove(this.View.Src.X, toRow, this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift));
                    this.textStore.NotifySelectionChanged();
                    this.Refresh();
                    break;
                case Key.Home:
                    if (this.IsPressedModifierKey(modiferKeys, ModifierKeys.Control))
                        this.Controller.JumpToHead(this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift));
                    else
                        this.Controller.JumpCaret(this.View.CaretPostion.row, 0);
                    this.textStore.NotifySelectionChanged();
                    this.Refresh();
                    break;
                case Key.End:
                    if (this.IsPressedModifierKey(modiferKeys, ModifierKeys.Control))
                        this.Controller.JumpToEnd(this.IsPressedModifierKey(modiferKeys, ModifierKeys.Shift));
                    else
                        this.Controller.JumpCaret(this.View.CaretPostion.row, this.View.LayoutLines[this.View.CaretPostion.row].Length - 1);
                    this.textStore.NotifySelectionChanged();
                    this.Refresh();
                    break;
            }
            base.OnPreviewKeyDown(e);
        }

        bool IsPressedModifierKey(ModifierKeys keys, ModifierKeys pressed)
        {
            if (keys == pressed)
                return true;
            if ((keys & pressed) == pressed)
                return true;
            return false;
        }

        /// <summary>
        /// ダブルクリックされたときに呼ばれます
        /// </summary>
        /// <param name="e">イベントパラメーター</param>
        protected override void OnPreviewMouseDoubleClick(MouseButtonEventArgs e)
        {
            if (e.LeftButton == MouseButtonState.Pressed)
            {
                System.Windows.Point p = e.GetPosition(this);
                TextPoint tp = this.View.GetTextPointFromPostion(p);
                int index = this.View.LayoutLines.GetIndexFromTextPoint(tp);

                this.Controller.SelectWord(index);
                this.textStore.NotifySelectionChanged();
                this.Refresh();
            }
            base.OnPreviewMouseDoubleClick(e);
        }

        /// <summary>
        /// マウスボタンが押されたときに呼ばれます
        /// </summary>
        /// <param name="e">イベントパラメーター</param>
        protected override void OnPreviewMouseDown(MouseButtonEventArgs e)
        {
            if (e.LeftButton == MouseButtonState.Pressed)
            {
                System.Windows.Point p = e.GetPosition(this);
                TextPoint tp = this.View.GetTextPointFromPostion(p);
                int index = this.View.LayoutLines.GetIndexFromTextPoint(tp);

                this.Controller.JumpCaret(tp.row, tp.col);
                this.textStore.NotifySelectionChanged();
                this.Refresh();
            }
            base.OnPreviewMouseDown(e);
        }

        /// <summary>
        /// マウスが移動したときに呼ばれます
        /// </summary>
        /// <param name="e">イベントパラメーター</param>
        protected override void OnPreviewMouseMove(MouseEventArgs e)
        {
            System.Windows.Point p = e.GetPosition(this);
            TextPoint tp = this.View.GetTextPointFromPostion(p);

            if (this.Controller.IsMarker(tp, HilightType.Url))
                this.Cursor = Cursors.Hand;
            else
                this.Cursor = Cursors.IBeam;

            if (e.LeftButton == MouseButtonState.Pressed)
            {
                this.Controller.MoveCaretAndSelect(tp);
                this.textStore.NotifySelectionChanged();
                this.Refresh();
            }

            base.OnPreviewMouseMove(e);
        }

        /// <summary>
        /// サイズが変更されたときに呼ばれます
        /// </summary>
        /// <param name="sizeInfo">イベントデータ</param>
        protected override void OnRenderSizeChanged(SizeChangedInfo sizeInfo)
        {
            this.View.PageBound = new Rectangle(0, 0, sizeInfo.NewSize.Width, sizeInfo.NewSize.Height);

            this.verticalScrollBar.LargeChange = this.View.LineCountOnScreen;
            this.verticalScrollBar.Maximum = this.verticalScrollBar.LargeChange + 1;

            this.imageSource.Lock();
            this.Render.Resize(sizeInfo.NewSize.Width, sizeInfo.NewSize.Height);
            this.imageSource.SetBackBuffer(D3DResourceType.IDirect3DSurface9,this.Render.Surface.Handel);
            this.imageSource.Unlock();

            this.Refresh();
        }

        void horizontalScrollBar_Scroll(object sender, ScrollEventArgs e)
        {
            if (this.horizontalScrollBar == null || this.verticalScrollBar == null)
                return;
            this.View.TryScroll(this.horizontalScrollBar.Value, (int)this.View.Src.Row);
            this.Refresh();
        }

        void verticalScrollBar_Scroll(object sender, ScrollEventArgs e)
        {
            if (this.horizontalScrollBar == null || this.verticalScrollBar == null)
                return;
            int newRow = (int)this.verticalScrollBar.Value;
            if (newRow >= this.View.LayoutLines.Count)
                return;
            this.Controller.ScrollWithCaretMove(
                this.View.Src.X,
                newRow,
                false);
            this.Refresh();
        }

        void Controller_CaretMoved(object sender, EventArgs e)
        {
            if (this.horizontalScrollBar == null || this.verticalScrollBar == null)
                return;
            View view = this.View;
            if (view.Src.Row > this.verticalScrollBar.Maximum)
                this.verticalScrollBar.Maximum = view.Src.Row + view.LineCountOnScreen + 1;
            if (view.Src.X > this.horizontalScrollBar.Maximum)
                this.horizontalScrollBar.Maximum = view.Src.X + view.PageBound.Width + 1;
            this.verticalScrollBar.Value = view.Src.Row;
            this.horizontalScrollBar.Value = view.Src.X;
            this.SelStart = this.SelEnd = this.Controller.SelectionStart;   //同期をとらないと挿入位置がずれる
            this.View.CaretBlink = this.View.CaretBlink;
        }

        /// <summary>
        /// プロパティーが変更されたときに呼ばれます
        /// </summary>
        /// <param name="e">イベントパラメーター</param>
        protected override void OnPropertyChanged(DependencyPropertyChangedEventArgs e)
        {
            switch (e.Property.Name)
            {
                case "WordRap":
                    this.View.isLineBreak = this.WordRap;
                    this.Controller.DeSelectAll();
                    this.View.AdjustCaret();
                    break;
                case "InsertMode":
                    this.View.InsertMode = this.InsertMode;
                    break;
                case "TabChars":
                    this.View.TabStops = this.TabChars;
                    break;
                case "RectSelectMode":
                    this.Controller.RectSelection = this.RectSelectMode;
                    break;
                case "DrawCaretLine":
                    this.View.HideLineMarker = this.DrawCaretLine;
                    break;
                case "DrawLineNumber":
                    this.View.DrawLineNumber = this.DrawLineNumber;
                    break;
                case "FontFamily":
                    this.Render.FontFamily = this.FontFamily;
                    break;
                case "FontSize":
                    this.Render.FontSize = this.FontSize;
                    break;
                case "Foreground":
                    this.Render.Foreground = this.Foreground;
                    break;
                case "Background":
                    this.Render.Background = this.Background;
                    break;
                case "ControlChar":
                    this.Render.ControlChar = this.ControlChar;
                    break;
                case "Keyword1":
                    this.Render.Keyword1 = this.Keyword1;
                    break;
                case "Keyword2":
                    this.Render.Keyword2 = this.Keyword2;
                    break;
                case "Comment":
                    this.Render.Comment = this.Comment;
                    break;
                case "Literal":
                    this.Render.Literal = this.Literal;
                    break;
                case "URL":
                    this.Render.Url = this.URL;
                    break;
                case "MarkURL":
                    this.View.UrlMark = this.MarkURL;
                    break;
            }
            base.OnPropertyChanged(e);
        }
        #endregion
        #region property
        /// <summary>
        /// シンタックスハイライターを表す
        /// </summary>
        public IHilighter hilighter
        {
            get { return this.View.Hilighter; }
            set { this.View.Hilighter = value; }
        }

        /// <summary>
        /// ドキュメントを表す
        /// </summary>
        public Document Document
        {
            get;
            private set;
        }

        /// <summary>
        /// レイアウト行を表す
        /// </summary>
        public LineToIndexTable LayoutLineCollection
        {
            get { return this.View.LayoutLines; }
        }

        /// <summary>
        /// 選択範囲の開始インデックス
        /// </summary>
        /// <remarks>SelectionLengthが0の場合はキャレット位置を表します</remarks>
        public int SelectionStart
        {
            get { return this.Controller.SelectionStart; }
        }

        /// <summary>
        /// 選択範囲の長さ
        /// </summary>
        /// <remarks>矩形選択モードの場合、選択範囲の文字数ではなく、開始位置から終了位置までの長さとなります</remarks>
        public int SelectionLength
        {
            get { return this.Controller.SelectionLength; }
        }

        /// <summary>
        /// 矩形選択を行うかどうか
        /// </summary>
        public bool RectSelection
        {
            get { return this.Controller.RectSelection; }
            set { this.Controller.RectSelection = value; }
        }

        public Color Foreground
        {
            get { return (Color)GetValue(ForegroundProperty); }
            set { SetValue(ForegroundProperty, value); }
        }

        // Using a DependencyProperty as the backing store for Foreground.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty ForegroundProperty =
            DependencyProperty.Register("Foreground", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(SystemColors.ControlTextColor));

        public Color Background
        {
            get { return (Color)GetValue(BackgroundProperty); }
            set { SetValue(BackgroundProperty, value); }
        }

        // Using a DependencyProperty as the backing store for Background.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty BackgroundProperty =
            DependencyProperty.Register("Background", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(SystemColors.ControlColor));
        
        /// <summary>
        /// コントロールコードのブラシを表す
        /// </summary>
        public Color ControlChar
        {
            get { return (Color)GetValue(ControlCharProperty); }
            set { SetValue(ControlCharProperty, value); }
        }

        // Using a DependencyProperty as the backing store for ControlChar.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty ControlCharProperty =
            DependencyProperty.Register("ControlChar", typeof(Color), typeof(FooTextBox),  new FrameworkPropertyMetadata(Colors.Gray));
        
        /// <summary>
        /// 選択時の背景ブラシを表す
        /// </summary>
        public Color Hilight
        {
            get { return (Color)GetValue(HilightProperty); }
            set { SetValue(HilightProperty, value); }
        }

        // Using a DependencyProperty as the backing store for Hilight.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty HilightProperty =
            DependencyProperty.Register("Hilight", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(SystemColors.HighlightColor));
        
        /// <summary>
        /// キーワード１のブラシを表す
        /// </summary>
        public Color Keyword1
        {
            get { return (Color)GetValue(Keyword1Property); }
            set { SetValue(Keyword1Property, value); }
        }

        // Using a DependencyProperty as the backing store for Keyword1.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty Keyword1Property =
            DependencyProperty.Register("Keyword1", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(Colors.Blue));

        /// <summary>
        /// キーワード2のブラシを表す
        /// </summary>
        public Color Keyword2
        {
            get { return (Color)GetValue(Keyword2Property); }
            set { SetValue(Keyword2Property, value); }
        }

        // Using a DependencyProperty as the backing store for Keyword2.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty Keyword2Property =
            DependencyProperty.Register("Keyword2", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(Colors.Aqua));

        /// <summary>
        /// コメントのブラシを表す
        /// </summary>
        public Color Comment
        {
            get { return (Color)GetValue(CommentProperty); }
            set { SetValue(CommentProperty, value); }
        }

        // Using a DependencyProperty as the backing store for Comment.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty CommentProperty =
            DependencyProperty.Register("Comment", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(Colors.Green));

        /// <summary>
        /// 文字リテラルのブラシを表す
        /// </summary>
        public Color Literal
        {
            get { return (Color)GetValue(LiteralProperty); }
            set { SetValue(LiteralProperty, value); }
        }

        // Using a DependencyProperty as the backing store for Literal.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty LiteralProperty =
            DependencyProperty.Register("Literal", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(Colors.Brown));

        /// <summary>
        /// URLのブラシを表す
        /// </summary>
        public Color URL
        {
            get { return (Color)GetValue(URLProperty); }
            set { SetValue(URLProperty, value); }
        }

        // Using a DependencyProperty as the backing store for Literal.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty URLProperty =
            DependencyProperty.Register("URL", typeof(Color), typeof(FooTextBox), new FrameworkPropertyMetadata(Colors.Blue));

        /// <summary>
        /// 挿入モードなら真を返し、そうでないなら、偽を返す
        /// </summary>
        public bool InsertMode
        {
            get { return (bool)GetValue(InsertModeProperty); }
            set { SetValue(InsertModeProperty, value); }
        }

        // Using a DependencyProperty as the backing store for InsertMode.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty InsertModeProperty =
            DependencyProperty.Register("InsertMode",
            typeof(bool),
            typeof(FooTextBox),
            new FrameworkPropertyMetadata(true));

        /// <summary>
        /// タブの文字数を表す
        /// </summary>
        public int TabChars
        {
            get { return (int)GetValue(TabCharsProperty); }
            set { SetValue(TabCharsProperty, value); }
        }

        // Using a DependencyProperty as the backing store for TabChars.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty TabCharsProperty =
            DependencyProperty.Register("TabChars",
            typeof(int),
            typeof(FooTextBox),
            new FrameworkPropertyMetadata(4));

        /// <summary>
        /// 矩形選択モードなら真を返し、そうでないなら偽を返す
        /// </summary>
        public bool RectSelectMode
        {
            get { return (bool)GetValue(RectSelectModeProperty); }
            set { SetValue(RectSelectModeProperty, value); }
        }

        // Using a DependencyProperty as the backing store for RectSelectMode.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty RectSelectModeProperty =
            DependencyProperty.Register("RectSelectMode", typeof(bool), typeof(FooTextBox), new FrameworkPropertyMetadata(false));

        /// <summary>
        /// ワードラップを行うなら真を返し、そうでないなら偽を返す
        /// </summary>
        public bool WordRap
        {
            get { return (bool)GetValue(WordRapProperty); }
            set { SetValue(WordRapProperty, value); }
        }

        // Using a DependencyProperty as the backing store for WordRap.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty WordRapProperty =
            DependencyProperty.Register("WordRap", typeof(bool), typeof(FooTextBox), new FrameworkPropertyMetadata(false));

        /// <summary>
        /// キャレットラインを描くなら真。そうでないなら偽を返す
        /// </summary>
        public bool DrawCaretLine
        {
            get { return (bool)GetValue(DrawCaretLineProperty); }
            set { SetValue(DrawCaretLineProperty, value); }
        }

        // Using a DependencyProperty as the backing store for DrawCaretLine.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty DrawCaretLineProperty =
            DependencyProperty.Register("DrawCaretLine", typeof(bool), typeof(FooTextBox), new FrameworkPropertyMetadata(true));

        /// <summary>
        /// 行番号を描くなら真。そうでなければ偽
        /// </summary>
        public bool DrawLineNumber
        {
            get { return (bool)GetValue(DrawLineNumberProperty); }
            set { SetValue(DrawLineNumberProperty, value); }
        }

        // Using a DependencyProperty as the backing store for DrawLineNumber.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty DrawLineNumberProperty =
            DependencyProperty.Register("DrawLineNumber", typeof(bool), typeof(FooTextBox), new FrameworkPropertyMetadata(false));

        public bool MarkURL
        {
            get { return (bool)GetValue(MarkURLProperty); }
            set { SetValue(MarkURLProperty, value); }
        }

        // Using a DependencyProperty as the backing store for MarkURL.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty MarkURLProperty =
            DependencyProperty.Register("MarkURL", typeof(bool), typeof(FooTextBox), new FrameworkPropertyMetadata(true));
        #endregion
    }
}
