#include "CustomTextRenderer.h"
#include "StdAfx.h"
#include "DrawingEffect.h"

#define SYMBOL_NODEFINE 0

/************************************************************************
 *
 * File: CustomTextRenderer.cpp
 *
 * Description: 
 * 
 * 
 *  This file is part of the Microsoft Windows SDK Code Samples.
 * 
 *  Copyright (C) Microsoft Corporation.  All rights reserved.
 * 
 * This source code is intended only as a supplement to Microsoft
 * Development Tools and/or on-line documentation.  See these other
 * materials for detailed information regarding Microsoft code samples.
 * 
 * THIS CODE AND INFORMATION ARE PROVIDED AS IS WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
 * PARTICULAR PURPOSE.
 * 
 ************************************************************************/

/******************************************************************
*                                                                 *
*  CustomTextRenderer::CustomTextRenderer                         *
*                                                                 *
*  The constructor stores the Direct2D factory, the render        *
*  target.                                                        *
*                                                                 *
******************************************************************/

CustomTextRenderer::CustomTextRenderer(
    ID2D1RenderTarget* pRT,
	ID2D1SolidColorBrush* defalutForeBrush,
	ID2D1SolidColorBrush* controlForeBrush
    )
:
cRefCount_(0), 
pRT_(pRT)
{
    pRT_->AddRef();
	this->defaultForeBrush = defalutForeBrush;
	this->defaultForeBrush->AddRef();
	this->controlForeBrush = controlForeBrush;
	this->controlForeBrush->AddRef();
}

/******************************************************************
*                                                                 *
*  CustomTextRenderer::~CustomTextRenderer                        *
*                                                                 *
*  Destructor releases the factory and render target.             * 
*                                                                 *
******************************************************************/

CustomTextRenderer::~CustomTextRenderer()
{
    SafeRelease(&pRT_);
	SafeRelease(&defaultForeBrush);
	SafeRelease(&controlForeBrush);
}

void CustomTextRenderer::SetDefalutForeBrush(ID2D1SolidColorBrush* fore)
{
	SafeRelease(&defaultForeBrush);
	this->defaultForeBrush = fore;
	this->defaultForeBrush->AddRef();
}

void CustomTextRenderer::SetControlForeBrush(ID2D1SolidColorBrush* control)
{
	SafeRelease(&controlForeBrush);
	this->controlForeBrush = control;
	this->controlForeBrush->AddRef();
}

void CustomTextRenderer::RemoveControlSymbol(wchar_t c)
{
	showSymbols.erase(c);
}

void CustomTextRenderer::AddControlSymbol(wchar_t c,UINT32 indic)
{
	showSymbols[c] = indic;
}

void CustomTextRenderer::SplitGlyphRun(std::vector<GLYPH_RUN_INFO>& runs,
	const DWRITE_GLYPH_RUN *run,
	UINT16 *indices,
	FLOAT *advances,
	const DWRITE_GLYPH_RUN_DESCRIPTION* glyphRunDescription,
	ID2D1SolidColorBrush* foreBrush)
{
	if(showSymbols.empty())
	{
		GLYPH_RUN_INFO newInfo;
		::memcpy(&(newInfo.run),run,sizeof(DWRITE_GLYPH_RUN));
		newInfo.ForeBrush = foreBrush;
		runs.push_back(newInfo);
		return;
	}

	FLOAT advance;
	UINT16 indic;
	int nonShowCharCount = 0;
	int controlCharCount = 0;
	unsigned int startPos = 0,i;
	for(i = startPos; i < glyphRunDescription->stringLength; i++)
	{
		if(GetAdvancesAndIndices(run,glyphRunDescription->string[i],&indic,&advance) == false)
			controlCharCount++;
		else
			indic = run->glyphIndices[i];

		if(controlCharCount > 0 && run->glyphAdvances[i] == 0)
			advances[i] = advance;
		else
			advances[i] = run->glyphAdvances[i];
		indices[i] = indic;

		if(indic == 0)
			nonShowCharCount++;

		if(controlCharCount > 0)
		{
			GLYPH_RUN_INFO newInfo;
			::memcpy(&(newInfo.run),run,sizeof(DWRITE_GLYPH_RUN));
			int length = i - 1 - startPos + 1;
			if(length > 0)
			{
				newInfo.run.glyphAdvances = &advances[startPos];
				newInfo.run.glyphIndices = &indices[startPos];
				newInfo.run.glyphCount = length - nonShowCharCount;
				newInfo.ForeBrush = foreBrush;
				runs.push_back(newInfo);
			}

			startPos = i;

			newInfo.ForeBrush = this->controlForeBrush;
			newInfo.run.glyphAdvances = &advances[startPos];
			newInfo.run.glyphIndices = &indices[startPos];
			newInfo.run.glyphCount = 1;
			runs.push_back(newInfo);

			startPos++;

			nonShowCharCount = 0;
			controlCharCount = 0;
		}
	}

	if(startPos < i)
	{
		GLYPH_RUN_INFO newInfo;
		::memcpy(&(newInfo.run),run,sizeof(DWRITE_GLYPH_RUN));
		newInfo.run.glyphAdvances = &advances[startPos];
		newInfo.run.glyphIndices = &indices[startPos];
		newInfo.run.glyphCount = i  - 1 - startPos + 1 - nonShowCharCount;
		newInfo.ForeBrush = foreBrush;
		runs.push_back(newInfo);
	}
	return;
}

/******************************************************************
*                                                                 *
*  CustomTextRenderer::DrawGlyphRun                               *
*                                                                 *
*  Gets GlyphRun outlines via IDWriteFontFace::GetGlyphRunOutline *
*  and then draws and fills them using Direct2D path geometries   *
*                                                                 *
******************************************************************/

STDMETHODIMP CustomTextRenderer::DrawGlyphRun(
    __maybenull void* clientDrawingContext,
    FLOAT baselineOriginX,
    FLOAT baselineOriginY,
    DWRITE_MEASURING_MODE measuringMode,
    __in DWRITE_GLYPH_RUN const* glyphRun,
    __in DWRITE_GLYPH_RUN_DESCRIPTION const* glyphRunDescription,
    IUnknown* clientDrawingEffect
    )
{
	ID2D1SolidColorBrush* foreBrush = NULL;
	ID2D1SolidColorBrush* back = NULL;
	ID2D1StrokeStyle* underline = NULL;
	this->GetDrawingEffect(clientDrawingEffect,&foreBrush,&back,&underline);
	if(foreBrush == NULL)
		foreBrush = this->defaultForeBrush;

	//Rg[R[h΂\悤ɃOt
	FLOAT* advances = new FLOAT[glyphRunDescription->stringLength];
	UINT16* glyphIndexs = new UINT16[glyphRunDescription->stringLength];
	std::vector<GLYPH_RUN_INFO> runs;

	SplitGlyphRun(runs,glyphRun,glyphIndexs,advances,glyphRunDescription,foreBrush);

	std::vector<GLYPH_RUN_INFO>::iterator it = runs.begin();
	while(it != runs.end())
	{
		//̈߂
		D2D1_RECT_F bounds;
		GLYPH_RUN_INFO info = *it;
		this->GetBounds(info.run,baselineOriginX,baselineOriginY,info.run.glyphCount,&bounds);

		//wihԂ
		if(back != NULL)
		{
			pRT_->SetAntialiasMode(D2D1_ANTIALIAS_MODE_ALIASED);

			pRT_->FillRectangle(&bounds,back);

			pRT_->SetAntialiasMode(D2D1_ANTIALIAS_MODE_PER_PRIMITIVE);
		}

		pRT_->DrawGlyphRun(D2D1::Point2F(baselineOriginX,baselineOriginY),&(info.run),info.ForeBrush,measuringMode);

		baselineOriginX = bounds.right;
	
		it++;
	}

	delete advances;
	delete glyphIndexs;

	return S_OK;
}

void CustomTextRenderer::GetBounds(DWRITE_GLYPH_RUN myGlyphRun,float baselineOriginX,float baselineOriginY,int glyphCount,D2D1_RECT_F* bounds)
{
	DWRITE_FONT_METRICS fontMetrics;
	myGlyphRun.fontFace->GetMetrics(&fontMetrics);

	float ascentPixel = myGlyphRun.fontEmSize * fontMetrics.ascent / fontMetrics.designUnitsPerEm;
	float dscentPixel = myGlyphRun.fontEmSize * fontMetrics.descent / fontMetrics.designUnitsPerEm;

	float right = baselineOriginX;

	for(int i = 0; i < glyphCount; i++)
    {
		if(myGlyphRun.bidiLevel % 2 == 1)
			right -= myGlyphRun.glyphAdvances[i];
		else
	        right += myGlyphRun.glyphAdvances[i];
    }

	bounds->left = baselineOriginX;
	if(glyphCount > 0)
		bounds->right = right;
	else
		bounds->right = baselineOriginX + myGlyphRun.glyphAdvances[0];
	bounds->top = baselineOriginY - ascentPixel;
	bounds->bottom = baselineOriginY + dscentPixel + 1;
}
/******************************************************************
*                                                                 *
*  CustomTextRenderer::DrawUnderline                              *
*                                                                 *
*  Draws underlines below the text using a Direct2D recatangle    *
*  geometry                                                       *
*                                                                 *
******************************************************************/

STDMETHODIMP CustomTextRenderer::DrawUnderline(
    __maybenull void* clientDrawingContext,
    FLOAT baselineOriginX,
    FLOAT baselineOriginY,
    __in DWRITE_UNDERLINE const* underline,
    IUnknown* clientDrawingEffect
    )
{
	ID2D1SolidColorBrush* pBrush = NULL;
	ID2D1SolidColorBrush* back = NULL;
	ID2D1StrokeStyle* stroke = NULL;
	this->GetDrawingEffect(clientDrawingEffect,&pBrush,&back,&stroke);

	D2D1_POINT_2F from = D2D1::Point2F(baselineOriginX,baselineOriginY);
	D2D1_POINT_2F to = D2D1::Point2F(baselineOriginX + underline->width,baselineOriginY);

	if(pBrush == NULL)
		pBrush = this->defaultForeBrush;

	pRT_->SetAntialiasMode(D2D1_ANTIALIAS_MODE_ALIASED);

	float thickness = underline->thickness;
	if(underline->measuringMode == DWRITE_MEASURING_MODE_GDI_CLASSIC ||
		underline->measuringMode == DWRITE_MEASURING_MODE_GDI_NATURAL ||
		thickness < 1)
		thickness = (float)(int)(thickness + 0.5);

	pRT_->DrawLine(from,to,pBrush,thickness,stroke);

	pRT_->SetAntialiasMode(D2D1_ANTIALIAS_MODE_PER_PRIMITIVE);

	return S_OK;
}

/******************************************************************
*                                                                 *
*  CustomTextRenderer::DrawStrikethrough                          *
*                                                                 *
*  Draws strikethroughs below the text using a Direct2D           *
*  recatangle geometry                                            *
*                                                                 *
******************************************************************/

STDMETHODIMP CustomTextRenderer::DrawStrikethrough(
    __maybenull void* clientDrawingContext,
    FLOAT baselineOriginX,
    FLOAT baselineOriginY,
    __in DWRITE_STRIKETHROUGH const* strikethrough,
    IUnknown* clientDrawingEffect
    )
{
	ID2D1SolidColorBrush* pBrush;
	ID2D1SolidColorBrush* back;
	ID2D1StrokeStyle* stroke = NULL;
	this->GetDrawingEffect(clientDrawingEffect,&pBrush,&back,&stroke);

	D2D1_POINT_2F from = D2D1::Point2F(baselineOriginX,baselineOriginY);
	D2D1_POINT_2F to = D2D1::Point2F(baselineOriginX + strikethrough->width,baselineOriginY);

	if(pBrush == NULL)
		pBrush = this->defaultForeBrush;

	pRT_->SetAntialiasMode(D2D1_ANTIALIAS_MODE_ALIASED);

	float thickness = strikethrough->thickness;
	if(strikethrough->measuringMode == DWRITE_MEASURING_MODE_GDI_CLASSIC ||
		strikethrough->measuringMode == DWRITE_MEASURING_MODE_GDI_NATURAL ||
		thickness < 1)
		thickness = (float)(int)(thickness + 0.5);

	pRT_->DrawLine(from,to,pBrush,thickness);

	pRT_->SetAntialiasMode(D2D1_ANTIALIAS_MODE_PER_PRIMITIVE);

	return S_OK;
}

/******************************************************************
*                                                                 *
*  CustomTextRenderer::DrawInlineObject                           *
*                                                                 *
*  This function is not implemented for the purposes of this      *
*  sample.                                                        *
*                                                                 *
******************************************************************/

STDMETHODIMP CustomTextRenderer::DrawInlineObject(
    __maybenull void* clientDrawingContext,
    FLOAT originX,
    FLOAT originY,
    IDWriteInlineObject* inlineObject,
    BOOL isSideways,
    BOOL isRightToLeft,
    IUnknown* clientDrawingEffect
    )
{
	inlineObject->Draw(clientDrawingContext,this,originX,originY,isSideways,isRightToLeft,clientDrawingEffect);
    return S_OK;
}

bool CustomTextRenderer::GetAdvancesAndIndices(const DWRITE_GLYPH_RUN *myGlyphRun,wchar_t c,UINT16 *indices,FLOAT *advance)
{
	UINT32 code = 0;

	std::hash_map<wchar_t,UINT32,my_wchar_hasher>::iterator it = showSymbols.find(c);
	if(it == showSymbols.end())
	{
		*advance = 0;
		*indices = 0;
		return true;
	}

	code = showSymbols[c];

	UINT16 indic;
	myGlyphRun->fontFace->GetGlyphIndicesW(&code,1,&indic);

	DWRITE_GLYPH_METRICS metrics;
	myGlyphRun->fontFace->GetDesignGlyphMetrics(&indic,1,&metrics);

	DWRITE_FONT_METRICS fontMetrics;
	myGlyphRun->fontFace->GetMetrics(&fontMetrics);

	*advance = myGlyphRun->fontEmSize * metrics.advanceWidth / fontMetrics.designUnitsPerEm;
	*indices = indic;
	return false;
}

void CustomTextRenderer::GetDrawingEffect(IUnknown* clientDrawingEffect,ID2D1SolidColorBrush** fore,ID2D1SolidColorBrush** back,ID2D1StrokeStyle** underline)
{
    if (clientDrawingEffect == NULL)
    {
		*fore = NULL;
		*back = NULL;
		return;
    }

	DrawingEffect* DrawingEffect;
	HRESULT hr;
    hr = clientDrawingEffect->QueryInterface(__uuidof(DrawingEffect), reinterpret_cast<void**>(&DrawingEffect));
	if(hr == S_OK)
	{
	    DrawingEffect->GetForeColor(fore);
		DrawingEffect->GetBackColor(back);
		DrawingEffect->GetUnderLine(underline);
		SafeRelease(&DrawingEffect);    
		return;
	}
	*fore = reinterpret_cast<ID2D1SolidColorBrush*>(clientDrawingEffect);
}

/******************************************************************
*                                                                 *
*  CustomTextRenderer::AddRef                                     *
*                                                                 *
*  Increments the ref count                                       *
*                                                                 *
******************************************************************/

STDMETHODIMP_(unsigned long) CustomTextRenderer::AddRef()
{
    return InterlockedIncrement(&cRefCount_);
}

/******************************************************************
*                                                                 *
*  CustomTextRenderer::Release                                    *
*                                                                 *
*  Decrements the ref count and deletes the instance if the ref   *
*  count becomes 0                                                *
*                                                                 *
******************************************************************/

STDMETHODIMP_(unsigned long) CustomTextRenderer::Release()
{
    unsigned long newCount = InterlockedDecrement(&cRefCount_);

    if (newCount == 0)
    {
        delete this;
        return 0;
    }

    return newCount;
}

/******************************************************************
*                                                                 *
*  CustomTextRenderer::IsPixelSnappingDisabled                    *
*                                                                 *
*  Determines whether pixel snapping is disabled. The recommended *
*  default is FALSE, unless doing animation that requires         *
*  subpixel vertical placement.                                   *
*                                                                 *
******************************************************************/

STDMETHODIMP CustomTextRenderer::IsPixelSnappingDisabled(
    __maybenull void* clientDrawingContext,
    __out BOOL* isDisabled
    )
{
    *isDisabled = FALSE;
    return S_OK;
}

/******************************************************************
*                                                                 *
*  CustomTextRenderer::GetCurrentTransform                        *
*                                                                 *
*  Returns the current transform applied to the render target..   *
*                                                                 *
******************************************************************/

STDMETHODIMP CustomTextRenderer::GetCurrentTransform(
    __maybenull void* clientDrawingContext,
    __out DWRITE_MATRIX* transform
    )
{
    //forward the render target's transform
    pRT_->GetTransform(reinterpret_cast<D2D1_MATRIX_3X2_F*>(transform));
    return S_OK;
}

/******************************************************************
*                                                                 *
*  CustomTextRenderer::GetPixelsPerDip                            *
*                                                                 *
*  This returns the number of pixels per DIP.                     *
*                                                                 *
******************************************************************/

STDMETHODIMP CustomTextRenderer::GetPixelsPerDip(
    __maybenull void* clientDrawingContext,
    __out FLOAT* pixelsPerDip
    )
{
    float x, yUnused;

    pRT_->GetDpi(&x, &yUnused);
    *pixelsPerDip = x / 96;

    return S_OK;
}

/******************************************************************
*                                                                 *
*  CustomTextRenderer::QueryInterface                             *
*                                                                 *
*  Query interface implementation                                 *
*                                                                 *
******************************************************************/

STDMETHODIMP CustomTextRenderer::QueryInterface(
    IID const& riid,
    void** ppvObject
    )
{
    if (__uuidof(IDWriteTextRenderer) == riid)
    {
        *ppvObject = this;
    }
    else if (__uuidof(IDWritePixelSnapping) == riid)
    {
        *ppvObject = this;
    }
    else if (__uuidof(IUnknown) == riid)
    {
        *ppvObject = this;
    }
    else
    {
        *ppvObject = NULL;
        return E_FAIL;
    }

    AddRef();

    return S_OK;
}
