﻿/*
 * Copyright (C) 2013 FooProject
 * * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
using System;
using Windows.Graphics.Printing;
using System.Runtime.InteropServices;
using Windows.UI;
using FooEditEngine.Metro;

namespace FooEditEngine
{
    /// <summary>
    /// イベントデータ
    /// </summary>
    public sealed class ParseCommandEventArgs
    {
        /// <summary>
        /// 印刷中のページ番号
        /// </summary>
        public int PageNumber;
        /// <summary>
        /// ページ範囲内で許容されている最大の番号
        /// </summary>
        public int MaxPageNumber;
        /// <summary>
        /// 処理前の文字列
        /// </summary>
        public string Original;
        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="nowPage">印刷中のページ番号</param>
        /// <param name="maxPage">印刷すべき最大のページ番号</param>
        /// <param name="org">処理前の文字列</param>
        public ParseCommandEventArgs(int nowPage, int maxPage, string org)
        {
            this.PageNumber = nowPage;
            this.MaxPageNumber = maxPage;
            this.Original = org;
        }
    }

    /// <summary>
    /// コマンド処理用デリゲート
    /// </summary>
    /// <param name="sender">送信元のクラス</param>
    /// <param name="e">イベントデータ</param>
    /// <returns>処理後の文字列</returns>
    public delegate string ParseCommandHandler(object sender, ParseCommandEventArgs e);

    sealed class PrintableViewFactory
    {
        Color foreground, comment, keyword1, keyword2, literal, url;
        string fontName;
        double fontSize;
        Document document;
        LineBreakMethod lineBreakMethod;
        int lineBreakCharCount, tabCount;
        bool drawLineNumber, urlMark;
        IHilighter hilighter;
        public PrintableViewFactory(FooTextBox textbox)
        {
            this.foreground = textbox.Foreground;
            this.comment = textbox.Comment;
            this.keyword1 = textbox.Keyword1;
            this.keyword2 = textbox.Keyword2;
            this.literal = textbox.Literal;
            this.url = textbox.URL;
            this.fontName = textbox.FontFamily.Source;
            this.fontSize = textbox.FontSize;
            this.tabCount = textbox.TabChars;
            this.document = new Document(textbox.Document);
            this.lineBreakMethod = textbox.LineBreakMethod;
            this.lineBreakCharCount = textbox.LineBreakCharCount;
            this.drawLineNumber = textbox.DrawLineNumber;
            this.urlMark = textbox.MarkURL;
            this.hilighter = textbox.Hilighter;
        }
        public D2DRenderBase CreateRender(PrintPageDescription pagedesc, IPrintDocumentPackageTarget docPackageTarget)
        {
            D2DRenderBase render;
            Size size = new Size(pagedesc.ImageableRect.Width, pagedesc.ImageableRect.Height);
            if (docPackageTarget == null)
                render = new D2DPrintPreviewRender(this.fontName, this.fontSize, size, Math.Min(pagedesc.DpiX, pagedesc.DpiY));
            else
                render = new D2DPrintRender(this.fontName, this.fontSize, size, Math.Min(pagedesc.DpiX, pagedesc.DpiY), docPackageTarget);
            render.Foreground = this.foreground;
            render.Comment = this.comment;
            render.Keyword1 = this.keyword1;
            render.Keyword2 = this.keyword2;
            render.Literal = this.literal;
            render.Url = this.url;
            return render;
        }
        public PrintableView CreateView(PrintPageDescription pagedesc, D2DRenderBase render, string header, string footer)
        {

            PrintableView view = new PrintableView(this.document, (IPrintableTextRender)render);
            view.TabStops = this.tabCount;
            view.LineBreak = this.lineBreakMethod == LineBreakMethod.None ? LineBreakMethod.PageBound : this.lineBreakMethod;
            view.LineBreakCharCount = this.lineBreakCharCount;
            view.DrawLineNumber = this.drawLineNumber;
            view.Header = header;
            view.Footer = footer;
            view.UrlMark = this.urlMark;
            view.PageBound = new Rectangle(pagedesc.ImageableRect.X, pagedesc.ImageableRect.Y, pagedesc.ImageableRect.Width, pagedesc.ImageableRect.Height);
            view.Hilighter = this.hilighter;
            view.PerfomLayouts();

            return view;
        }
    }

    sealed class DocumentSource : IPrintDocumentPageSource, IPrintPreviewPageCollection, IPrintDocumentSource
    {
        IPrintPreviewDxgiPackageTarget dxgiPreviewTarget;
        bool paginateCalled = false;
        Size imageRect;
        PrintableViewFactory factory;
        D2DPrintPreviewRender previewRender;
        PrintableView previewView;

        public ParseCommandHandler ParseHF;
        public string Header = string.Empty;
        public string Fotter = string.Empty;

        public DocumentSource(FooTextBox textbox)
        {
            this.factory = new PrintableViewFactory(textbox);
            this.ParseHF = (s, e) => { return e.Original; };
        }

        public void GetPreviewPageCollection(IPrintDocumentPackageTarget docPackageTarget,out IPrintPreviewPageCollection docPageCollection)
        {
            Guid guid = new Guid(PreviewPackageIds.IID_PREVIEWPACKAGETARGET_DXGI);
            IntPtr target;
            docPackageTarget.GetPackageTarget(guid, guid, out target);
            this.dxgiPreviewTarget = (IPrintPreviewDxgiPackageTarget)Marshal.GetObjectForIUnknown(target);
            docPageCollection = (IPrintPreviewPageCollection)this;
        }

        public void MakeDocument(object printTaskOptions, IPrintDocumentPackageTarget docPackageTarget)
        {
            PrintTaskOptions options = (PrintTaskOptions)printTaskOptions;
            PrintPageDescription pagedesc = options.GetPageDescription(1);

            D2DRenderBase render = this.factory.CreateRender(pagedesc, docPackageTarget);
            PrintableView view = this.factory.CreateView(pagedesc, render, this.Header, this.Fotter);

            bool result = false;
            int currentPage = 0;

            while (!result)
            {
                if(!string.IsNullOrEmpty(this.Header))
                    view.Header = this.ParseHF(this, new ParseCommandEventArgs(currentPage, -1, this.Header));
                if (!string.IsNullOrEmpty(this.Fotter))
                    view.Footer = this.ParseHF(this, new ParseCommandEventArgs(currentPage, -1, this.Fotter));

                render.BeginDraw();
                view.Draw(view.PageBound);
                render.EndDraw();
                result = view.TryPageDown();
                currentPage++;
            }

            render.Dispose();
            view.Dispose();
        }

        public void Paginate(uint currentJobPage, object printTaskOptions)
        {
            PrintTaskOptions options = (PrintTaskOptions)printTaskOptions;
            PrintPageDescription pagedesc = options.GetPageDescription(currentJobPage);

            this.imageRect = new Size(pagedesc.ImageableRect.Width, pagedesc.ImageableRect.Height);

            this.previewRender = (D2DPrintPreviewRender)this.factory.CreateRender(pagedesc, null);
            this.previewView = this.factory.CreateView(pagedesc, this.previewRender, this.Header, this.Fotter);

            int maxPage = 1;
            while (!this.previewView.TryPageDown())
                maxPage++;

            this.dxgiPreviewTarget.SetJobPageCount(PageCountType.FinalPageCount, (uint)maxPage);

            this.dxgiPreviewTarget.InvalidatePreview();

            this.paginateCalled = true;
        }

        public void MakePage(uint desiredJobPage, float width, float height)
        {
            if (width <= 0 || height <= 0)
                throw new COMException("", 0x70057/*E_INVALIDARG*/);
            if (!this.paginateCalled)
                return;
            if (desiredJobPage == 0xFFFFFFFF)
                desiredJobPage = 1;

            this.previewView.TryScroll(0, 0);   //元に戻さないとページ番号が変わった時に正しく動作しない

            for (int i = 1; i < desiredJobPage; i++)
                this.previewView.TryPageDown();

            this.previewRender.BeginDraw();
            this.previewView.Draw(this.previewView.PageBound);
            this.previewRender.EndDraw(this.dxgiPreviewTarget,desiredJobPage);
        }

        public void Dispose()
        {
            if (this.previewView != null)
                this.previewView.Dispose();
            if (this.previewRender != null)
                this.previewRender.Dispose();
        }
    }
}
