/****************************************************************************
**
** Copyright (C) 2012 Takumi Asaki
** All rights reserved.
** Contact: Takumi Asaki (takumi.asaki@gmail.com)
**
** This file is part of the fontmanager application.
**
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
****************************************************************************/

import QtQuick 1.1
import com.nokia.meego 1.0
import com.nokia.extras 1.0
import 'UIConstants.js' as UI

import "HelpDialog.js" as HelpHelper

Page {
    id: mainPage
    tools: mainTools

    property bool disableTools: false

    property bool showHelp: false
    property string helpMessage: qsTr(
                                     "<h1>How to use</h1>" +
                                     "<hr/>" +
                                     "<h2>Toolbar</h2>" +
                                     "<p><img src=\"image://theme/icon-m-toolbar-add\">: Install font.  Please copy font file(s) to this device in advance.  Currently *.ttf and *.ttc are supported.</p>" +
                                     "<p><img src=\"image://theme/icon-m-toolbar-settings\">: Configure fonts(expert).  You can assign/unassign any fonts to Serif/Sans Serif/Monospace/System.</p>" +
                                     "<hr/>" +
                                     "<h2>Menu(<img src=\"image://theme/icon-m-toolbar-view-menu\">)</h2>" +
                                     "<p><b>Create Recommended Settings</b>: Create recommended settings.  <i>Normal</i> priority is given to user installed font(s).  System fonts have <i>Higher</i> priority.  System CJK fonts conflicts your installed fonts have <i>Lower</i> priority.</p>" +
                                     "<p><b>Backup Config</b>: Save current fonts config into ~/MyDocs/Documents</p>" +
                                     "<p><b>Restore Config</b>: Restore fonts config from backuped file</p>"
                                     )

    Connections {
        target: controller
        onClearInstalledFontList: {
            installedFontList.clear()
        }
        onAppendInstalledFont: {
            installedFontList.append({ "title": family, "subtitle": fullname })
        }
    }

    ListModel {
        id: installedFontList
    }

    PageHeader {
        id: pageHeader
        anchors.top: parent.top
        anchors.left: parent.left
        anchors.right: parent.right
        enabled: !disableTools
        text: qsTr("Installed Fonts")
    }

    Item {
        id: contents
        anchors.top: pageHeader.bottom
        anchors.bottom: parent.bottom
        anchors.left: parent.left
        anchors.right: parent.right
        anchors.margins: UI.DEFAULT_MARGIN / 2
        clip: true

        ListView {
            id: installedFontListView
            anchors.fill: parent
            enabled: !disableTools
            opacity: enabled ? 1.0 : 0.5

            model: installedFontList

            delegate: ListDelegate {
                onClicked: {
                    if (mainMenu.status !== DialogStatus.Closed)
                        mainMenu.close()
                    var component = Qt.createComponent("InstalledFontInfoPage.qml")
                    if (component.status === Component.Error)
                        console.log("Error: " + component.errorString())
                    pageStack.push(component, { "fontInfo": controller.fontInfo(title, subtitle) } );
                }
            }

        }

        Flickable {
            id: helpView
            visible: false
            anchors.fill: parent
            contentHeight: helpText.paintedHeight
            Text {
                id: helpText
                wrapMode: Text.WrapAtWordBoundaryOrAnywhere
                width: parent.width
                text: helpMessage
                font.pixelSize: UI.FONT_DEFAULT
                color: "black"
                horizontalAlignment: Text.AlignLeft
            }
        }

        ScrollDecorator {
            id: scrollDecorator
            flickableItem: installedFontListView
        }


    }

    BusyIndicator {
        id: busyIndicator
        anchors.centerIn: parent
        platformStyle: BusyIndicatorStyle { size: "large" }
        running: mainPage.state == "busy"
        visible: running
    }

    ToolBarLayout {
        id: mainTools
        visible: true
        enabled: !disableTools
        opacity: enabled ? 1.0 : 0.5
        ToolIcon {
            platformIconId: "toolbar-add"
            onClicked: {
                if (mainMenu.status !== DialogStatus.Closed)
                    mainMenu.close()
                pageStack.push(fontSelectionPage)
            }
        }
        ToolIcon {
            platformIconId: "toolbar-settings"
            onClicked: {
                if (mainMenu.status !== DialogStatus.Closed)
                    mainMenu.close()
                var component = Qt.createComponent("EditFontsConfPage.qml")
                if (component.status === Component.Error)
                    console.log("Error: " + component.errorString())
                pageStack.push(component)
            }
        }
        ToolIcon {
            platformIconId: "toolbar-view-menu"
            anchors.right: (parent === undefined) ? undefined : parent.right
            onClicked: {
                if (mainMenu.status === DialogStatus.Closed)
                    mainMenu.open()
                else
                    mainMenu.close()
            }
        }
    }

    Menu {
        id: mainMenu
        visualParent: pageStack
        MenuLayout {
            MenuItem {
                id: createRecommendedSettingsItem
                text: qsTr("Create Recommended Settings");
                onClicked: {
                    if (mainMenu.status !== DialogStatus.Closed)
                        mainMenu.close()
//                    if (controller.installedFonts.length > 0) {
//                        console.log("Installed Fonts:" + controller.installedFonts)
//                    }
                    if (controller.localFontsConfExists)
                        openCreateConfirmDialog()
                    else
                        createRecommendedSettings()
                }
            }
            MenuItem {
                id: backupConfigItem
                text: qsTr("Backup Config")
                onClicked: backupConfig()
                enabled: !controller.isEmptyFontsConf
            }
            MenuItem {
                id: restoreConfigItem
                text: qsTr("Restore Config")
                onClicked: {
                    var component = Qt.createComponent("RestoreFontsConfPage.qml")
                    if (component.status === Component.Error)
                        console.log("Error: " + component.errorString())
                    pageStack.push(component)
                }
            }
            MenuItem {
                text: qsTr("About")
                onClicked: openAboutDialog()
            }
            MenuItem {
                text: qsTr("Help")
                onClicked: {
                    showHelp = true
                    HelpHelper.openHelpDialog(mainPage, contents, helpMessage, closeHelp)
                }
            }
        }
    }

    function openCreateConfirmDialog()
    {
        queryDialog.titleText = qsTr("Replace it?")
        queryDialog.message = qsTr("Existing %1 will be replaced by recommended settings.  Are you sure?").arg(controller.localFontsConfPath)
        queryDialog.acceptButtonText = qsTr("OK")
        queryDialog.rejectButtonText = qsTr("Cancel")
        queryDialog.accepted.connect(createRecommendedSettings)
        queryDialog.open()
    }

    function createRecommendedSettings()
    {
        queryDialog.accepted.disconnect(createRecommendedSettings)
        controller.createRecommendedSettings()
    }

    function openAboutDialog()
    {
        queryDialog.titleText = qsTr("Font Manager for N9")
        queryDialog.message = qsTr("Version: %1\nAuthor: Takumi Asaki <takumi.asaki@gmail.com>\n\nApplication Icon is created by hirao").arg(controller.version)
        queryDialog.acceptButtonText = qsTr("Close")
        queryDialog.rejectButtonText = ""
        queryDialog.open()
    }

    function closeHelp()
    {
        showHelp = false
    }

    function backupConfig()
    {
        var filepath = controller.defaultBackupFilename()
        controller.backupConfig(filepath)
    }

    states: [
        State {
            name: "nofonts"
            when: (installedFontListView.count == 0) && !controller.working && !showHelp
            PropertyChanges {
                target: createRecommendedSettingsItem
                enabled: false
            }
//            PropertyChanges {
//                target: installedFontListView
//                visible: false
//            }
            PropertyChanges {
                target: helpView
                visible: true
            }
            PropertyChanges {
                target: scrollDecorator
                flickableItem: helpView
            }
            PropertyChanges {
                target: pageHeader
                text: qsTr("No Fonts Installed")
            }
        },
        State {
            name: "busy"
            when: controller.working && !showHelp
            PropertyChanges {
                target: mainPage
                disableTools: true
            }
        },
        State {
            name: "showHelp"
            when: showHelp
            PropertyChanges {
                target: mainPage
                disableTools: true
            }
        }

    ]
}
