/****************************************************************************
**
** Copyright (C) 2012 Takumi Asaki
** All rights reserved.
** Contact: Takumi Asaki (takumi.asaki@gmail.com)
**
** This file is part of the fontmanager application.
**
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
****************************************************************************/

// import QtQuick 1.0 // to target S60 5th Edition or Maemo 5
import QtQuick 1.1
import com.nokia.meego 1.0
import com.nokia.extras 1.0
import 'UIConstants.js' as UI
import 'EditorHelper.js' as Helper

Item {
    id: editorPage

    property string family
    property string familyName
    property string curSection
    property string curSectionName

    property variant editorController: null

    property alias listView: preferFamilyListView

    onEditorControllerChanged: {
        if (editorController) {
            Helper.setConnection(editorController)
        }
    }

    function addHeader(section, sectionName)
    {
        Helper.setIndex(curSection, preferFamilyList.count)
        curSection = section
        curSectionName = sectionName
        preferFamilyList.append({
                                    "title": qsTr("Priority: %1").arg(sectionName),
                                    "subtitle": "",
                                    "systemFont": false,
                                    "header": true,
                                    "priority": section
                                })
        Helper.setIndex(curSection, preferFamilyList.count)
    }

    function insertFamily(family, priority, systemFont)
    {
        var index = Helper.index(priority)
        preferFamilyList.insert(index, {
                                    "title": controller.localeFamily(family),
                                    "subtitle": (systemFont ? qsTr("System Font") : qsTr("User Font")),
                                    "systemFont": systemFont,
                                    "header": false,
                                    "priority": priority,
                                    "family": family,
                                })
        Helper.updateInsertIndexes(index)
    }

    function addFamily(section, family, priority, systemFont)
    {
        preferFamilyList.append({
                                    "title": controller.localeFamily(family),
                                    "subtitle": (systemFont ? qsTr("System Font") : qsTr("User Font")),
                                    "systemFont": systemFont,
                                    "header": false,
                                    "priority": curSection,
                                    "family": family,
                                })
        Helper.setIndex(curSection, preferFamilyList.count)
    }

    function moveUpFamily()
    {
        var index = preferFamilyListView.currentIndex
        if (index < 1)
            return;
        var prevItem = preferFamilyList.get(index - 1)
        if (!prevItem.header) {
            var item = preferFamilyList.get(index)
            editorController.moveFamilyUp(item.family, item.priority)
            preferFamilyList.move(index, index - 1, 1)
        }
    }

    function moveDownFamily()
    {
        var index = preferFamilyListView.currentIndex
        if (index >= preferFamilyList.count - 1)
            return;
        var nextItem = preferFamilyList.get(index + 1)
        if (!nextItem.header) {
            var item = preferFamilyList.get(index)
            editorController.moveFamilyDown(item.family, item.priority)
            preferFamilyList.move(index, index + 1, 1)
        }
    }

    function removeFamily()
    {
        var index = preferFamilyListView.currentIndex
        var item = preferFamilyList.get(index)
        editorController.removeFamily(item.family, item.priority)
        preferFamilyList.remove(index)
        Helper.updateRemoveIndexes(index)
        preferFamilyListView.currentIndex = -1
    }

    Component {
        id: modelSyncConnections
        Connections {
            onClearFamilyList: {
                preferFamilyList.clear()
                Helper.init()
            }
            onStartSection: addHeader(section, sectionName)
            onAppendFamilyToList: addFamily(section, family, priority, systemFont)
        }
    }

    ListModel {
        id: preferFamilyList
    }

    Item {
//        anchors.top: pageHeader.bottom
        anchors.fill: parent
        anchors.topMargin: UI.DEFAULT_MARGIN / 2
//        anchors.bottomMargin: UI.DEFAULT_MARGIN / 2
        clip: true

        ListView {
            id: preferFamilyListView
            anchors.fill: parent
            anchors.leftMargin: UI.DEFAULT_MARGIN
            anchors.rightMargin: UI.DEFAULT_MARGIN

            cacheBuffer: 100

            currentIndex: -1

            model: preferFamilyList
            delegate: EditorListDelegate { }

            highlight: Rectangle {
                x: - UI.MARGIN_XLARGE / 2
                width: preferFamilyListView.width + UI.MARGIN_XLARGE
                height: UI.LIST_ITEM_HEIGHT_SMALL
                color: "transparent"
                border.width: 5
                border.color: "#bd543d"
                radius: 5
//                z: 1
            }
            highlightFollowsCurrentItem: true
            highlightMoveDuration: 100
        }

        ScrollDecorator {
            flickableItem: preferFamilyListView
        }

    }

}

